# Development Guidelines

## Code Standards

### PHP

**File Headers**
```php
<?php
/**
 * Feature Name
 * Brief description of what this file does
 */

require_once '../includes/config.php';
requireLogin(); // If authentication required
```

**Naming Conventions**
- Variables: `$snake_case`
- Functions: `camelCase()`
- Classes: `PascalCase`
- Constants: `UPPER_CASE`
- Files: `kebab-case.php`

**Best Practices**
```php
// Good
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);

// Bad
$userID = $_SESSION['user_id'];
$result = mysql_query("SELECT * FROM users WHERE id = $userID");
```

### JavaScript

**Naming Conventions**
- Variables: `camelCase`
- Constants: `UPPER_CASE`
- Functions: `camelCase()`
- Classes: `PascalCase`

**Best Practices**
```javascript
// Good
const userId = getUserId();
async function fetchAssessments() {
    const response = await fetch('/api/assessments.php');
    return await response.json();
}

// Bad
var user_id = getUserId();
function fetch_assessments() {
    $.ajax({ url: '/api/assessments.php' });
}
```

### CSS

**Naming Conventions**
- Classes: `kebab-case`
- IDs: `camelCase` (use sparingly)
- Variables: `--kebab-case`

**Best Practices**
```css
/* Good */
.card-header {
    background: var(--primary);
}

/* Bad */
#CardHeader {
    background: #667eea;
}
```

### SQL

**Naming Conventions**
- Tables: `snake_case`
- Columns: `snake_case`
- Indexes: `idx_table_column`
- Foreign Keys: `fk_table_reference`

**Best Practices**
```sql
-- Good
CREATE TABLE user_assessments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    assessment_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id),
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Bad
CREATE TABLE UserAssessments (
    ID int,
    UserId int,
    Date timestamp
);
```

## Security Guidelines

### Input Validation
```php
// Always validate and sanitize
$user_input = filter_input(INPUT_POST, 'field', FILTER_SANITIZE_STRING);

// Use prepared statements
$stmt = $pdo->prepare("INSERT INTO table (column) VALUES (?)");
$stmt->execute([$user_input]);

// Never trust user input
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $error = "Invalid email";
}
```

### Output Escaping
```php
// Always escape output
echo sanitizeOutput($user_data);
echo htmlspecialchars($user_data, ENT_QUOTES, 'UTF-8');

// In HTML attributes
<input value="<?php echo htmlspecialchars($value); ?>">
```

### Authentication
```php
// Always check authentication
requireLogin();

// Always check permissions
requirePermission('admin_access');

// Never trust client-side validation
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}
```

## Error Handling

### PHP
```php
// Use try-catch for risky operations
try {
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
} catch (PDOException $e) {
    logDatabaseError($query, $e->getMessage(), $params);
    $error = "An error occurred. Please try again.";
}

// Log errors, don't expose details
error_log("Error in file.php: " . $e->getMessage());

// Show user-friendly messages
$error = "We couldn't process your request. Please try again.";
```

### JavaScript
```javascript
// Handle fetch errors
try {
    const response = await fetch(url);
    if (!response.ok) throw new Error('Network error');
    const data = await response.json();
} catch (error) {
    console.error('Error:', error);
    showError('Failed to load data');
}
```

## Database Guidelines

### Migrations
```sql
-- Always check before adding
ALTER TABLE locations 
ADD COLUMN IF NOT EXISTS new_field VARCHAR(100);

-- Always comment
-- Migration: Add email fields to locations
-- Date: 2026-01-08
ALTER TABLE locations ADD COLUMN report_emails TEXT 
COMMENT 'Comma-separated email addresses';
```

### Queries
```php
// Use indexes
CREATE INDEX idx_user_id ON assessments(user_id);

// Use JOINs efficiently
SELECT a.*, u.full_name 
FROM assessments a
LEFT JOIN users u ON a.user_id = u.id
WHERE a.status = 'active';

// Avoid N+1 queries
// Good: One query with JOIN
// Bad: Loop with query per iteration
```

## Testing Guidelines

### Before Commit
1. Run syntax check: `php -l file.php`
2. Test functionality manually
3. Check error logs
4. Verify database queries
5. Test permissions

### Before Deploy
1. Run `./check-system.sh`
2. Test on staging environment
3. Review database migrations
4. Check file permissions
5. Test backup/restore

## Git Workflow

### Commit Messages
```
Good:
- "Add email fields to locations table"
- "Fix login error handling"
- "Update action items report with filters"

Bad:
- "changes"
- "fix bug"
- "update"
```

### Branch Names
```
feature/email-automation
bugfix/login-error
hotfix/security-patch
```

## Documentation

### Code Comments
```php
/**
 * Calculate risk score based on likelihood and severity
 * 
 * @param int $likelihood Value from 1-5
 * @param int $severity Value from 1-5
 * @return int Risk score (1-25)
 */
function calculateRiskScore($likelihood, $severity) {
    return $likelihood * $severity;
}
```

### File Headers
```php
<?php
/**
 * Action Items Report
 * 
 * Displays pending action items from assessments with filtering
 * by location, status, and priority. Supports export to Excel.
 * 
 * @requires permissions.php (view_reports permission)
 * @uses action_items_report view
 */
```

## Performance

### Database
- Use indexes on frequently queried columns
- Limit result sets with LIMIT
- Use prepared statements (cached)
- Avoid SELECT * (specify columns)

### PHP
- Use output buffering for large pages
- Cache expensive calculations
- Minimize database queries
- Use isset() before accessing arrays

### Frontend
- Minimize HTTP requests
- Compress CSS/JS
- Use CDN for libraries
- Lazy load images

## Maintenance

### Regular Tasks
- Review error logs weekly
- Update dependencies monthly
- Backup database daily
- Test backups monthly
- Security audit quarterly

### Code Reviews
- Check for security issues
- Verify error handling
- Confirm proper validation
- Review SQL queries
- Test edge cases
