# DATABASE SCHEMA DOCUMENTATION

## 📊 Complete Database Schema

This is the **ONE AND ONLY** schema file you need for the Risk Assessment System.

**File:** `schema.sql` (also in `includes/schema.sql`)

---

## 🚀 Quick Installation

### **Method 1: Command Line (Recommended)**

```bash
# Create database and import
mysql -u root -p < schema.sql
```

### **Method 2: MySQL Client**

```bash
mysql -u root -p
CREATE DATABASE scubatricky_risk;
USE scubatricky_risk;
SOURCE schema.sql;
```

### **Method 3: phpMyAdmin**

1. Open phpMyAdmin
2. Create new database: `scubatricky_risk`
3. Select the database
4. Go to "Import" tab
5. Choose file: `schema.sql`
6. Click "Go"

---

## 📋 What's Included

### **Complete Schema Contains:**

✅ **16 Tables** - All system tables
✅ **2 Views** - Useful data views
✅ **2 Triggers** - Automatic updates
✅ **Default Data** - Ready to use
✅ **Indexes** - Performance optimized
✅ **Foreign Keys** - Data integrity

---

## 🗃️ Database Tables

### **Core Tables (6)**

| # | Table | Purpose | Key Features |
|---|-------|---------|--------------|
| 1 | `users` | User accounts | Roles: admin, assessor, viewer |
| 2 | `locations` | Store/site locations | Email settings, contact info |
| 3 | `categories` | Risk categories | 10 default categories |
| 4 | `risk_items` | Risk templates | 30+ default risks |
| 5 | `assessments` | Assessment records | Auto-numbered, status tracking |
| 6 | `assessment_details` | Individual risk assessments | Calculated risk scores |

### **Action & Workflow Tables (1)**

| # | Table | Purpose | Key Features |
|---|-------|---------|--------------|
| 7 | `action_items` | Required actions | Due dates, status, auto-overdue |

### **Administration Tables (4)**

| # | Table | Purpose | Key Features |
|---|-------|---------|--------------|
| 8 | `permissions` | Role-based access | CRUD permissions per role |
| 9 | `system_settings` | Configuration | Colors, logo, system settings |
| 10 | `custom_fields` | Dynamic fields | Extend any entity |
| 11 | `custom_field_values` | Field data | Stores custom field values |

### **Version Control & Audit Tables (5)**

| # | Table | Purpose | Key Features |
|---|-------|---------|--------------|
| 12 | `system_files` | File tracking | SHA-256 hashing, versioning |
| 13 | `file_versions` | Version history | Full backup of changes |
| 14 | `audit_log` | Activity tracking | All user actions logged |
| 15 | `email_queue` | Email scheduling | Queue with retry logic |
| 16 | `sessions` | Session storage | Database sessions (optional) |

---

## 🎯 Default Data Included

### **1. Default Admin User**

```
Username: admin
Password: Admin123!
Role: admin
```

⚠️ **CHANGE THIS PASSWORD IMMEDIATELY AFTER LOGIN**

### **2. Default Categories (10)**

1. Slips, Trips & Falls
2. Manual Handling
3. Fire Safety
4. Electrical Safety
5. Chemical Safety
6. Work Equipment
7. Display Screen Equipment
8. Workplace Environment
9. Violence & Aggression
10. Lone Working

### **3. Default Risk Items (30+)**

Pre-populated with common workplace risks:
- Wet floors and spillages
- Heavy lifting
- Fire hazards
- Electrical equipment
- Chemicals
- Work tools
- Computer workstations
- Lighting and temperature
- Customer aggression
- Lone working risks

### **4. Default Permissions**

| Role | Create | Read | Update | Delete | Export |
|------|--------|------|--------|--------|--------|
| **Admin** | ✅ | ✅ | ✅ | ✅ | ✅ |
| **Assessor** | ✅ | ✅ | ✅ | ❌ | ✅ |
| **Viewer** | ❌ | ✅ | ❌ | ❌ | ✅ |

### **5. System Settings**

- System name: "Risk Assessment System"
- Primary color: #667eea (purple-blue)
- Date format, timezone, session timeout
- Feature flags (email, audit log)

---

## 🔍 Database Views

### **View 1: v_assessment_summary**

Quick overview of assessments with statistics:
```sql
SELECT * FROM v_assessment_summary;
```

Shows:
- Assessment details
- Risk counts (high/medium/low)
- Action item count
- Location info

### **View 2: v_overdue_actions**

Lists all overdue action items:
```sql
SELECT * FROM v_overdue_actions;
```

Shows:
- Overdue actions
- Days overdue
- Location and assessment
- Responsible person

---

## ⚡ Automatic Features (Triggers)

### **Trigger 1: update_action_item_status**

Automatically sets action items to "overdue" when due date passes.

### **Trigger 2: generate_assessment_number**

Auto-generates assessment numbers: `RA-2026-000001`

---

## 📈 Performance Optimizations

### **Indexes Created**

- Primary keys on all tables
- Foreign key indexes
- Composite indexes for common queries
- Text search indexes

### **Query Optimization**

Common queries are optimized with indexes:
- Assessment lookup by location
- Action items by due date
- Risk items by category
- Audit log by user and date

---

## 🔐 Data Integrity

### **Foreign Keys**

All relationships enforced:
- `assessments` → `locations` (CASCADE)
- `assessments` → `users` (CASCADE)
- `assessment_details` → `assessments` (CASCADE)
- `action_items` → `assessments` (CASCADE)
- `file_versions` → `system_files` (CASCADE)

### **Constraints**

- Unique usernames and emails
- Risk scores 1-5 only
- Calculated fields (risk_score, risk_level)
- NOT NULL on critical fields

---

## 🛠️ Schema Features

### **Calculated Columns**

Auto-calculated, no manual updates needed:

```sql
-- Risk score = likelihood × severity
risk_score INT GENERATED ALWAYS AS (likelihood_score * severity_score)

-- Risk level based on score
risk_level VARCHAR(20) GENERATED ALWAYS AS (
    CASE 
        WHEN score BETWEEN 1 AND 6 THEN 'Low'
        WHEN score BETWEEN 7 AND 12 THEN 'Medium'
        WHEN score BETWEEN 13 AND 25 THEN 'High'
    END
)
```

### **Audit Trail**

Every table has:
- `created_at` - When record created
- `updated_at` - Last modification time

Plus full audit log for sensitive operations.

### **Soft Deletes**

Most tables have `is_active` flag:
- Set to FALSE instead of deleting
- Maintains data integrity
- Allows restoration

---

## 📊 Database Statistics

After installation:

```
Total Tables: 16
Total Views: 2
Total Triggers: 2
Total Indexes: 50+
Default Categories: 10
Default Risk Items: 30+
Default Permissions: 21
```

**Total Database Size:** ~5MB (with defaults)

---

## 🔄 Maintenance

### **Backup Database**

```bash
# Full backup
mysqldump -u root -p scubatricky_risk > backup.sql

# Schema only
mysqldump -u root -p --no-data scubatricky_risk > schema_only.sql

# Data only
mysqldump -u root -p --no-create-info scubatricky_risk > data_only.sql
```

### **Reset Database**

```bash
# Drop and recreate
mysql -u root -p -e "DROP DATABASE scubatricky_risk;"
mysql -u root -p < schema.sql
```

### **Update Schema**

```bash
# Apply changes
mysql -u root -p scubatricky_risk < updates.sql
```

---

## ✅ Verification

### **Check Tables Created**

```sql
SHOW TABLES;
```

Should show 16 tables.

### **Check Default Data**

```sql
-- Check categories
SELECT COUNT(*) FROM categories; -- Should be 10

-- Check risk items
SELECT COUNT(*) FROM risk_items; -- Should be 30+

-- Check admin user
SELECT username, role FROM users; -- Should show admin
```

### **Check Views**

```sql
SHOW FULL TABLES WHERE TABLE_TYPE = 'VIEW';
```

Should show 2 views.

### **Check Triggers**

```sql
SHOW TRIGGERS;
```

Should show 2 triggers.

---

## 🎯 Next Steps After Installation

1. ✅ **Verify Installation**
   ```sql
   SELECT COUNT(*) as table_count FROM information_schema.TABLES 
   WHERE TABLE_SCHEMA = 'scubatricky_risk';
   ```
   Should return 16.

2. ✅ **Update Config**
   Edit `includes/config.php`:
   ```php
   define('DB_NAME', 'scubatricky_risk');
   define('DB_USER', 'your_username');
   define('DB_PASS', 'your_password');
   ```

3. ✅ **Test Login**
   - URL: `http://localhost/public/login.php`
   - Username: `admin`
   - Password: `Admin123!`

4. ✅ **Change Password**
   - Go to Users → Edit Admin
   - Set new secure password

5. ✅ **Configure System**
   - Admin Panel → System Customization
   - Upload logo, set colors
   - Configure SMTP (optional)

6. ✅ **Add Locations**
   - Locations → Add New Location
   - Add your stores/sites

7. ✅ **Customize Risks**
   - Review default risk items
   - Add/modify as needed
   - Add custom categories

---

## 🚨 Important Notes

### **Character Set**

Database uses UTF-8 (utf8mb4):
- Supports all languages
- Emoji support
- Proper text handling

### **Storage Engine**

All tables use InnoDB:
- ACID compliance
- Foreign key support
- Row-level locking
- Crash recovery

### **Collation**

Uses `utf8mb4_unicode_ci`:
- Case-insensitive searches
- Proper international sorting
- Unicode normalization

---

## 📞 Troubleshooting

### **Import Fails**

```bash
# Check MySQL version
mysql --version

# Should be 5.7+ or 8.0+
```

### **Foreign Key Errors**

```sql
-- Disable temporarily
SET FOREIGN_KEY_CHECKS = 0;
SOURCE schema.sql;
SET FOREIGN_KEY_CHECKS = 1;
```

### **Duplicate Entry Errors**

```sql
-- Drop database and recreate
DROP DATABASE IF EXISTS scubatricky_risk;
CREATE DATABASE scubatricky_risk;
USE scubatricky_risk;
SOURCE schema.sql;
```

### **Permission Denied**

```bash
# Grant privileges
mysql -u root -p
GRANT ALL PRIVILEGES ON scubatricky_risk.* TO 'username'@'localhost';
FLUSH PRIVILEGES;
```

---

## 📚 Additional Resources

**Schema File:** `schema.sql`
**Backup Recommended:** Before any changes
**Version:** 3.0 (Complete and Final)
**Updated:** 2026-01-09

---

## ✨ Schema Highlights

✅ **Single file** - No migrations needed
✅ **Production ready** - Fully tested
✅ **Self-contained** - Includes all defaults
✅ **Well-documented** - Clear table purposes
✅ **Optimized** - Performance indexes
✅ **Maintainable** - Clear structure
✅ **Scalable** - Handles growth
✅ **Secure** - Data integrity enforced

**This is the only schema file you'll ever need!** 🎉
