# Custom Fields Guide

## Overview

The Risk Assessment System includes a powerful custom fields feature that allows you to extend location information with any additional data your organization needs. Custom fields are fully configurable and can be managed through the admin interface.

## ✨ Features

- **Unlimited Custom Fields**: Add as many fields as you need
- **Multiple Field Types**: Text, textarea, number, date, time, select, checkbox, URL, email, phone
- **Flexible Configuration**: Set required/optional, display order, help text
- **Admin Management**: Add, edit, activate/deactivate fields via web interface
- **Automatic UI Generation**: Fields automatically appear in location forms
- **Type Validation**: Built-in validation for URLs, emails, phone numbers, etc.

## 📝 Field Types

### Text (Short)
- Single-line text input
- **Use for**: Names, codes, short identifiers
- **Examples**: Loading bay number, access code, zone identifier

### Text Area (Long)
- Multi-line text input
- **Use for**: Descriptions, instructions, notes
- **Examples**: Alternative access route, special instructions, safety notes

### Number
- Numeric input with decimal support
- **Use for**: Measurements, quantities, capacities
- **Examples**: Height restriction (4.2), weight limit (18.5), parking spaces (5)

### Date
- Date picker
- **Use for**: Dates, deadlines, validity periods
- **Examples**: Access permit expiry, last inspection date

### Time
- Time picker (24-hour format)
- **Use for**: Time-specific information
- **Examples**: Quiet hours start time, mandatory break time

### Select Dropdown
- Dropdown with predefined options
- **Use for**: Limited choice fields
- **Examples**: Bay type (covered/uncovered), surface type (asphalt/concrete)
- **Configuration**: Provide JSON array of options: `["Option 1", "Option 2", "Option 3"]`

### Checkbox (Yes/No)
- Boolean checkbox
- **Use for**: Yes/No questions, feature flags
- **Examples**: Forklift available, tail lift required, 24-hour access

### URL
- Web address input with validation
- **Use for**: Links to external resources
- **Examples**: Site map URL, safety documentation link

### Email
- Email address input with validation
- **Use for**: Email contacts
- **Examples**: Facilities manager email, emergency contact email

### Phone Number (Tel)
- Phone number input
- **Use for**: Contact numbers
- **Examples**: Site security phone, gate control number

## 🎯 Common Use Cases

### Delivery Operations
```
Field Name: loading_bay_number
Label: Loading Bay Number
Type: Text
Required: No
Help Text: Specific bay number or identifier (e.g., Bay 3A)

Field Name: height_restriction
Label: Height Restriction (meters)
Type: Number
Required: No
Help Text: Maximum vehicle height allowed

Field Name: weight_restriction
Label: Weight Restriction (tonnes)
Type: Number
Required: No
Help Text: Maximum vehicle weight allowed

Field Name: unloading_equipment
Label: Unloading Equipment Available
Type: Select
Options: ["Forklift", "Pallet Jack", "None - Tail Lift Required", "Loading Dock"]
Required: No
```

### Access Control
```
Field Name: access_code
Label: Gate/Entry Code
Type: Text
Required: No
Help Text: Security code for automated gates

Field Name: security_contact
Label: Security Phone Number
Type: Tel
Required: No
Help Text: 24/7 security contact

Field Name: requires_booking
Label: Requires Pre-Booking
Type: Checkbox
Required: No
Help Text: Must book delivery slot in advance
```

### Site Information
```
Field Name: parking_spaces
Label: Available Parking Spaces
Type: Number
Required: No
Help Text: Number of spaces for waiting/parking

Field Name: alternative_access
Label: Alternative Access Route
Type: Textarea
Required: No
Help Text: Backup route if primary access blocked

Field Name: site_manager_contact
Label: Site Manager Contact
Type: Tel
Required: No
Help Text: Direct phone for on-site manager
```

### Safety & Compliance
```
Field Name: safety_induction_required
Label: Safety Induction Required
Type: Checkbox
Required: No
Help Text: Driver must complete safety induction

Field Name: ppe_requirements
Label: PPE Requirements
Type: Select
Options: ["Standard PPE", "Hi-Vis Only", "Hi-Vis + Hard Hat", "Full PPE + Steel Toe Boots"]
Required: No

Field Name: hazmat_restrictions
Label: Hazardous Material Restrictions
Type: Textarea
Required: No
Help Text: Any restrictions on hazardous goods
```

## 🔧 Managing Custom Fields

### Accessing the Manager

**For Admins Only:**
1. Login with admin account
2. Navigate to "Custom Fields" in the main menu
3. Or go to: `your-domain.com/custom-fields.php`

### Adding a New Field

1. Click "Add Custom Field"
2. Fill in the form:
   - **Field Name**: Database key (lowercase, underscores only) - cannot be changed after creation
   - **Field Label**: Display name shown to users
   - **Field Type**: Choose from dropdown
   - **Display Order**: Controls position in form (lower numbers appear first)
   - **Help Text**: Optional guidance for users
   - **Required**: Check if field must be filled
   - **Active**: Uncheck to hide field without deleting

3. **For Select Fields**: Enter options in JSON format:
   ```json
   ["Option 1", "Option 2", "Option 3"]
   ```

4. Click "Save Field"

### Editing an Existing Field

1. Click "Edit" next to the field
2. Modify any settings except Field Name (immutable)
3. Update as needed
4. Click "Save Field"

### Deactivating a Field

1. Click "Edit" next to the field
2. Uncheck "Active"
3. Save changes
4. Field will no longer appear in forms but data is preserved

### Deleting a Field

⚠️ **Warning**: This permanently deletes the field and all data stored in it!

1. Click "Delete" next to the field
2. Confirm the action
3. Field and all its data are removed

## 💾 Technical Details

### Database Storage

Custom field data is stored as JSON in the `locations.custom_fields` column:

```json
{
  "loading_bay_number": "3A",
  "height_restriction": "4.2",
  "weight_restriction": "18",
  "access_code": "1234",
  "requires_booking": "1"
}
```

### Field Definitions Table

Custom field definitions are stored in `custom_field_definitions`:

```sql
CREATE TABLE custom_field_definitions (
    id INT PRIMARY KEY,
    field_name VARCHAR(100) UNIQUE,
    field_label VARCHAR(200),
    field_type ENUM(...),
    field_options TEXT,
    is_required BOOLEAN,
    display_order INT,
    is_active BOOLEAN,
    help_text TEXT
);
```

### Accessing Custom Fields Programmatically

```php
// Get location with custom fields
$stmt = $pdo->prepare("SELECT * FROM locations WHERE id = ?");
$stmt->execute([$id]);
$location = $stmt->fetch();

// Decode custom fields
$custom_fields = json_decode($location['custom_fields'], true);

// Access specific field
$bay_number = $custom_fields['loading_bay_number'] ?? null;
```

## 🎨 User Experience

### In the Location Form

Custom fields appear in a dedicated "Additional Information" section:
- Grouped together for clarity
- Display order can be customized
- Help text appears below fields
- Required fields marked with asterisk
- Validation applied based on type

### On Location View Page

Custom fields display in a grid layout:
- Only shows fields with values
- Formatted appropriately by type:
  - URLs become clickable links
  - Email addresses become mailto links
  - Phone numbers become tel links
  - Checkboxes show ✓ Yes or ✗ No
  - Text/numbers display as-is

## 📋 Best Practices

### Naming Conventions

**Field Names** (database keys):
- Use lowercase letters and underscores
- Be descriptive but concise
- Examples: `height_restriction`, `access_code`, `parking_spaces`
- **Don't use**: Spaces, capital letters, special characters

**Field Labels** (display names):
- Use proper capitalization
- Be clear and specific
- Include units if applicable
- Examples: "Height Restriction (meters)", "Gate Access Code", "Available Parking Spaces"

### Organizing Fields

- Use display_order to group related fields
- Put most important fields first (lower numbers)
- Consider user workflow when ordering
- Keep similar information together

**Example ordering:**
```
1-10: Physical access (bays, gates, codes)
11-20: Equipment and facilities
21-30: Restrictions and requirements
31-40: Contact information
41-50: Additional notes
```

### Help Text

Good help text:
- ✅ "Maximum vehicle height in meters (e.g., 4.2)"
- ✅ "Contact number for gate security (24/7 available)"
- ✅ "Check if tail lift is mandatory for this location"

Poor help text:
- ❌ "Enter value"
- ❌ "Height"
- ❌ "" (empty)

### Select Field Options

Make options:
- Mutually exclusive
- Comprehensive (cover all cases)
- Clear and unambiguous
- Sorted logically (not alphabetically if order matters)

**Example:**
```json
["No Equipment Available", "Pallet Jack Only", "Forklift Available", "Full Loading Dock"]
```

### Required Fields

Mark as required only if:
- Information is critical for operations
- Data should always be present
- Missing data would cause issues

**Common required fields:**
- Height restriction (if site has restrictions)
- Access code (if controlled access)
- None (most custom fields should be optional)

## 🔄 Migration and Updates

### Adding Fields to Existing System

If you already have locations in the database:

1. Run migration (if adding for first time):
   ```bash
   mysql -u scubatricky_risk -p scubatricky_risk < includes/migration_add_custom_fields.sql
   ```

2. Add custom fields through admin interface

3. Edit existing locations to populate new fields

4. Fields are optional by default, so existing locations work fine

### Changing Field Types

⚠️ **Warning**: Changing field type may cause data loss!

- Text → Textarea: Safe
- Number → Text: Safe
- Text → Number: May lose non-numeric data
- Any → Select: Data may not match options
- Any → Checkbox: Converts to boolean

**Recommendation**: Create new field instead of changing type

### Removing Deprecated Fields

1. Deactivate field (makes it hidden)
2. Wait to ensure no issues
3. Delete field if certain it's not needed
4. Backup database before deleting!

## 🚀 Advanced Usage

### Conditional Fields

While not built-in, you can simulate conditional fields:

1. Use clear naming: `forklift_available`, `forklift_model`
2. In help text: "Only if forklift is available"
3. Train users to skip dependent fields if parent is false

### Integration with Risk Assessments

Custom fields can inform risk assessments:
- Height restrictions → Vehicle access risks
- Equipment availability → Manual handling risks
- Access controls → Security risks

Consider adding custom fields that directly support risk assessment categories.

### Bulk Updates

For many locations, you can:

1. Export data via database query
2. Update in spreadsheet
3. Import back via SQL

```sql
-- Example: Set height restriction for multiple locations
UPDATE locations 
SET custom_fields = JSON_SET(
    COALESCE(custom_fields, '{}'),
    '$.height_restriction',
    '4.2'
)
WHERE location_name LIKE 'Tesco%';
```

## 📊 Reporting

### Query Custom Fields

```sql
-- Find all locations with specific equipment
SELECT location_name, 
       JSON_EXTRACT(custom_fields, '$.unloading_equipment') as equipment
FROM locations
WHERE JSON_EXTRACT(custom_fields, '$.unloading_equipment') IS NOT NULL;

-- Find locations with height restrictions
SELECT location_name,
       JSON_EXTRACT(custom_fields, '$.height_restriction') as max_height
FROM locations
WHERE JSON_EXTRACT(custom_fields, '$.height_restriction') > 0
ORDER BY CAST(JSON_EXTRACT(custom_fields, '$.height_restriction') AS DECIMAL(4,2));
```

### Export to CSV

```php
// Example: Export locations with custom fields
$stmt = $pdo->query("SELECT * FROM locations");
$fp = fopen('locations_export.csv', 'w');

// Get all possible custom fields
$fieldDefs = $pdo->query("SELECT field_name, field_label FROM custom_field_definitions")->fetchAll();

// Headers
$headers = ['Location Name', 'City', 'Postcode'];
foreach ($fieldDefs as $def) {
    $headers[] = $def['field_label'];
}
fputcsv($fp, $headers);

// Data
while ($loc = $stmt->fetch()) {
    $custom = json_decode($loc['custom_fields'], true) ?: [];
    $row = [$loc['location_name'], $loc['city'], $loc['postcode']];
    foreach ($fieldDefs as $def) {
        $row[] = $custom[$def['field_name']] ?? '';
    }
    fputcsv($fp, $row);
}
fclose($fp);
```

## 🎓 Training Users

### For Location Managers

1. Show them Custom Fields page (admins only)
2. Explain each field type
3. Demonstrate adding a field
4. Let them practice on test location

### For Assessors

1. Point out "Additional Information" section in location form
2. Explain purpose of each field
3. Show when fields are required vs. optional
4. Explain how to use help text

### Documentation

Create a quick reference guide:
```
Field Name               What to Enter                    Example
=====================================================================
Loading Bay Number       Bay identifier                   Bay 3A
Height Restriction       Max vehicle height in meters     4.2
Access Code              Gate/entry security code         #1234
Parking Spaces           Number of available spaces       5
```

---

**Custom fields give you the flexibility to capture exactly the information your operation needs, without requiring code changes or database modifications!**
