# Complete Schema Installation Guide

## Overview

The `complete_schema.sql` file contains **ALL** database tables, default data, and features in a single file. Use this for **NEW installations**.

## What's Included

### Core Tables (11)
1. **users** - User accounts and authentication
2. **locations** - Delivery locations with full address details
3. **vehicle_types** - Configurable vehicle types
4. **delivery_volumes** - Configurable delivery volumes
5. **risk_categories** - Risk categories (6 default)
6. **risk_items** - Risk items by category (24 default)
7. **assessments** - Risk assessments
8. **assessment_details** - Risk evaluation data
9. **photos** - Assessment photos

### Custom Fields System (2 tables)
10. **custom_fields** - Custom field definitions
11. **custom_field_values** - Custom field data per location

### Version Control System (5 tables)
12. **assessment_versions** - Version snapshots
13. **assessment_version_details** - Risk items per version
14. **assessment_version_photos** - Photos per version
15. **assessment_change_log** - Complete audit trail
16. **assessment_archives** - Archive metadata

### PDF Template System (2 tables)
17. **pdf_templates** - Template definitions
18. **pdf_field_mapping** - Available fields (35 default)

### Views (2)
19. **view_assessment_summary** - Reporting view
20. **view_high_risk_items** - High risk reporting

### Default Data Included
- ✅ Admin user (username: admin, password: admin123)
- ✅ 5 Vehicle types
- ✅ 5 Delivery volumes
- ✅ 6 Risk categories
- ✅ 24 Risk items across all categories
- ✅ 35 PDF field mappings

## Installation Methods

### Method 1: New Installation (Recommended)

**For brand new installations on a fresh server:**

```bash
# Step 1: Create database user (if doesn't exist)
mysql -u root -p
CREATE USER 'scubatricky_risk'@'localhost' IDENTIFIED BY 'your_secure_password';
FLUSH PRIVILEGES;
exit;

# Step 2: Import complete schema
mysql -u scubatricky_risk -p < includes/complete_schema.sql

# Step 3: Update config.php with your password
nano includes/config.php
# Set DB_PASS to your password

# Step 4: Set permissions
chmod 755 reports/
chmod 755 uploads/
chown www-data:www-data reports/ uploads/
```

**That's it! All tables, data, and features are now installed.**

### Method 2: cPanel Installation

**For shared hosting with cPanel:**

1. **Login to cPanel**

2. **Create Database User** (if needed)
   - MySQL® Databases → MySQL Users
   - Username: `scubatricky_risk`
   - Password: (strong password)
   - Create User

3. **Create Database**
   - MySQL® Databases → MySQL Databases
   - Database Name: `scubatricky_risk`
   - Create Database

4. **Add User to Database**
   - MySQL® Databases → Add User To Database
   - User: `scubatricky_risk`
   - Database: `scubatricky_risk`
   - Privileges: ALL PRIVILEGES
   - Make Changes

5. **Import Schema**
   - phpMyAdmin → Select database
   - Import tab
   - Choose file: `includes/complete_schema.sql`
   - Click Go
   - Wait for "Import has been successfully finished"

6. **Update Config**
   - File Manager → `includes/config.php`
   - Edit file
   - Update `DB_PASS` with your password
   - Save

7. **Done!** Visit your site URL

### Method 3: Command Line (Advanced)

```bash
# One-command installation
mysql -u scubatricky_risk -p scubatricky_risk < includes/complete_schema.sql

# Or if you need to create database first
mysql -u root -p << EOF
CREATE DATABASE IF NOT EXISTS scubatricky_risk CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
GRANT ALL PRIVILEGES ON scubatricky_risk.* TO 'scubatricky_risk'@'localhost' IDENTIFIED BY 'your_password';
FLUSH PRIVILEGES;
EOF

mysql -u scubatricky_risk -p scubatricky_risk < includes/complete_schema.sql
```

## Verification

### Check Installation Success

```bash
# Method 1: Count tables
mysql -u scubatricky_risk -p scubatricky_risk -e "SHOW TABLES;"

# Should show 20 tables

# Method 2: Check default data
mysql -u scubatricky_risk -p scubatricky_risk -e "
SELECT 
    (SELECT COUNT(*) FROM users) as users,
    (SELECT COUNT(*) FROM risk_categories) as categories,
    (SELECT COUNT(*) FROM risk_items) as risk_items,
    (SELECT COUNT(*) FROM vehicle_types) as vehicles,
    (SELECT COUNT(*) FROM delivery_volumes) as volumes,
    (SELECT COUNT(*) FROM pdf_field_mapping) as pdf_fields;
"

# Expected output:
# users: 1 (admin)
# categories: 6
# risk_items: 24
# vehicles: 5
# volumes: 5
# pdf_fields: 35
```

### Test Database Connection

Create `test-db.php` in public folder:

```php
<?php
require_once '../includes/config.php';

echo "<h1>Database Connection Test</h1>";

try {
    $stmt = $pdo->query("
        SELECT 
            (SELECT COUNT(*) FROM users) as users,
            (SELECT COUNT(*) FROM risk_categories) as categories,
            (SELECT COUNT(*) FROM risk_items) as risk_items,
            (SELECT COUNT(*) FROM assessments) as assessments
    ");
    $counts = $stmt->fetch();
    
    echo "<p>✅ <strong>Connection successful!</strong></p>";
    echo "<ul>";
    echo "<li>Users: {$counts['users']}</li>";
    echo "<li>Risk Categories: {$counts['categories']}</li>";
    echo "<li>Risk Items: {$counts['risk_items']}</li>";
    echo "<li>Assessments: {$counts['assessments']}</li>";
    echo "</ul>";
    
} catch (Exception $e) {
    echo "<p>❌ <strong>Connection failed:</strong> " . $e->getMessage() . "</p>";
}
?>
```

Visit: `http://yoursite.com/test-db.php`

**Delete this file after testing!**

## Login

**Default Admin Credentials:**
- Username: `admin`
- Password: `admin123`

**⚠️ IMPORTANT: Change this password immediately after first login!**

## Post-Installation Steps

### 1. Change Admin Password

```
Login → Click user avatar → Settings → Change Password
```

Or via SQL:

```sql
-- Replace 'new_secure_password' with your password
UPDATE users 
SET password = '$2y$10$YourHashedPasswordHere' 
WHERE username = 'admin';
```

Use PHP to generate hash:
```php
<?php echo password_hash('new_secure_password', PASSWORD_DEFAULT); ?>
```

### 2. Create Additional Users

```
Admin Panel → Users → Add User
```

### 3. Configure Locations

```
Locations → Add Location
Fill in delivery site details
```

### 4. Customize Risk Items (Optional)

```
Admin Panel → Risk Categories
Admin Panel → Risk Items
Add/edit/remove items as needed
```

### 5. Configure Custom Fields (Optional)

```
Admin Panel → Custom Fields
Add location-specific data fields
```

### 6. Set Up Vehicle Types (Optional)

```
Admin Panel → Vehicle Types & Volumes
Customize to match your fleet
```

## What's Different from Migrations?

### Old Way (Multiple Files):
```
1. Run schema.sql (base tables)
2. Run migration_add_custom_fields.sql
3. Run migration_add_location_fields.sql
4. Run migration_photos_enhanced.sql
5. Run migration_vehicle_types.sql
6. Run migration_pdf_templates.sql
7. Run migration_version_control.sql
```

### New Way (One File):
```
1. Run complete_schema.sql (everything!)
```

**Benefits:**
- ✅ Single file = simple installation
- ✅ No migration order issues
- ✅ All features included
- ✅ Default data populated
- ✅ Faster setup

## Existing Installations

**If you already have the system installed:**

**DON'T** run complete_schema.sql - it will try to recreate everything!

**Instead:**
- Keep using your existing database
- Already have all tables via migrations
- No action needed

**For reference only:**
```sql
-- To see what you have
SHOW TABLES;

-- Should show all 20 tables
```

## Troubleshooting

### Issue: "Table already exists"

**Cause:** Database already has tables

**Solution:** Either:
1. Drop and recreate database (LOSES ALL DATA!)
```sql
DROP DATABASE scubatricky_risk;
-- Then run complete_schema.sql
```

2. Or keep existing installation (recommended)

### Issue: "Access denied"

**Cause:** Wrong database credentials

**Solution:**
```bash
# Check user exists
mysql -u root -p -e "SELECT user, host FROM mysql.user WHERE user='scubatricky_risk';"

# Grant privileges
mysql -u root -p
GRANT ALL PRIVILEGES ON scubatricky_risk.* TO 'scubatricky_risk'@'localhost';
FLUSH PRIVILEGES;
```

### Issue: "Unknown database"

**Cause:** Database doesn't exist

**Solution:**
```sql
CREATE DATABASE scubatricky_risk CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

### Issue: Import fails at certain line

**Cause:** MySQL version or settings

**Solution:**
```bash
# Import with verbose error messages
mysql -u scubatricky_risk -p scubatricky_risk < includes/complete_schema.sql 2>&1 | tee import.log

# Check log for specific error
cat import.log
```

## File Comparison

### Complete Schema vs Individual Files:

**complete_schema.sql** (ONE FILE):
- All tables
- All default data
- All indexes
- All views
- 20 tables total
- ~500 lines

**Individual Files** (SEVEN FILES):
- schema.sql (base tables)
- migration_add_custom_fields.sql
- migration_add_location_fields.sql
- migration_photos_enhanced.sql
- migration_vehicle_types.sql
- migration_pdf_templates.sql
- migration_version_control.sql

## Backup Before Installation

**Always backup first if updating:**

```bash
# Backup existing database
mysqldump -u scubatricky_risk -p scubatricky_risk > backup_$(date +%Y%m%d).sql

# Backup application files
tar -czf backup_files_$(date +%Y%m%d).tar.gz public/ includes/
```

## Summary

**For New Installations:**
✅ Use `complete_schema.sql`
✅ Single file install
✅ Everything included
✅ Fastest setup

**For Existing Installations:**
✅ Keep your database
✅ No action needed
✅ Already have all features

**Next Steps:**
1. Run complete_schema.sql
2. Update config.php password
3. Login as admin
4. Change admin password
5. Start creating assessments!

**Simple, fast, complete!** 🚀✅
