<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/ThemeManager.php';

$auth = new Auth();
$auth->requireRole('admin');

$db = Database::getInstance();
$themeManager = new ThemeManager();
$pageTitle = 'Color Themes';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create' || $action === 'update') {
        try {
            $data = [
                'theme_name' => $_POST['theme_name'],
                'primary_color' => $_POST['primary_color'],
                'secondary_color' => $_POST['secondary_color'],
                'accent_color' => $_POST['accent_color'],
                'success_color' => $_POST['success_color'],
                'warning_color' => $_POST['warning_color'],
                'danger_color' => $_POST['danger_color'],
                'background_gradient_start' => $_POST['background_gradient_start'],
                'background_gradient_end' => $_POST['background_gradient_end']
            ];
            
            if ($action === 'create') {
                $themeManager->createTheme($data);
                setFlashMessage('Theme created successfully', 'success');
            } else {
                $themeManager->updateTheme($_POST['theme_id'], $data);
                setFlashMessage('Theme updated successfully', 'success');
            }
            
            header('Location: themes.php');
            exit;
        } catch (Exception $e) {
            setFlashMessage('Error: ' . $e->getMessage(), 'danger');
        }
    } elseif ($action === 'set_default') {
        $themeManager->setDefaultTheme($_POST['theme_id']);
        setFlashMessage('Default theme updated', 'success');
        header('Location: themes.php');
        exit;
    } elseif ($action === 'delete') {
        try {
            $themeManager->deleteTheme($_POST['theme_id']);
            setFlashMessage('Theme deleted successfully', 'success');
        } catch (Exception $e) {
            setFlashMessage('Error: ' . $e->getMessage(), 'danger');
        }
        header('Location: themes.php');
        exit;
    }
}

// Get all themes
$themes = $themeManager->getThemes();

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">🎨 Color Themes</h1>
    <button onclick="showCreateModal()" class="btn btn-primary">+ Create Theme</button>
</div>

<div class="card">
    <div class="card-body">
        <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 1.5rem;">
            <?php foreach ($themes as $theme): ?>
                <div class="theme-card" style="border: 2px solid #e5e7eb; border-radius: var(--radius-lg); padding: 1.5rem; position: relative;">
                    <?php if ($theme['is_default']): ?>
                        <div style="position: absolute; top: 10px; right: 10px;">
                            <span class="badge badge-primary">Default</span>
                        </div>
                    <?php endif; ?>
                    
                    <h3 style="margin-bottom: 1rem;"><?php echo htmlspecialchars($theme['theme_name']); ?></h3>
                    
                    <!-- Color Preview -->
                    <div style="display: flex; gap: 0.5rem; margin-bottom: 1rem; flex-wrap: wrap;">
                        <div style="width: 40px; height: 40px; border-radius: 0.5rem; background: <?php echo $theme['primary_color']; ?>; border: 2px solid #fff; box-shadow: 0 2px 4px rgba(0,0,0,0.1);" title="Primary"></div>
                        <div style="width: 40px; height: 40px; border-radius: 0.5rem; background: <?php echo $theme['secondary_color']; ?>; border: 2px solid #fff; box-shadow: 0 2px 4px rgba(0,0,0,0.1);" title="Secondary"></div>
                        <div style="width: 40px; height: 40px; border-radius: 0.5rem; background: <?php echo $theme['accent_color']; ?>; border: 2px solid #fff; box-shadow: 0 2px 4px rgba(0,0,0,0.1);" title="Accent"></div>
                    </div>
                    
                    <!-- Gradient Preview -->
                    <div style="height: 60px; border-radius: 0.5rem; background: linear-gradient(135deg, <?php echo $theme['background_gradient_start']; ?>, <?php echo $theme['background_gradient_end']; ?>); margin-bottom: 1rem;"></div>
                    
                    <!-- Actions -->
                    <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                        <button onclick="applyTheme(<?php echo $theme['id']; ?>)" class="btn btn-sm btn-primary">Apply</button>
                        <button onclick="editTheme(<?php echo $theme['id']; ?>)" class="btn btn-sm btn-secondary">Edit</button>
                        
                        <?php if (!$theme['is_default']): ?>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="set_default">
                                <input type="hidden" name="theme_id" value="<?php echo $theme['id']; ?>">
                                <button type="submit" class="btn btn-sm btn-secondary">Set Default</button>
                            </form>
                            
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this theme?');">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="theme_id" value="<?php echo $theme['id']; ?>">
                                <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<!-- Create/Edit Modal -->
<div id="themeModal" class="modal-overlay" style="display: none;">
    <div class="modal" style="max-width: 700px;">
        <div class="modal-header">
            <h2 class="modal-title" id="modalTitle">Create Theme</h2>
            <button onclick="closeModal()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">&times;</button>
        </div>
        <form method="POST" id="themeForm">
            <input type="hidden" name="action" id="formAction" value="create">
            <input type="hidden" name="theme_id" id="themeId">
            
            <div class="modal-body">
                <div class="form-group">
                    <label class="form-label required">Theme Name</label>
                    <input type="text" name="theme_name" id="themeName" class="form-control" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Primary Color</label>
                        <input type="color" name="primary_color" id="primaryColor" class="form-control" value="#6366f1" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Secondary Color</label>
                        <input type="color" name="secondary_color" id="secondaryColor" class="form-control" value="#8b5cf6" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Accent Color</label>
                        <input type="color" name="accent_color" id="accentColor" class="form-control" value="#ec4899" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Success Color</label>
                        <input type="color" name="success_color" id="successColor" class="form-control" value="#10b981">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Warning Color</label>
                        <input type="color" name="warning_color" id="warningColor" class="form-control" value="#f59e0b">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Danger Color</label>
                        <input type="color" name="danger_color" id="dangerColor" class="form-control" value="#ef4444">
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Background Start</label>
                        <input type="color" name="background_gradient_start" id="bgStart" class="form-control" value="#667eea" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Background End</label>
                        <input type="color" name="background_gradient_end" id="bgEnd" class="form-control" value="#764ba2" required>
                    </div>
                </div>
                
                <!-- Preview -->
                <div style="margin-top: 1.5rem; padding: 1rem; border: 2px solid #e5e7eb; border-radius: var(--radius-lg);">
                    <h4>Preview</h4>
                    <div id="preview" style="height: 100px; border-radius: 0.5rem; margin-top: 0.5rem;"></div>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="submit" class="btn btn-primary">Save Theme</button>
                <button type="button" onclick="closeModal()" class="btn btn-secondary">Cancel</button>
            </div>
        </form>
    </div>
</div>

<script>
const themes = <?php echo json_encode($themes); ?>;

function showCreateModal() {
    document.getElementById('modalTitle').textContent = 'Create Theme';
    document.getElementById('formAction').value = 'create';
    document.getElementById('themeForm').reset();
    document.getElementById('themeModal').style.display = 'flex';
    updatePreview();
}

function editTheme(themeId) {
    const theme = themes.find(t => t.id == themeId);
    if (!theme) return;
    
    document.getElementById('modalTitle').textContent = 'Edit Theme';
    document.getElementById('formAction').value = 'update';
    document.getElementById('themeId').value = theme.id;
    document.getElementById('themeName').value = theme.theme_name;
    document.getElementById('primaryColor').value = theme.primary_color;
    document.getElementById('secondaryColor').value = theme.secondary_color;
    document.getElementById('accentColor').value = theme.accent_color;
    document.getElementById('successColor').value = theme.success_color;
    document.getElementById('warningColor').value = theme.warning_color;
    document.getElementById('dangerColor').value = theme.danger_color;
    document.getElementById('bgStart').value = theme.background_gradient_start;
    document.getElementById('bgEnd').value = theme.background_gradient_end;
    
    document.getElementById('themeModal').style.display = 'flex';
    updatePreview();
}

function closeModal() {
    document.getElementById('themeModal').style.display = 'none';
}

function updatePreview() {
    const start = document.getElementById('bgStart').value;
    const end = document.getElementById('bgEnd').value;
    document.getElementById('preview').style.background = `linear-gradient(135deg, ${start}, ${end})`;
}

function applyTheme(themeId) {
    fetch('api/set-user-theme.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ theme_id: themeId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('Failed to apply theme');
        }
    });
}

// Update preview on color change
['primaryColor', 'secondaryColor', 'accentColor', 'bgStart', 'bgEnd'].forEach(id => {
    document.getElementById(id)?.addEventListener('input', updatePreview);
});

// Close modal on overlay click
document.getElementById('themeModal')?.addEventListener('click', function(e) {
    if (e.target === this) closeModal();
});
</script>

<style>
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
    padding: 1rem;
}

.modal {
    background: white;
    border-radius: var(--radius-xl);
    box-shadow: var(--shadow-2xl);
    width: 100%;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    padding: 1.5rem 2rem;
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-title {
    font-size: 1.5rem;
    font-weight: 700;
    margin: 0;
}

.modal-body {
    padding: 2rem;
}

.modal-footer {
    padding: 1.5rem 2rem;
    border-top: 1px solid var(--gray-200);
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
}
</style>

<?php include '../includes/footer.php'; ?>
