<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';

$auth = new Auth();
$auth->requireRole('admin');

$db = Database::getInstance();
$pageTitle = 'Permissions Management';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'update_permissions') {
        try {
            // Update all permissions
            foreach ($_POST['permissions'] as $permId => $perms) {
                $sql = "UPDATE permissions SET 
                        can_create = ?,
                        can_read = ?,
                        can_update = ?,
                        can_delete = ?
                        WHERE id = ?";
                
                $db->query($sql, [
                    isset($perms['create']) ? 1 : 0,
                    isset($perms['read']) ? 1 : 0,
                    isset($perms['update']) ? 1 : 0,
                    isset($perms['delete']) ? 1 : 0,
                    $permId
                ]);
            }
            
            logAudit($_SESSION['user_id'], 'update', 'permissions', 0);
            setFlashMessage('Permissions updated successfully', 'success');
            header('Location: permissions.php');
            exit;
        } catch (Exception $e) {
            setFlashMessage('Error updating permissions: ' . $e->getMessage(), 'danger');
        }
    } elseif ($action === 'add_resource') {
        try {
            $resource = $_POST['resource_name'];
            $roles = ['admin', 'assessor', 'viewer'];
            
            foreach ($roles as $role) {
                // Check if already exists
                $existing = $db->fetchOne(
                    "SELECT id FROM permissions WHERE role = ? AND resource = ?",
                    [$role, $resource]
                );
                
                if (!$existing) {
                    // Admin gets all permissions by default
                    $canAll = ($role === 'admin') ? 1 : 0;
                    
                    $sql = "INSERT INTO permissions (role, resource, can_create, can_read, can_update, can_delete) 
                            VALUES (?, ?, ?, ?, ?, ?)";
                    $db->query($sql, [$role, $resource, $canAll, 1, $canAll, $canAll]);
                }
            }
            
            setFlashMessage("Resource '$resource' added successfully", 'success');
            header('Location: permissions.php');
            exit;
        } catch (Exception $e) {
            setFlashMessage('Error adding resource: ' . $e->getMessage(), 'danger');
        }
    }
}

// Get all permissions grouped by resource
$permissions = $db->fetchAll("
    SELECT p.*, 
           CONCAT(UPPER(SUBSTRING(p.role, 1, 1)), LOWER(SUBSTRING(p.role, 2))) as role_display
    FROM permissions p
    ORDER BY p.resource, 
             FIELD(p.role, 'admin', 'assessor', 'viewer')
");

// Group by resource
$permissionsByResource = [];
foreach ($permissions as $perm) {
    $permissionsByResource[$perm['resource']][] = $perm;
}

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">🔐 Permissions Management</h1>
    <button onclick="showAddResourceModal()" class="btn btn-primary">+ Add Resource</button>
</div>

<div class="card">
    <div class="card-header">
        <h2 class="card-title">Role-Based Access Control</h2>
    </div>
    <div class="card-body">
        <form method="POST">
            <input type="hidden" name="action" value="update_permissions">
            
            <?php foreach ($permissionsByResource as $resource => $resourcePerms): ?>
                <div style="margin-bottom: 2rem;">
                    <h3 style="padding: 0.75rem 1rem; background: linear-gradient(to right, rgba(99, 102, 241, 0.1), rgba(139, 92, 246, 0.1)); border-left: 4px solid var(--primary); border-radius: var(--radius-lg); margin-bottom: 1rem;">
                        <?php echo ucfirst($resource); ?>
                    </h3>
                    
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>Role</th>
                                    <th style="text-align: center; width: 15%;">Create</th>
                                    <th style="text-align: center; width: 15%;">Read</th>
                                    <th style="text-align: center; width: 15%;">Update</th>
                                    <th style="text-align: center; width: 15%;">Delete</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resourcePerms as $perm): ?>
                                    <tr>
                                        <td><strong><?php echo $perm['role_display']; ?></strong></td>
                                        <td style="text-align: center;">
                                            <label class="checkbox-label">
                                                <input type="checkbox" 
                                                       name="permissions[<?php echo $perm['id']; ?>][create]" 
                                                       <?php echo $perm['can_create'] ? 'checked' : ''; ?>>
                                                <span class="checkmark"></span>
                                            </label>
                                        </td>
                                        <td style="text-align: center;">
                                            <label class="checkbox-label">
                                                <input type="checkbox" 
                                                       name="permissions[<?php echo $perm['id']; ?>][read]" 
                                                       <?php echo $perm['can_read'] ? 'checked' : ''; ?>>
                                                <span class="checkmark"></span>
                                            </label>
                                        </td>
                                        <td style="text-align: center;">
                                            <label class="checkbox-label">
                                                <input type="checkbox" 
                                                       name="permissions[<?php echo $perm['id']; ?>][update]" 
                                                       <?php echo $perm['can_update'] ? 'checked' : ''; ?>>
                                                <span class="checkmark"></span>
                                            </label>
                                        </td>
                                        <td style="text-align: center;">
                                            <label class="checkbox-label">
                                                <input type="checkbox" 
                                                       name="permissions[<?php echo $perm['id']; ?>][delete]" 
                                                       <?php echo $perm['can_delete'] ? 'checked' : ''; ?>>
                                                <span class="checkmark"></span>
                                            </label>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php endforeach; ?>
            
            <div style="margin-top: 2rem;">
                <button type="submit" class="btn btn-primary btn-lg">💾 Save All Permissions</button>
            </div>
        </form>
    </div>
</div>

<!-- Legend -->
<div class="card">
    <div class="card-body">
        <h3 style="margin-bottom: 1rem;">Permission Levels</h3>
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem;">
            <div style="padding: 1rem; background: rgba(16, 185, 129, 0.1); border-radius: var(--radius-lg);">
                <strong style="color: #065f46;">✓ Create</strong><br>
                <small>Can add new records</small>
            </div>
            <div style="padding: 1rem; background: rgba(59, 130, 246, 0.1); border-radius: var(--radius-lg);">
                <strong style="color: #1e40af;">✓ Read</strong><br>
                <small>Can view records</small>
            </div>
            <div style="padding: 1rem; background: rgba(245, 158, 11, 0.1); border-radius: var(--radius-lg);">
                <strong style="color: #92400e;">✓ Update</strong><br>
                <small>Can edit records</small>
            </div>
            <div style="padding: 1rem; background: rgba(239, 68, 68, 0.1); border-radius: var(--radius-lg);">
                <strong style="color: #991b1b;">✓ Delete</strong><br>
                <small>Can remove records</small>
            </div>
        </div>
    </div>
</div>

<!-- Add Resource Modal -->
<div id="addResourceModal" class="modal-overlay" style="display: none;">
    <div class="modal" style="max-width: 500px;">
        <div class="modal-header">
            <h2 class="modal-title">Add New Resource</h2>
            <button onclick="closeAddResourceModal()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">&times;</button>
        </div>
        <form method="POST">
            <input type="hidden" name="action" value="add_resource">
            
            <div class="modal-body">
                <div class="form-group">
                    <label class="form-label required">Resource Name</label>
                    <input type="text" name="resource_name" class="form-control" 
                           placeholder="e.g., reports, documents, settings" required>
                    <small class="form-help">Use lowercase, singular form (e.g., 'report' not 'Reports')</small>
                </div>
                
                <div style="padding: 1rem; background: rgba(59, 130, 246, 0.1); border-radius: var(--radius-lg); margin-top: 1rem;">
                    <strong>ℹ️ Default Permissions:</strong><br>
                    <small>Admin: Full access (CRUD)<br>
                    Assessor: Read only<br>
                    Viewer: Read only</small>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="submit" class="btn btn-primary">Add Resource</button>
                <button type="button" onclick="closeAddResourceModal()" class="btn btn-secondary">Cancel</button>
            </div>
        </form>
    </div>
</div>

<style>
.checkbox-label {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    position: relative;
}

.checkbox-label input[type="checkbox"] {
    width: 24px;
    height: 24px;
    cursor: pointer;
    accent-color: var(--primary);
}

.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
    padding: 1rem;
}

.modal {
    background: white;
    border-radius: var(--radius-xl);
    box-shadow: var(--shadow-2xl);
    width: 100%;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    padding: 1.5rem 2rem;
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-title {
    font-size: 1.5rem;
    font-weight: 700;
    margin: 0;
}

.modal-body {
    padding: 2rem;
}

.modal-footer {
    padding: 1.5rem 2rem;
    border-top: 1px solid var(--gray-200);
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
}
</style>

<script>
function showAddResourceModal() {
    document.getElementById('addResourceModal').style.display = 'flex';
}

function closeAddResourceModal() {
    document.getElementById('addResourceModal').style.display = 'none';
}

// Close modal on overlay click
document.getElementById('addResourceModal')?.addEventListener('click', function(e) {
    if (e.target === this) closeAddResourceModal();
});
</script>

<?php include '../includes/footer.php'; ?>
