<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/PDFTemplateManager.php';

$auth = new Auth();
$auth->requireRole('admin');

$db = Database::getInstance();
$pdfManager = new PDFTemplateManager();
$pageTitle = 'PDF Templates';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create' || $action === 'update') {
        try {
            $data = [
                'template_name' => $_POST['template_name'],
                'header_logo_url' => $_POST['header_logo_url'],
                'header_text' => $_POST['header_text'],
                'footer_text' => $_POST['footer_text'],
                'show_risk_matrix' => isset($_POST['show_risk_matrix']) ? 1 : 0,
                'show_photos' => isset($_POST['show_photos']) ? 1 : 0,
                'show_action_items' => isset($_POST['show_action_items']) ? 1 : 0,
                'show_signatures' => isset($_POST['show_signatures']) ? 1 : 0,
                'page_orientation' => $_POST['page_orientation'],
                'font_family' => $_POST['font_family'],
                'font_size' => $_POST['font_size'],
                'include_fields' => $_POST['include_fields'] ?? [],
                'custom_css' => $_POST['custom_css']
            ];
            
            if ($action === 'create') {
                $pdfManager->createTemplate($data);
                setFlashMessage('PDF template created successfully', 'success');
            } else {
                $pdfManager->updateTemplate($_POST['template_id'], $data);
                setFlashMessage('PDF template updated successfully', 'success');
            }
            
            header('Location: pdf-templates.php');
            exit;
        } catch (Exception $e) {
            setFlashMessage('Error: ' . $e->getMessage(), 'danger');
        }
    } elseif ($action === 'set_default') {
        $pdfManager->setDefaultTemplate($_POST['template_id']);
        setFlashMessage('Default template updated', 'success');
        header('Location: pdf-templates.php');
        exit;
    } elseif ($action === 'delete') {
        try {
            $pdfManager->deleteTemplate($_POST['template_id']);
            setFlashMessage('Template deleted successfully', 'success');
        } catch (Exception $e) {
            setFlashMessage('Error: ' . $e->getMessage(), 'danger');
        }
        header('Location: pdf-templates.php');
        exit;
    }
}

// Get all templates
$templates = $pdfManager->getTemplates();

// Available fields
$availableFields = [
    'title' => 'Assessment Title',
    'location' => 'Location',
    'visit_date' => 'Visit Date',
    'assessor' => 'Assessor Name',
    'vehicle_type' => 'Vehicle Type',
    'delivery_type' => 'Delivery Type',
    'weather' => 'Weather Conditions',
    'description' => 'Description'
];

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">📄 PDF Export Templates</h1>
    <button onclick="showCreateModal()" class="btn btn-primary">+ Create Template</button>
</div>

<div class="table-container">
    <table>
        <thead>
            <tr>
                <th>Template Name</th>
                <th>Orientation</th>
                <th>Font</th>
                <th>Options</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($templates as $template): ?>
                <tr>
                    <td>
                        <strong><?php echo htmlspecialchars($template['template_name']); ?></strong>
                        <?php if ($template['is_default']): ?>
                            <span class="badge badge-primary" style="margin-left: 0.5rem;">Default</span>
                        <?php endif; ?>
                    </td>
                    <td><?php echo ucfirst($template['page_orientation']); ?></td>
                    <td><?php echo $template['font_family']; ?>, <?php echo $template['font_size']; ?>pt</td>
                    <td>
                        <?php if ($template['show_photos']): ?><span class="badge badge-success">Photos</span><?php endif; ?>
                        <?php if ($template['show_action_items']): ?><span class="badge badge-success">Actions</span><?php endif; ?>
                        <?php if ($template['show_signatures']): ?><span class="badge badge-success">Signatures</span><?php endif; ?>
                    </td>
                    <td>
                        <?php if ($template['is_active']): ?>
                            <span class="badge badge-success">Active</span>
                        <?php else: ?>
                            <span class="badge badge-gray">Inactive</span>
                        <?php endif; ?>
                    </td>
                    <td class="table-actions">
                        <button onclick="editTemplate(<?php echo $template['id']; ?>)" class="btn btn-sm btn-secondary">Edit</button>
                        
                        <?php if (!$template['is_default']): ?>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="set_default">
                                <input type="hidden" name="template_id" value="<?php echo $template['id']; ?>">
                                <button type="submit" class="btn btn-sm btn-secondary">Set Default</button>
                            </form>
                            
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this template?');">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="template_id" value="<?php echo $template['id']; ?>">
                                <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                            </form>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<!-- Create/Edit Modal -->
<div id="templateModal" class="modal-overlay" style="display: none;">
    <div class="modal" style="max-width: 800px;">
        <div class="modal-header">
            <h2 class="modal-title" id="modalTitle">Create PDF Template</h2>
            <button onclick="closeModal()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">&times;</button>
        </div>
        <form method="POST" id="templateForm">
            <input type="hidden" name="action" id="formAction" value="create">
            <input type="hidden" name="template_id" id="templateId">
            
            <div class="modal-body">
                <div class="form-group">
                    <label class="form-label required">Template Name</label>
                    <input type="text" name="template_name" id="templateName" class="form-control" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Page Orientation</label>
                        <select name="page_orientation" id="pageOrientation" class="form-select">
                            <option value="portrait">Portrait</option>
                            <option value="landscape">Landscape</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Font Family</label>
                        <select name="font_family" id="fontFamily" class="form-select">
                            <option value="Arial">Arial</option>
                            <option value="Helvetica">Helvetica</option>
                            <option value="Times New Roman">Times New Roman</option>
                            <option value="Courier">Courier</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Font Size (pt)</label>
                        <input type="number" name="font_size" id="fontSize" class="form-control" value="12" min="8" max="16">
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Header Logo URL</label>
                    <input type="url" name="header_logo_url" id="headerLogoUrl" class="form-control" placeholder="https://example.com/logo.png">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Header Text</label>
                    <input type="text" name="header_text" id="headerText" class="form-control" value="HGV Delivery Risk Assessment Report">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Footer Text</label>
                    <input type="text" name="footer_text" id="footerText" class="form-control" value="Confidential - Generated on {{date}}">
                    <small class="form-help">Use {{date}} for current date</small>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Include Fields</label>
                    <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 0.5rem;">
                        <?php foreach ($availableFields as $key => $label): ?>
                            <label style="display: flex; align-items: center; gap: 0.5rem;">
                                <input type="checkbox" name="include_fields[]" value="<?php echo $key; ?>" class="include-field">
                                <span><?php echo $label; ?></span>
                            </label>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Options</label>
                    <div style="display: flex; flex-direction: column; gap: 0.5rem;">
                        <label style="display: flex; align-items: center; gap: 0.5rem;">
                            <input type="checkbox" name="show_risk_matrix" id="showRiskMatrix" checked>
                            <span>Show Risk Matrix</span>
                        </label>
                        <label style="display: flex; align-items: center; gap: 0.5rem;">
                            <input type="checkbox" name="show_photos" id="showPhotos" checked>
                            <span>Show Photos</span>
                        </label>
                        <label style="display: flex; align-items: center; gap: 0.5rem;">
                            <input type="checkbox" name="show_action_items" id="showActionItems" checked>
                            <span>Show Action Items</span>
                        </label>
                        <label style="display: flex; align-items: center; gap: 0.5rem;">
                            <input type="checkbox" name="show_signatures" id="showSignatures">
                            <span>Show Signature Section</span>
                        </label>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Custom CSS</label>
                    <textarea name="custom_css" id="customCss" class="form-textarea" rows="4" placeholder="/* Add custom CSS here */"></textarea>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="submit" class="btn btn-primary">Save Template</button>
                <button type="button" onclick="closeModal()" class="btn btn-secondary">Cancel</button>
            </div>
        </form>
    </div>
</div>

<script>
const templates = <?php echo json_encode($templates); ?>;

function showCreateModal() {
    document.getElementById('modalTitle').textContent = 'Create PDF Template';
    document.getElementById('formAction').value = 'create';
    document.getElementById('templateForm').reset();
    document.querySelectorAll('.include-field').forEach(cb => cb.checked = true);
    document.getElementById('templateModal').style.display = 'flex';
}

function editTemplate(templateId) {
    const template = templates.find(t => t.id == templateId);
    if (!template) return;
    
    document.getElementById('modalTitle').textContent = 'Edit PDF Template';
    document.getElementById('formAction').value = 'update';
    document.getElementById('templateId').value = template.id;
    document.getElementById('templateName').value = template.template_name;
    document.getElementById('pageOrientation').value = template.page_orientation;
    document.getElementById('fontFamily').value = template.font_family;
    document.getElementById('fontSize').value = template.font_size;
    document.getElementById('headerLogoUrl').value = template.header_logo_url || '';
    document.getElementById('headerText').value = template.header_text || '';
    document.getElementById('footerText').value = template.footer_text || '';
    document.getElementById('showRiskMatrix').checked = template.show_risk_matrix == 1;
    document.getElementById('showPhotos').checked = template.show_photos == 1;
    document.getElementById('showActionItems').checked = template.show_action_items == 1;
    document.getElementById('showSignatures').checked = template.show_signatures == 1;
    document.getElementById('customCss').value = template.custom_css || '';
    
    // Set include fields
    const includeFields = template.include_fields ? JSON.parse(template.include_fields) : [];
    document.querySelectorAll('.include-field').forEach(cb => {
        cb.checked = includeFields.length === 0 || includeFields.includes(cb.value);
    });
    
    document.getElementById('templateModal').style.display = 'flex';
}

function closeModal() {
    document.getElementById('templateModal').style.display = 'none';
}

document.getElementById('templateModal')?.addEventListener('click', function(e) {
    if (e.target === this) closeModal();
});
</script>

<style>
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
    padding: 1rem;
}

.modal {
    background: white;
    border-radius: var(--radius-xl);
    box-shadow: var(--shadow-2xl);
    width: 100%;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    padding: 1.5rem 2rem;
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-title {
    font-size: 1.5rem;
    font-weight: 700;
    margin: 0;
}

.modal-body {
    padding: 2rem;
}

.modal-footer {
    padding: 1.5rem 2rem;
    border-top: 1px solid var(--gray-200);
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
}
</style>

<?php include '../includes/footer.php'; ?>
