<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';

$auth = new Auth();
$auth->requireLogin();
$db = Database::getInstance();
$pageTitle = 'Locations';

$locations = $db->fetchAll("SELECT * FROM locations WHERE active = 1 ORDER BY name");

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">📍 Locations</h1>
    <?php if ($auth->hasPermission('locations', 'create')): ?>
        <a href="location-create.php" class="btn btn-primary">+ New Location</a>
    <?php endif; ?>
</div>

<?php if (empty($locations)): ?>
    <div class="card">
        <div class="card-body" style="text-align: center; padding: 3rem;">
            <div style="font-size: 4rem; margin-bottom: 1rem;">📍</div>
            <h3>No Locations Yet</h3>
            <p style="color: var(--gray-600); margin-bottom: 1.5rem;">
                Create your first location to start managing assessments.
            </p>
            <?php if ($auth->hasPermission('locations', 'create')): ?>
                <a href="location-create.php" class="btn btn-primary">+ Create First Location</a>
            <?php endif; ?>
        </div>
    </div>
<?php else: ?>
    <div class="locations-grid">
        <?php foreach ($locations as $location): 
            $openingHours = json_decode($location['opening_hours'] ?? '{}', true);
            $deliveryWindows = json_decode($location['delivery_windows'] ?? '{}', true);
            
            // Get today's info
            $today = strtolower(date('l'));
            $todayHours = $openingHours[$today] ?? null;
            $todayWindows = $deliveryWindows[$today] ?? [];
            
            // Check if open now
            $isOpenNow = false;
            $nextOpen = null;
            if ($todayHours && !($todayHours['closed'] ?? true)) {
                $now = date('H:i');
                $isOpenNow = $now >= ($todayHours['open'] ?? '00:00') && 
                             $now <= ($todayHours['close'] ?? '23:59');
            }
        ?>
            <div class="location-card card">
                <div class="card-header" style="background: linear-gradient(135deg, var(--primary-light), var(--primary));">
                    <h3 style="margin: 0; color: white; font-size: 1.25rem;">
                        <?php echo htmlspecialchars($location['name']); ?>
                    </h3>
                    <?php if ($location['chain']): ?>
                        <span style="color: rgba(255,255,255,0.9); font-size: 0.875rem;">
                            <?php echo htmlspecialchars($location['chain']); ?>
                        </span>
                    <?php endif; ?>
                </div>
                
                <div class="card-body">
                    <!-- Address -->
                    <?php if ($location['address']): ?>
                        <div style="margin-bottom: 1rem; padding-bottom: 1rem; border-bottom: 1px solid var(--gray-200);">
                            <div style="font-size: 0.875rem; color: var(--gray-600); margin-bottom: 0.25rem;">📮 Address</div>
                            <div>
                                <?php echo htmlspecialchars($location['address']); ?>
                                <?php if ($location['city']): ?>
                                    <br><?php echo htmlspecialchars($location['city']); ?>
                                <?php endif; ?>
                                <?php if ($location['postcode']): ?>
                                    <?php echo htmlspecialchars($location['postcode']); ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Opening Status -->
                    <div style="margin-bottom: 1rem; padding: 0.75rem; background: <?php echo $isOpenNow ? 'var(--success-light)' : 'var(--gray-100)'; ?>; border-radius: var(--radius-md);">
                        <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.25rem;">
                            <span style="font-size: 1.25rem;"><?php echo $isOpenNow ? '🟢' : '🔴'; ?></span>
                            <strong><?php echo $isOpenNow ? 'Open Now' : 'Closed'; ?></strong>
                        </div>
                        <?php if ($todayHours && !($todayHours['closed'] ?? true)): ?>
                            <div style="font-size: 0.875rem; color: var(--gray-600);">
                                Today: <?php echo $todayHours['open']; ?> - <?php echo $todayHours['close']; ?>
                            </div>
                        <?php else: ?>
                            <div style="font-size: 0.875rem; color: var(--gray-600);">
                                Closed today
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Delivery Windows Today -->
                    <?php if (!empty($todayWindows)): ?>
                        <div style="margin-bottom: 1rem;">
                            <div style="font-size: 0.875rem; color: var(--gray-600); margin-bottom: 0.5rem;">
                                🚚 Delivery Windows Today
                            </div>
                            <div style="display: flex; flex-wrap: wrap; gap: 0.5rem;">
                                <?php foreach ($todayWindows as $window): ?>
                                    <span class="badge badge-primary">
                                        <?php echo $window['name'] ?? 'Window'; ?>: 
                                        <?php echo $window['start']; ?>-<?php echo $window['end']; ?>
                                    </span>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Contact Info -->
                    <?php if ($location['contact_name'] || $location['contact_phone']): ?>
                        <div style="margin-bottom: 1rem; font-size: 0.875rem;">
                            <div style="color: var(--gray-600); margin-bottom: 0.25rem;">👤 Contact</div>
                            <?php if ($location['contact_name']): ?>
                                <div><?php echo htmlspecialchars($location['contact_name']); ?></div>
                            <?php endif; ?>
                            <?php if ($location['contact_phone']): ?>
                                <div>📞 <?php echo htmlspecialchars($location['contact_phone']); ?></div>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Parking Info -->
                    <?php if ($location['parking_info']): ?>
                        <div style="margin-bottom: 1rem; padding: 0.5rem; background: var(--warning-light); border-radius: var(--radius-md); font-size: 0.875rem;">
                            <strong>🅿️ Parking:</strong> <?php echo htmlspecialchars(substr($location['parking_info'], 0, 100)); ?>
                            <?php if (strlen($location['parking_info']) > 100): ?>...<?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="card-footer" style="display: flex; gap: 0.5rem;">
                    <a href="location-view.php?id=<?php echo $location['id']; ?>" class="btn btn-sm btn-primary" style="flex: 1;">
                        View Details
                    </a>
                    <?php if ($auth->hasPermission('locations', 'update')): ?>
                        <a href="location-edit.php?id=<?php echo $location['id']; ?>" class="btn btn-sm btn-secondary">
                            Edit
                        </a>
                    <?php endif; ?>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
<?php endif; ?>

<style>
.locations-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.location-card {
    transition: transform 0.2s, box-shadow 0.2s;
}

.location-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-xl);
}

.card-footer {
    padding: 1rem;
    background: var(--gray-50);
    border-top: 1px solid var(--gray-200);
}
</style>

<?php include '../includes/footer.php'; ?>
