<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';

$auth = new Auth();
$auth->requireLogin();
$db = Database::getInstance();
$pageTitle = 'Location Details';

$locationId = $_GET['id'] ?? 0;
$location = $db->fetchOne("SELECT * FROM locations WHERE id = ?", [$locationId]);

if (!$location) {
    setFlashMessage('Location not found', 'danger');
    header('Location: locations.php');
    exit;
}

$openingHours = json_decode($location['opening_hours'] ?? '{}', true) ?: [];
$deliveryWindows = json_decode($location['delivery_windows'] ?? '{}', true) ?: [];

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">📍 <?php echo htmlspecialchars($location['name']); ?></h1>
    <div style="display: flex; gap: 0.5rem;">
        <a href="locations.php" class="btn btn-secondary">← Back</a>
        <?php if ($auth->hasPermission('locations', 'update')): ?>
            <a href="location-edit.php?id=<?php echo $location['id']; ?>" class="btn btn-primary">Edit</a>
        <?php endif; ?>
    </div>
</div>

<div class="card">
    <div class="card-header">
        <h2 class="card-title">Basic Information</h2>
    </div>
    <div class="card-body">
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1.5rem;">
            <div>
                <strong>Location Name:</strong><br>
                <?php echo htmlspecialchars($location['name']); ?>
            </div>
            
            <?php if ($location['chain']): ?>
                <div>
                    <strong>Chain/Group:</strong><br>
                    <?php echo htmlspecialchars($location['chain']); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($location['address']): ?>
                <div>
                    <strong>Address:</strong><br>
                    <?php echo nl2br(htmlspecialchars($location['address'])); ?>
                    <?php if ($location['city']): ?><br><?php echo htmlspecialchars($location['city']); ?><?php endif; ?>
                    <?php if ($location['postcode']): ?><br><?php echo htmlspecialchars($location['postcode']); ?><?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Contact Information -->
<?php if ($location['contact_name'] || $location['contact_phone'] || $location['contact_email']): ?>
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">👤 Contact Information</h2>
        </div>
        <div class="card-body">
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1.5rem;">
                <?php if ($location['contact_name']): ?>
                    <div>
                        <strong>Contact Name:</strong><br>
                        <?php echo htmlspecialchars($location['contact_name']); ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($location['contact_phone']): ?>
                    <div>
                        <strong>Phone:</strong><br>
                        <a href="tel:<?php echo htmlspecialchars($location['contact_phone']); ?>">
                            <?php echo htmlspecialchars($location['contact_phone']); ?>
                        </a>
                    </div>
                <?php endif; ?>
                
                <?php if ($location['contact_email']): ?>
                    <div>
                        <strong>Email:</strong><br>
                        <a href="mailto:<?php echo htmlspecialchars($location['contact_email']); ?>">
                            <?php echo htmlspecialchars($location['contact_email']); ?>
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- Opening Hours -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">⏰ Opening Hours</h2>
    </div>
    <div class="card-body">
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>Day</th>
                        <th>Opening Time</th>
                        <th>Closing Time</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $days = [
                        'monday' => 'Monday',
                        'tuesday' => 'Tuesday',
                        'wednesday' => 'Wednesday',
                        'thursday' => 'Thursday',
                        'friday' => 'Friday',
                        'saturday' => 'Saturday',
                        'sunday' => 'Sunday'
                    ];
                    
                    foreach ($days as $key => $label):
                        $dayHours = $openingHours[$key] ?? ['closed' => true];
                        $isClosed = $dayHours['closed'] ?? true;
                    ?>
                        <tr style="<?php echo $isClosed ? 'opacity: 0.6;' : ''; ?>">
                            <td><strong><?php echo $label; ?></strong></td>
                            <td><?php echo !$isClosed ? ($dayHours['open'] ?? '—') : '—'; ?></td>
                            <td><?php echo !$isClosed ? ($dayHours['close'] ?? '—') : '—'; ?></td>
                            <td>
                                <?php if ($isClosed): ?>
                                    <span class="badge badge-gray">Closed</span>
                                <?php else: ?>
                                    <span class="badge badge-success">Open</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Delivery Windows -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">🚚 Delivery Time Windows</h2>
    </div>
    <div class="card-body">
        <?php 
        $hasDeliveryWindows = false;
        foreach ($deliveryWindows as $windows) {
            if (!empty($windows)) {
                $hasDeliveryWindows = true;
                break;
            }
        }
        ?>
        
        <?php if (!$hasDeliveryWindows): ?>
            <p style="text-align: center; color: var(--gray-500); padding: 2rem;">
                No delivery windows configured
            </p>
        <?php else: ?>
            <div style="display: grid; gap: 1.5rem;">
                <?php foreach ($days as $key => $label):
                    $dayWindows = $deliveryWindows[$key] ?? [];
                    if (empty($dayWindows)) continue;
                ?>
                    <div style="padding: 1rem; background: var(--gray-50); border-radius: var(--radius-lg);">
                        <h4 style="margin: 0 0 0.75rem 0;"><?php echo $label; ?></h4>
                        <div style="display: flex; flex-wrap: wrap; gap: 0.75rem;">
                            <?php foreach ($dayWindows as $window): ?>
                                <div style="padding: 0.75rem 1rem; background: white; border: 2px solid var(--primary); border-radius: var(--radius-md);">
                                    <div style="font-weight: 600; color: var(--primary); margin-bottom: 0.25rem;">
                                        <?php echo htmlspecialchars($window['name'] ?? 'Window'); ?>
                                    </div>
                                    <div style="font-size: 0.875rem; color: var(--gray-600);">
                                        <?php echo $window['start']; ?> - <?php echo $window['end']; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Delivery & Access Info -->
<?php if ($location['parking_info'] || $location['delivery_restrictions'] || $location['special_instructions']): ?>
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Delivery & Access Information</h2>
        </div>
        <div class="card-body">
            <?php if ($location['parking_info']): ?>
                <div style="margin-bottom: 1.5rem; padding: 1rem; background: var(--info-light); border-left: 4px solid var(--info); border-radius: var(--radius-md);">
                    <strong>🅿️ Parking Information:</strong><br>
                    <?php echo nl2br(htmlspecialchars($location['parking_info'])); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($location['delivery_restrictions']): ?>
                <div style="margin-bottom: 1.5rem; padding: 1rem; background: var(--warning-light); border-left: 4px solid var(--warning); border-radius: var(--radius-md);">
                    <strong>⚠️ Delivery Restrictions:</strong><br>
                    <?php echo nl2br(htmlspecialchars($location['delivery_restrictions'])); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($location['special_instructions']): ?>
                <div style="padding: 1rem; background: var(--primary-light); border-left: 4px solid var(--primary); border-radius: var(--radius-md);">
                    <strong>📋 Special Instructions:</strong><br>
                    <?php echo nl2br(htmlspecialchars($location['special_instructions'])); ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
<?php endif; ?>

<!-- Map/Coordinates -->
<?php if ($location['latitude'] || $location['longitude'] || $location['what3words']): ?>
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">🗺️ Location Coordinates</h2>
        </div>
        <div class="card-body">
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1.5rem;">
                <?php if ($location['latitude']): ?>
                    <div>
                        <strong>Latitude:</strong><br>
                        <?php echo htmlspecialchars($location['latitude']); ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($location['longitude']): ?>
                    <div>
                        <strong>Longitude:</strong><br>
                        <?php echo htmlspecialchars($location['longitude']); ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($location['what3words']): ?>
                    <div>
                        <strong>what3words:</strong><br>
                        <?php echo htmlspecialchars($location['what3words']); ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <?php if ($location['latitude'] && $location['longitude']): ?>
                <div style="margin-top: 1rem;">
                    <a href="https://www.google.com/maps?q=<?php echo $location['latitude']; ?>,<?php echo $location['longitude']; ?>" 
                       target="_blank" class="btn btn-primary">
                        View on Google Maps
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>
<?php endif; ?>

<?php include '../includes/footer.php'; ?>
