<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';

$auth = new Auth();
$auth->requireLogin();
$auth->hasPermission('locations', 'update');

$db = Database::getInstance();
$pageTitle = 'Edit Location';

$locationId = $_GET['id'] ?? 0;

// Get location
$location = $db->fetchOne("SELECT * FROM locations WHERE id = ?", [$locationId]);

if (!$location) {
    setFlashMessage('Location not found', 'danger');
    header('Location: locations.php');
    exit;
}

// Parse JSON fields
$openingHours = json_decode($location['opening_hours'] ?? '{}', true) ?: [];
$deliveryWindows = json_decode($location['delivery_windows'] ?? '{}', true) ?: [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Prepare opening hours JSON
        $newOpeningHours = [];
        $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
        
        foreach ($days as $day) {
            $newOpeningHours[$day] = [
                'open' => $_POST["open_{$day}"] ?? null,
                'close' => $_POST["close_{$day}"] ?? null,
                'closed' => isset($_POST["closed_{$day}"]) ? true : false
            ];
        }
        
        // Prepare delivery windows JSON
        $newDeliveryWindows = [];
        foreach ($days as $day) {
            $newDeliveryWindows[$day] = [];
            if (isset($_POST["delivery_windows_{$day}"])) {
                foreach ($_POST["delivery_windows_{$day}"] as $window) {
                    if (!empty($window['start']) && !empty($window['end'])) {
                        $newDeliveryWindows[$day][] = [
                            'start' => $window['start'],
                            'end' => $window['end'],
                            'name' => $window['name'] ?? ''
                        ];
                    }
                }
            }
        }
        
        $sql = "UPDATE locations SET
            name = ?, chain = ?, address = ?, city = ?, postcode = ?,
            latitude = ?, longitude = ?, what3words = ?,
            opening_hours = ?, delivery_windows = ?, special_instructions = ?,
            contact_name = ?, contact_phone = ?, contact_email = ?,
            parking_info = ?, delivery_restrictions = ?
            WHERE id = ?";
        
        $db->query($sql, [
            $_POST['name'],
            $_POST['chain'] ?? null,
            $_POST['address'] ?? null,
            $_POST['city'] ?? null,
            $_POST['postcode'] ?? null,
            $_POST['latitude'] ?? null,
            $_POST['longitude'] ?? null,
            $_POST['what3words'] ?? null,
            json_encode($newOpeningHours),
            json_encode($newDeliveryWindows),
            $_POST['special_instructions'] ?? null,
            $_POST['contact_name'] ?? null,
            $_POST['contact_phone'] ?? null,
            $_POST['contact_email'] ?? null,
            $_POST['parking_info'] ?? null,
            $_POST['delivery_restrictions'] ?? null,
            $locationId
        ]);
        
        // Log the action
        logAudit('update', 'location', $locationId, $location, $_POST, $_POST['name']);
        
        setFlashMessage('Location updated successfully', 'success');
        header('Location: locations.php');
        exit;
    } catch (Exception $e) {
        setFlashMessage('Error: ' . $e->getMessage(), 'danger');
    }
}

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">Edit Location</h1>
    <a href="locations.php" class="btn btn-secondary">← Back to Locations</a>
</div>

<form method="POST">
    <!-- Basic Information -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Basic Information</h2>
        </div>
        <div class="card-body">
            <div class="form-row">
                <div class="form-group">
                    <label class="form-label required">Location Name</label>
                    <input type="text" name="name" class="form-control" required 
                           value="<?php echo htmlspecialchars($location['name']); ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Chain/Group</label>
                    <input type="text" name="chain" class="form-control" 
                           value="<?php echo htmlspecialchars($location['chain'] ?? ''); ?>">
                </div>
            </div>
            
            <div class="form-group">
                <label class="form-label">Address</label>
                <input type="text" name="address" class="form-control" 
                       value="<?php echo htmlspecialchars($location['address'] ?? ''); ?>">
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label class="form-label">City</label>
                    <input type="text" name="city" class="form-control" 
                           value="<?php echo htmlspecialchars($location['city'] ?? ''); ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Postcode</label>
                    <input type="text" name="postcode" class="form-control" 
                           value="<?php echo htmlspecialchars($location['postcode'] ?? ''); ?>">
                </div>
            </div>
        </div>
    </div>
    
    <!-- Contact Information -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Contact Information</h2>
        </div>
        <div class="card-body">
            <div class="form-row">
                <div class="form-group">
                    <label class="form-label">Contact Name</label>
                    <input type="text" name="contact_name" class="form-control" 
                           value="<?php echo htmlspecialchars($location['contact_name'] ?? ''); ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Contact Phone</label>
                    <input type="tel" name="contact_phone" class="form-control" 
                           value="<?php echo htmlspecialchars($location['contact_phone'] ?? ''); ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Contact Email</label>
                    <input type="email" name="contact_email" class="form-control" 
                           value="<?php echo htmlspecialchars($location['contact_email'] ?? ''); ?>">
                </div>
            </div>
        </div>
    </div>
    
    <!-- Opening Hours -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">⏰ Opening Hours</h2>
        </div>
        <div class="card-body">
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Day</th>
                            <th>Opening Time</th>
                            <th>Closing Time</th>
                            <th>Closed</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        $days = [
                            'monday' => 'Monday',
                            'tuesday' => 'Tuesday',
                            'wednesday' => 'Wednesday',
                            'thursday' => 'Thursday',
                            'friday' => 'Friday',
                            'saturday' => 'Saturday',
                            'sunday' => 'Sunday'
                        ];
                        
                        foreach ($days as $key => $label): 
                            $dayHours = $openingHours[$key] ?? ['open' => '', 'close' => '', 'closed' => false];
                        ?>
                            <tr>
                                <td><strong><?php echo $label; ?></strong></td>
                                <td>
                                    <input type="time" name="open_<?php echo $key; ?>" class="form-control" 
                                           value="<?php echo $dayHours['open'] ?? ''; ?>"
                                           id="open_<?php echo $key; ?>"
                                           <?php echo ($dayHours['closed'] ?? false) ? 'disabled' : ''; ?>>
                                </td>
                                <td>
                                    <input type="time" name="close_<?php echo $key; ?>" class="form-control" 
                                           value="<?php echo $dayHours['close'] ?? ''; ?>"
                                           id="close_<?php echo $key; ?>"
                                           <?php echo ($dayHours['closed'] ?? false) ? 'disabled' : ''; ?>>
                                </td>
                                <td>
                                    <label style="display: flex; align-items: center; gap: 0.5rem;">
                                        <input type="checkbox" name="closed_<?php echo $key; ?>" 
                                               <?php echo ($dayHours['closed'] ?? false) ? 'checked' : ''; ?>
                                               onchange="toggleDayHours('<?php echo $key; ?>', this.checked)">
                                        <span>Closed</span>
                                    </label>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <!-- Delivery Windows -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">🚚 Delivery Time Windows</h2>
        </div>
        <div class="card-body">
            <?php foreach ($days as $key => $label): 
                $dayWindows = $deliveryWindows[$key] ?? [];
            ?>
                <div class="delivery-day-section" style="margin-bottom: 2rem; padding: 1rem; background: var(--gray-50); border-radius: var(--radius-lg);">
                    <h4><?php echo $label; ?></h4>
                    <div id="windows_<?php echo $key; ?>">
                        <?php 
                        if (empty($dayWindows)) {
                            $dayWindows = [['start' => '', 'end' => '', 'name' => '']];
                        }
                        foreach ($dayWindows as $idx => $window): 
                        ?>
                            <div class="delivery-window" style="display: grid; grid-template-columns: 1fr 1fr 2fr auto; gap: 0.5rem; margin-bottom: 0.5rem;">
                                <input type="time" name="delivery_windows_<?php echo $key; ?>[<?php echo $idx; ?>][start]" 
                                       class="form-control" value="<?php echo $window['start'] ?? ''; ?>">
                                <input type="time" name="delivery_windows_<?php echo $key; ?>[<?php echo $idx; ?>][end]" 
                                       class="form-control" value="<?php echo $window['end'] ?? ''; ?>">
                                <input type="text" name="delivery_windows_<?php echo $key; ?>[<?php echo $idx; ?>][name]" 
                                       class="form-control" value="<?php echo htmlspecialchars($window['name'] ?? ''); ?>" 
                                       placeholder="Window Name">
                                <button type="button" onclick="removeWindow(this)" class="btn btn-sm btn-danger">×</button>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <button type="button" onclick="addWindow('<?php echo $key; ?>')" class="btn btn-sm btn-secondary">
                        + Add Window
                    </button>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    
    <!-- Delivery Information -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Delivery & Access Information</h2>
        </div>
        <div class="card-body">
            <div class="form-group">
                <label class="form-label">Parking Information</label>
                <textarea name="parking_info" class="form-textarea" rows="3"><?php echo htmlspecialchars($location['parking_info'] ?? ''); ?></textarea>
            </div>
            
            <div class="form-group">
                <label class="form-label">Delivery Restrictions</label>
                <textarea name="delivery_restrictions" class="form-textarea" rows="3"><?php echo htmlspecialchars($location['delivery_restrictions'] ?? ''); ?></textarea>
            </div>
            
            <div class="form-group">
                <label class="form-label">Special Instructions</label>
                <textarea name="special_instructions" class="form-textarea" rows="3"><?php echo htmlspecialchars($location['special_instructions'] ?? ''); ?></textarea>
            </div>
        </div>
    </div>
    
    <!-- Location Coordinates -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Location Coordinates</h2>
        </div>
        <div class="card-body">
            <div class="form-row">
                <div class="form-group">
                    <label class="form-label">Latitude</label>
                    <input type="text" name="latitude" class="form-control" 
                           value="<?php echo htmlspecialchars($location['latitude'] ?? ''); ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Longitude</label>
                    <input type="text" name="longitude" class="form-control" 
                           value="<?php echo htmlspecialchars($location['longitude'] ?? ''); ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">what3words</label>
                    <input type="text" name="what3words" class="form-control" 
                           value="<?php echo htmlspecialchars($location['what3words'] ?? ''); ?>">
                </div>
            </div>
        </div>
    </div>
    
    <div style="display: flex; gap: 1rem; margin-bottom: 2rem;">
        <button type="submit" class="btn btn-primary btn-lg">💾 Update Location</button>
        <a href="locations.php" class="btn btn-secondary btn-lg">Cancel</a>
    </div>
</form>

<script>
function toggleDayHours(day, closed) {
    document.getElementById('open_' + day).disabled = closed;
    document.getElementById('close_' + day).disabled = closed;
    if (closed) {
        document.getElementById('open_' + day).value = '';
        document.getElementById('close_' + day).value = '';
    }
}

let windowCounters = {};
<?php foreach ($days as $key => $label): ?>
windowCounters['<?php echo $key; ?>'] = <?php echo count($deliveryWindows[$key] ?? []); ?>;
<?php endforeach; ?>

function addWindow(day) {
    const container = document.getElementById('windows_' + day);
    const index = windowCounters[day]++;
    
    const windowDiv = document.createElement('div');
    windowDiv.className = 'delivery-window';
    windowDiv.style.cssText = 'display: grid; grid-template-columns: 1fr 1fr 2fr auto; gap: 0.5rem; margin-bottom: 0.5rem;';
    
    windowDiv.innerHTML = `
        <input type="time" name="delivery_windows_${day}[${index}][start]" class="form-control">
        <input type="time" name="delivery_windows_${day}[${index}][end]" class="form-control">
        <input type="text" name="delivery_windows_${day}[${index}][name]" class="form-control" placeholder="Window Name">
        <button type="button" onclick="removeWindow(this)" class="btn btn-sm btn-danger">×</button>
    `;
    
    container.appendChild(windowDiv);
}

function removeWindow(button) {
    button.closest('.delivery-window').remove();
}
</script>

<?php include '../includes/footer.php'; ?>
