<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/HazardLibrary.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$hazardLib = new HazardLibrary();
$pageTitle = 'View Hazard';

$id = $_GET['id'] ?? 0;
$hazard = $hazardLib->getHazard($id);

if (!$hazard) {
    setFlashMessage('Hazard not found', 'danger');
    header('Location: hazard-library.php');
    exit;
}

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">⚠️ <?php echo htmlspecialchars($hazard['name']); ?></h1>
    <div style="display: flex; gap: 0.5rem;">
        <a href="hazard-library.php" class="btn btn-secondary">← Back to Library</a>
        <?php if ($auth->hasPermission('assessments', 'update')): ?>
            <a href="hazard-edit.php?id=<?php echo $id; ?>" class="btn btn-primary">✏️ Edit</a>
        <?php endif; ?>
    </div>
</div>

<!-- Quick Info Banner -->
<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 2rem;">
    <div style="padding: 1rem; background: white; border-radius: var(--radius-lg); box-shadow: var(--shadow-md);">
        <div style="color: var(--gray-600); font-size: 0.875rem; margin-bottom: 0.25rem;">Category</div>
        <div style="font-weight: 700;"><?php echo htmlspecialchars($hazard['category_name']); ?></div>
    </div>
    
    <div style="padding: 1rem; background: white; border-radius: var(--radius-lg); box-shadow: var(--shadow-md);">
        <div style="color: var(--gray-600); font-size: 0.875rem; margin-bottom: 0.25rem;">Risk Level</div>
        <span class="badge badge-<?php echo $hazard['residual_risk']; ?>">
            <?php echo strtoupper($hazard['residual_risk']); ?>
        </span>
    </div>
    
    <div style="padding: 1rem; background: white; border-radius: var(--radius-lg); box-shadow: var(--shadow-md);">
        <div style="color: var(--gray-600); font-size: 0.875rem; margin-bottom: 0.25rem;">Default Score</div>
        <div style="font-weight: 700; font-size: 1.5rem; color: var(--primary);">
            <?php echo $hazard['default_likelihood'] * $hazard['default_severity']; ?>
        </div>
        <div style="font-size: 0.75rem; color: var(--gray-500);">
            L:<?php echo $hazard['default_likelihood']; ?> × S:<?php echo $hazard['default_severity']; ?>
        </div>
    </div>
    
    <div style="padding: 1rem; background: white; border-radius: var(--radius-lg); box-shadow: var(--shadow-md);">
        <div style="color: var(--gray-600); font-size: 0.875rem; margin-bottom: 0.25rem;">Times Used</div>
        <div style="font-weight: 700; font-size: 1.5rem; color: var(--success);">
            <?php echo $hazard['usage_count']; ?>
        </div>
    </div>
</div>

<div style="display: grid; grid-template-columns: 1fr 400px; gap: 2rem;">
    <!-- Left Column: Details -->
    <div>
        <!-- Description -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Description</h2>
            </div>
            <div class="card-body">
                <p style="line-height: 1.6; color: var(--gray-700);">
                    <?php echo nl2br(htmlspecialchars($hazard['description'])); ?>
                </p>
                
                <?php if ($hazard['affected_persons']): ?>
                    <div style="margin-top: 1.5rem; padding: 1rem; background: rgba(239, 68, 68, 0.1); border-left: 4px solid var(--danger); border-radius: var(--radius-lg);">
                        <strong style="color: var(--danger);">⚠️ Who is at Risk:</strong><br>
                        <span style="color: var(--gray-700);"><?php echo htmlspecialchars($hazard['affected_persons']); ?></span>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Control Measures -->
        <?php if (!empty($hazard['controls'])): ?>
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">🛡️ Control Measures</h2>
                </div>
                <div class="card-body">
                    <?php 
                    $controlTypes = [
                        'elimination' => ['icon' => '🚫', 'color' => '#dc2626', 'label' => 'Elimination'],
                        'substitution' => ['icon' => '🔄', 'color' => '#ea580c', 'label' => 'Substitution'],
                        'engineering' => ['icon' => '⚙️', 'color' => '#ca8a04', 'label' => 'Engineering'],
                        'administrative' => ['icon' => '📋', 'color' => '#2563eb', 'label' => 'Administrative'],
                        'ppe' => ['icon' => '🦺', 'color' => '#7c3aed', 'label' => 'PPE']
                    ];
                    
                    foreach ($hazard['controls'] as $control):
                        $type = $controlTypes[$control['control_type']] ?? ['icon' => '•', 'color' => '#6b7280', 'label' => $control['control_type']];
                    ?>
                        <div style="margin-bottom: 1.5rem; padding: 1rem; background: var(--gray-50); border-left: 4px solid <?php echo $type['color']; ?>; border-radius: var(--radius-lg);">
                            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 0.5rem;">
                                <strong style="color: <?php echo $type['color']; ?>;">
                                    <?php echo $type['icon']; ?> <?php echo $type['label']; ?>
                                </strong>
                                <span class="badge" style="background: rgba(0,0,0,0.1);">
                                    <?php echo ucfirst($control['effectiveness']); ?>
                                </span>
                            </div>
                            <p style="margin: 0; color: var(--gray-700);">
                                <?php echo nl2br(htmlspecialchars($control['control_description'])); ?>
                            </p>
                            <?php if ($control['implementation_notes']): ?>
                                <p style="margin-top: 0.5rem; font-size: 0.875rem; color: var(--gray-600); font-style: italic;">
                                    💡 <?php echo htmlspecialchars($control['implementation_notes']); ?>
                                </p>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php elseif ($hazard['typical_controls']): ?>
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">🛡️ Typical Control Measures</h2>
                </div>
                <div class="card-body">
                    <div style="white-space: pre-wrap; line-height: 1.8; color: var(--gray-700);">
                        <?php echo htmlspecialchars($hazard['typical_controls']); ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
    
    <!-- Right Column: Photos & Actions -->
    <div>
        <!-- Quick Actions -->
        <div class="card" style="margin-bottom: 1.5rem;">
            <div class="card-body">
                <button onclick="showAddToAssessmentModal()" class="btn btn-primary" style="width: 100%; margin-bottom: 0.5rem;">
                    ➕ Add to Assessment
                </button>
                
                <?php if ($auth->hasPermission('assessments', 'update')): ?>
                    <button onclick="document.getElementById('photoUpload').click()" class="btn btn-secondary" style="width: 100%;">
                        📸 Upload Photo
                    </button>
                    <input type="file" id="photoUpload" accept="image/*" style="display: none;" onchange="uploadPhoto(this)">
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Reference Photos -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📸 Reference Photos</h2>
            </div>
            <div class="card-body">
                <div id="photosGrid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 0.5rem;">
                    <?php if (empty($hazard['photos'])): ?>
                        <p style="grid-column: 1 / -1; text-align: center; color: var(--gray-500); padding: 2rem;">
                            No photos yet.<br>
                            <small>Upload reference images to help identify this hazard.</small>
                        </p>
                    <?php else: ?>
                        <?php foreach ($hazard['photos'] as $photo): ?>
                            <div style="position: relative; aspect-ratio: 1; border-radius: var(--radius-lg); overflow: hidden;">
                                <img src="<?php echo $photo['thumbnail']; ?>" 
                                     alt="<?php echo htmlspecialchars($photo['caption'] ?? ''); ?>"
                                     style="width: 100%; height: 100%; object-fit: cover; cursor: pointer;"
                                     onclick="viewPhoto('<?php echo $photo['url']; ?>', '<?php echo htmlspecialchars($photo['caption'] ?? '', ENT_QUOTES); ?>')">
                                
                                <?php if ($photo['is_primary']): ?>
                                    <div style="position: absolute; top: 0.5rem; right: 0.5rem;">
                                        <span class="badge badge-primary">Primary</span>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($auth->hasPermission('assessments', 'update')): ?>
                                    <button onclick="deletePhoto(<?php echo $photo['id']; ?>)" 
                                            style="position: absolute; bottom: 0.5rem; right: 0.5rem; background: rgba(239, 68, 68, 0.9); color: white; border: none; border-radius: 50%; width: 32px; height: 32px; cursor: pointer; display: flex; align-items: center; justify-content: center;">
                                        ×
                                    </button>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Photo Modal -->
<div id="photoModal" class="modal-overlay" style="display: none;" onclick="closePhotoModal()">
    <div style="max-width: 90vw; max-height: 90vh; position: relative;">
        <button onclick="closePhotoModal()" style="position: absolute; top: 1rem; right: 1rem; background: white; border: none; border-radius: 50%; width: 40px; height: 40px; font-size: 1.5rem; cursor: pointer; box-shadow: var(--shadow-lg); z-index: 10;">×</button>
        <img id="photoModalImg" src="" alt="" style="max-width: 100%; max-height: 90vh; border-radius: var(--radius-xl); box-shadow: var(--shadow-2xl);">
    </div>
</div>

<!-- Add to Assessment Modal -->
<div id="addToAssessmentModal" class="modal-overlay" style="display: none;">
    <div class="modal" style="max-width: 600px;">
        <div class="modal-header">
            <h2 class="modal-title">Add Hazard to Assessment</h2>
            <button onclick="closeAddToAssessmentModal()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">&times;</button>
        </div>
        <div class="modal-body">
            <p style="margin-bottom: 1.5rem; color: var(--gray-600);">
                This will create a link to quickly add this hazard template to an assessment. Copy the link or open directly.
            </p>
            
            <div class="form-group">
                <label class="form-label">Quick Add Link</label>
                <div style="display: flex; gap: 0.5rem;">
                    <input type="text" id="quickAddLink" class="form-control" readonly 
                           value="<?php echo SITE_URL; ?>api/add-hazard-to-assessment.php?hazard=<?php echo $id; ?>&assessment=ASSESSMENT_ID">
                    <button onclick="copyLink()" class="btn btn-secondary">Copy</button>
                </div>
                <small class="form-help">Replace ASSESSMENT_ID with the actual assessment ID</small>
            </div>
            
            <div style="padding: 1rem; background: rgba(59, 130, 246, 0.1); border-radius: var(--radius-lg); margin-top: 1rem;">
                <strong>💡 How to use:</strong><br>
                <small>1. Open the assessment you want to add this hazard to<br>
                2. Look for "Add from Library" button<br>
                3. Search and select this hazard</small>
            </div>
        </div>
        <div class="modal-footer">
            <button onclick="closeAddToAssessmentModal()" class="btn btn-secondary">Close</button>
        </div>
    </div>
</div>

<script>
const hazardId = <?php echo $id; ?>;

function viewPhoto(url, caption) {
    document.getElementById('photoModalImg').src = url;
    document.getElementById('photoModal').style.display = 'flex';
    event.stopPropagation();
}

function closePhotoModal() {
    document.getElementById('photoModal').style.display = 'none';
}

function showAddToAssessmentModal() {
    document.getElementById('addToAssessmentModal').style.display = 'flex';
}

function closeAddToAssessmentModal() {
    document.getElementById('addToAssessmentModal').style.display = 'none';
}

function copyLink() {
    const input = document.getElementById('quickAddLink');
    input.select();
    document.execCommand('copy');
    alert('Link copied to clipboard!');
}

function uploadPhoto(input) {
    if (!input.files || !input.files[0]) return;
    
    const formData = new FormData();
    formData.append('photo', input.files[0]);
    formData.append('hazard_id', hazardId);
    
    fetch('api/upload-hazard-photo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('Error uploading photo: ' + data.message);
        }
    })
    .catch(error => {
        alert('Error uploading photo');
        console.error(error);
    });
}

function deletePhoto(photoId) {
    if (!confirm('Delete this photo?')) return;
    
    fetch('api/delete-hazard-photo.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ photo_id: photoId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('Error deleting photo: ' + data.message);
        }
    });
}

// Close modal on overlay click
document.getElementById('addToAssessmentModal')?.addEventListener('click', function(e) {
    if (e.target === this) closeAddToAssessmentModal();
});
</script>

<style>
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
    padding: 1rem;
}
</style>

<?php include '../includes/footer.php'; ?>
