<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/HazardLibrary.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$hazardLib = new HazardLibrary();
$pageTitle = 'Hazard Library';

// Get filter parameters
$categoryFilter = $_GET['category'] ?? null;
$searchTerm = $_GET['search'] ?? null;

// Get hazards and categories
$hazards = $hazardLib->getHazards($categoryFilter, $searchTerm, true);
$categories = $hazardLib->getCategories();
$stats = $hazardLib->getStatistics();

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">📚 Hazard Library</h1>
    <?php if ($auth->hasPermission('assessments', 'create')): ?>
        <button onclick="window.location.href='hazard-create.php'" class="btn btn-primary">+ Create Hazard Template</button>
    <?php endif; ?>
</div>

<!-- Statistics Cards -->
<div class="stats-grid" style="margin-bottom: 2rem;">
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(99, 102, 241, 0.1), rgba(139, 92, 246, 0.1));">
            📋
        </div>
        <h3>Total Hazards</h3>
        <div class="value"><?php echo $stats['total_hazards']; ?></div>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.1), rgba(5, 150, 105, 0.1));">
            🛡️
        </div>
        <h3>Control Measures</h3>
        <div class="value"><?php echo $stats['total_controls']; ?></div>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(245, 158, 11, 0.1), rgba(217, 119, 6, 0.1));">
            📸
        </div>
        <h3>Reference Photos</h3>
        <div class="value"><?php echo $stats['total_photos']; ?></div>
    </div>
</div>

<!-- Filters -->
<div class="card" style="margin-bottom: 2rem;">
    <div class="card-body">
        <form method="GET" class="filters">
            <div class="filter-item">
                <label class="form-label">Search</label>
                <input type="text" name="search" class="form-control" placeholder="Search hazards..." 
                       value="<?php echo htmlspecialchars($searchTerm ?? ''); ?>">
            </div>
            
            <div class="filter-item">
                <label class="form-label">Category</label>
                <select name="category" class="form-select">
                    <option value="">All Categories</option>
                    <?php foreach ($categories as $cat): ?>
                        <option value="<?php echo $cat['id']; ?>" <?php echo $categoryFilter == $cat['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($cat['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="filter-item" style="display: flex; align-items: flex-end; gap: 0.5rem;">
                <button type="submit" class="btn btn-primary">🔍 Filter</button>
                <a href="hazard-library.php" class="btn btn-secondary">Clear</a>
            </div>
        </form>
    </div>
</div>

<!-- Hazard Cards Grid -->
<?php if (empty($hazards)): ?>
    <div class="card">
        <div class="card-body text-center" style="padding: 3rem;">
            <div style="font-size: 4rem; margin-bottom: 1rem;">📋</div>
            <h3>No Hazards Found</h3>
            <p style="color: var(--gray-600); margin-bottom: 1.5rem;">
                <?php if ($searchTerm || $categoryFilter): ?>
                    Try adjusting your filters or search term.
                <?php else: ?>
                    Create your first hazard template to get started.
                <?php endif; ?>
            </p>
            <?php if ($auth->hasPermission('assessments', 'create')): ?>
                <a href="hazard-create.php" class="btn btn-primary">+ Create Hazard Template</a>
            <?php endif; ?>
        </div>
    </div>
<?php else: ?>
    <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(350px, 1fr)); gap: 1.5rem;">
        <?php foreach ($hazards as $hazard): ?>
            <div class="card" style="height: 100%;">
                <!-- Primary Photo -->
                <?php 
                $primaryPhoto = null;
                if (!empty($hazard['photos'])) {
                    foreach ($hazard['photos'] as $photo) {
                        if ($photo['is_primary']) {
                            $primaryPhoto = $photo;
                            break;
                        }
                    }
                    if (!$primaryPhoto) $primaryPhoto = $hazard['photos'][0];
                }
                ?>
                
                <?php if ($primaryPhoto): ?>
                    <div style="height: 200px; overflow: hidden; background: var(--gray-100);">
                        <img src="<?php echo $primaryPhoto['url']; ?>" 
                             alt="<?php echo htmlspecialchars($hazard['name']); ?>"
                             style="width: 100%; height: 100%; object-fit: cover;">
                    </div>
                <?php else: ?>
                    <div style="height: 200px; background: linear-gradient(135deg, var(--gray-100), var(--gray-200)); display: flex; align-items: center; justify-content: center; font-size: 4rem;">
                        ⚠️
                    </div>
                <?php endif; ?>
                
                <div class="card-body">
                    <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1rem;">
                        <h3 style="margin: 0; flex: 1;"><?php echo htmlspecialchars($hazard['name']); ?></h3>
                        <span class="badge badge-<?php echo $hazard['residual_risk']; ?>">
                            <?php echo strtoupper($hazard['residual_risk']); ?>
                        </span>
                    </div>
                    
                    <div style="margin-bottom: 0.5rem;">
                        <span class="badge badge-primary"><?php echo htmlspecialchars($hazard['category_name']); ?></span>
                    </div>
                    
                    <p style="color: var(--gray-600); font-size: 0.875rem; margin-bottom: 1rem; line-height: 1.5;">
                        <?php echo htmlspecialchars(substr($hazard['description'], 0, 120)); ?>
                        <?php if (strlen($hazard['description']) > 120): ?>...<?php endif; ?>
                    </p>
                    
                    <div style="display: flex; gap: 1rem; margin-bottom: 1rem; font-size: 0.875rem; color: var(--gray-600);">
                        <div>
                            <strong>L:</strong> <?php echo $hazard['default_likelihood']; ?>
                        </div>
                        <div>
                            <strong>S:</strong> <?php echo $hazard['default_severity']; ?>
                        </div>
                        <div>
                            <strong>Score:</strong> <?php echo $hazard['default_likelihood'] * $hazard['default_severity']; ?>
                        </div>
                    </div>
                    
                    <div style="display: flex; gap: 0.5rem; margin-bottom: 1rem; font-size: 0.75rem;">
                        <?php if ($hazard['control_count'] > 0): ?>
                            <span style="padding: 0.25rem 0.5rem; background: rgba(16, 185, 129, 0.1); color: #065f46; border-radius: 0.25rem;">
                                🛡️ <?php echo $hazard['control_count']; ?> Controls
                            </span>
                        <?php endif; ?>
                        
                        <?php if (!empty($hazard['photos'])): ?>
                            <span style="padding: 0.25rem 0.5rem; background: rgba(59, 130, 246, 0.1); color: #1e40af; border-radius: 0.25rem;">
                                📸 <?php echo count($hazard['photos']); ?> Photos
                            </span>
                        <?php endif; ?>
                        
                        <?php if ($hazard['usage_count'] > 0): ?>
                            <span style="padding: 0.25rem 0.5rem; background: rgba(245, 158, 11, 0.1); color: #92400e; border-radius: 0.25rem;">
                                📊 Used <?php echo $hazard['usage_count']; ?>x
                            </span>
                        <?php endif; ?>
                    </div>
                    
                    <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                        <a href="hazard-view.php?id=<?php echo $hazard['id']; ?>" class="btn btn-sm btn-primary">
                            👁️ View Details
                        </a>
                        
                        <?php if ($auth->hasPermission('assessments', 'update')): ?>
                            <a href="hazard-edit.php?id=<?php echo $hazard['id']; ?>" class="btn btn-sm btn-secondary">
                                ✏️ Edit
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
<?php endif; ?>

<style>
.hazard-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-xl);
}
</style>

<?php include '../includes/footer.php'; ?>
