<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/HazardLibrary.php';

$auth = new Auth();
$auth->requireLogin();
$auth->hasPermission('assessments', 'update');

$db = Database::getInstance();
$hazardLib = new HazardLibrary();
$pageTitle = 'Edit Hazard Template';

$id = $_GET['id'] ?? 0;
$hazard = $hazardLib->getHazard($id);

if (!$hazard) {
    setFlashMessage('Hazard not found', 'danger');
    header('Location: hazard-library.php');
    exit;
}

$categories = $hazardLib->getCategories();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $hazardLib->updateHazard($id, $_POST);
        setFlashMessage('Hazard updated successfully', 'success');
        header('Location: hazard-view.php?id=' . $id);
        exit;
    } catch (Exception $e) {
        setFlashMessage('Error: ' . $e->getMessage(), 'danger');
    }
}

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">✏️ Edit Hazard Template</h1>
    <a href="hazard-view.php?id=<?php echo $id; ?>" class="btn btn-secondary">← Back to Hazard</a>
</div>

<form method="POST" class="needs-validation">
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Basic Information</h2>
        </div>
        <div class="card-body">
            <div class="form-group">
                <label class="form-label required">Hazard Name</label>
                <input type="text" name="name" class="form-control" required 
                       value="<?php echo htmlspecialchars($hazard['name']); ?>">
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label class="form-label required">Category</label>
                    <select name="category_id" class="form-select" required>
                        <?php foreach ($categories as $cat): ?>
                            <option value="<?php echo $cat['id']; ?>" 
                                    <?php echo $cat['id'] == $hazard['category_id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cat['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Residual Risk Level</label>
                    <select name="residual_risk" class="form-select" required>
                        <option value="low" <?php echo $hazard['residual_risk'] == 'low' ? 'selected' : ''; ?>>Low</option>
                        <option value="medium" <?php echo $hazard['residual_risk'] == 'medium' ? 'selected' : ''; ?>>Medium</option>
                        <option value="high" <?php echo $hazard['residual_risk'] == 'high' ? 'selected' : ''; ?>>High</option>
                        <option value="critical" <?php echo $hazard['residual_risk'] == 'critical' ? 'selected' : ''; ?>>Critical</option>
                    </select>
                </div>
            </div>
            
            <div class="form-group">
                <label class="form-label required">Description</label>
                <textarea name="description" class="form-textarea" rows="3" required><?php echo htmlspecialchars($hazard['description']); ?></textarea>
            </div>
            
            <div class="form-group">
                <label class="form-label">Who is Affected?</label>
                <input type="text" name="affected_persons" class="form-control" 
                       value="<?php echo htmlspecialchars($hazard['affected_persons'] ?? ''); ?>">
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Risk Assessment</h2>
        </div>
        <div class="card-body">
            <div class="form-row">
                <div class="form-group">
                    <label class="form-label required">Default Likelihood (1-5)</label>
                    <select name="default_likelihood" class="form-control likelihood-input" required>
                        <?php for ($i = 1; $i <= 5; $i++): ?>
                            <option value="<?php echo $i; ?>" <?php echo $hazard['default_likelihood'] == $i ? 'selected' : ''; ?>>
                                <?php echo $i; ?> - <?php echo ['Rare', 'Unlikely', 'Possible', 'Likely', 'Almost Certain'][$i-1]; ?>
                            </option>
                        <?php endfor; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Default Severity (1-5)</label>
                    <select name="default_severity" class="form-control severity-input" required>
                        <?php for ($i = 1; $i <= 5; $i++): ?>
                            <option value="<?php echo $i; ?>" <?php echo $hazard['default_severity'] == $i ? 'selected' : ''; ?>>
                                <?php echo $i; ?> - <?php echo ['Negligible', 'Minor', 'Moderate', 'Major', 'Catastrophic'][$i-1]; ?>
                            </option>
                        <?php endfor; ?>
                    </select>
                </div>
            </div>
            
            <div class="risk-calculator">
                <div style="padding: 1rem; background: var(--gray-50); border-radius: var(--radius-lg); display: flex; gap: 2rem; align-items: center;">
                    <div>
                        <strong>Risk Score:</strong> 
                        <span class="risk-score" style="font-size: 2rem; font-weight: 800;"><?php echo $hazard['default_likelihood'] * $hazard['default_severity']; ?></span>
                    </div>
                    <div>
                        <strong>Risk Level:</strong> 
                        <span class="badge badge-<?php echo calculateRiskLevel($hazard['default_likelihood'], $hazard['default_severity']); ?> risk-level">
                            <?php echo strtoupper(calculateRiskLevel($hazard['default_likelihood'], $hazard['default_severity'])); ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Typical Control Measures</h2>
        </div>
        <div class="card-body">
            <div class="form-group">
                <label class="form-label">Typical Controls (one per line)</label>
                <textarea name="typical_controls" class="form-textarea" rows="6"><?php echo htmlspecialchars($hazard['typical_controls'] ?? ''); ?></textarea>
            </div>
        </div>
    </div>
    
    <div style="display: flex; gap: 1rem; margin-bottom: 2rem;">
        <button type="submit" class="btn btn-primary btn-lg">💾 Save Changes</button>
        <a href="hazard-view.php?id=<?php echo $id; ?>" class="btn btn-secondary btn-lg">Cancel</a>
        
        <?php if ($auth->hasPermission('assessments', 'delete')): ?>
            <button type="button" onclick="deleteHazard()" class="btn btn-danger btn-lg" style="margin-left: auto;">
                🗑️ Delete Hazard
            </button>
        <?php endif; ?>
    </div>
</form>

<script>
function deleteHazard() {
    if (!confirm('Are you sure you want to delete this hazard template? This action cannot be undone.\n\nNote: Existing assessments using this hazard will not be affected.')) {
        return;
    }
    
    fetch('api/delete-hazard.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ hazard_id: <?php echo $id; ?> })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            window.location.href = 'hazard-library.php';
        } else {
            alert('Error: ' + data.message);
        }
    });
}
</script>

<?php include '../includes/footer.php'; ?>
