<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/HazardLibrary.php';

$auth = new Auth();
$auth->requireLogin();
$auth->hasPermission('assessments', 'create');

$db = Database::getInstance();
$hazardLib = new HazardLibrary();
$pageTitle = 'Create Hazard Template';

$categories = $hazardLib->getCategories();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $hazardId = $hazardLib->createHazard($_POST);
        setFlashMessage('Hazard template created successfully', 'success');
        header('Location: hazard-view.php?id=' . $hazardId);
        exit;
    } catch (Exception $e) {
        setFlashMessage('Error: ' . $e->getMessage(), 'danger');
    }
}

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">➕ Create Hazard Template</h1>
    <a href="hazard-library.php" class="btn btn-secondary">← Back to Library</a>
</div>

<form method="POST" class="needs-validation">
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Basic Information</h2>
        </div>
        <div class="card-body">
            <div class="form-group">
                <label class="form-label required">Hazard Name</label>
                <input type="text" name="name" class="form-control" required 
                       placeholder="e.g., Reversing Collisions">
                <small class="form-help">Clear, descriptive name for the hazard</small>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label class="form-label required">Category</label>
                    <select name="category_id" class="form-select" required>
                        <option value="">Select Category</option>
                        <?php foreach ($categories as $cat): ?>
                            <option value="<?php echo $cat['id']; ?>"><?php echo htmlspecialchars($cat['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Residual Risk Level</label>
                    <select name="residual_risk" class="form-select" required>
                        <option value="low">Low</option>
                        <option value="medium" selected>Medium</option>
                        <option value="high">High</option>
                        <option value="critical">Critical</option>
                    </select>
                </div>
            </div>
            
            <div class="form-group">
                <label class="form-label required">Description</label>
                <textarea name="description" class="form-textarea" rows="3" required
                          placeholder="Detailed description of the hazard and when it typically occurs"></textarea>
            </div>
            
            <div class="form-group">
                <label class="form-label">Who is Affected?</label>
                <input type="text" name="affected_persons" class="form-control" 
                       placeholder="e.g., Drivers, Pedestrians, Site workers">
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Risk Assessment</h2>
        </div>
        <div class="card-body">
            <div class="form-row">
                <div class="form-group">
                    <label class="form-label required">Default Likelihood (1-5)</label>
                    <select name="default_likelihood" class="form-control likelihood-input" required>
                        <option value="1">1 - Rare</option>
                        <option value="2">2 - Unlikely</option>
                        <option value="3" selected>3 - Possible</option>
                        <option value="4">4 - Likely</option>
                        <option value="5">5 - Almost Certain</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Default Severity (1-5)</label>
                    <select name="default_severity" class="form-control severity-input" required>
                        <option value="1">1 - Negligible</option>
                        <option value="2">2 - Minor</option>
                        <option value="3" selected>3 - Moderate</option>
                        <option value="4">4 - Major</option>
                        <option value="5">5 - Catastrophic</option>
                    </select>
                </div>
            </div>
            
            <div class="risk-calculator">
                <div style="padding: 1rem; background: var(--gray-50); border-radius: var(--radius-lg); display: flex; gap: 2rem; align-items: center;">
                    <div>
                        <strong>Risk Score:</strong> 
                        <span class="risk-score" style="font-size: 2rem; font-weight: 800;">9</span>
                    </div>
                    <div>
                        <strong>Risk Level:</strong> 
                        <span class="badge badge-medium risk-level">MEDIUM</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Standard Control Measures</h2>
        </div>
        <div class="card-body">
            <div class="form-group">
                <label class="form-label">Typical Controls (one per line)</label>
                <textarea name="typical_controls" class="form-textarea" rows="6" 
                          placeholder="List standard control measures, one per line:
• Install reversing cameras and sensors
• Use banksman for guidance
• Designated reversing areas with clear signage
• Regular driver training"></textarea>
                <small class="form-help">These will be suggested when adding this hazard to an assessment</small>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Detailed Control Measures (Optional)</h2>
        </div>
        <div class="card-body">
            <div id="controlsContainer">
                <p style="color: var(--gray-600); margin-bottom: 1rem;">
                    Add detailed control measures using the hierarchy of controls. You can add these now or later.
                </p>
                <button type="button" onclick="addControlField()" class="btn btn-secondary btn-sm">
                    + Add Control Measure
                </button>
            </div>
        </div>
    </div>
    
    <div style="display: flex; gap: 1rem; margin-bottom: 2rem;">
        <button type="submit" class="btn btn-primary btn-lg">💾 Create Hazard Template</button>
        <a href="hazard-library.php" class="btn btn-secondary btn-lg">Cancel</a>
    </div>
</form>

<script>
let controlIndex = 0;

function addControlField() {
    const container = document.getElementById('controlsContainer');
    const controlDiv = document.createElement('div');
    controlDiv.className = 'control-item';
    controlDiv.style.cssText = 'border: 2px solid var(--gray-200); padding: 1rem; border-radius: var(--radius-lg); margin-top: 1rem;';
    
    controlDiv.innerHTML = `
        <div style="display: flex; justify-content: between; align-items: center; margin-bottom: 1rem;">
            <h4 style="margin: 0;">Control Measure ${controlIndex + 1}</h4>
            <button type="button" onclick="this.closest('.control-item').remove()" class="btn btn-sm btn-danger">Remove</button>
        </div>
        
        <div class="form-row">
            <div class="form-group">
                <label class="form-label">Control Type</label>
                <select name="controls[${controlIndex}][control_type]" class="form-select">
                    <option value="elimination">Elimination</option>
                    <option value="substitution">Substitution</option>
                    <option value="engineering">Engineering Controls</option>
                    <option value="administrative" selected>Administrative Controls</option>
                    <option value="ppe">PPE</option>
                </select>
            </div>
            
            <div class="form-group">
                <label class="form-label">Effectiveness</label>
                <select name="controls[${controlIndex}][effectiveness]" class="form-select">
                    <option value="high" selected>High</option>
                    <option value="medium">Medium</option>
                    <option value="low">Low</option>
                </select>
            </div>
        </div>
        
        <div class="form-group">
            <label class="form-label">Control Description</label>
            <textarea name="controls[${controlIndex}][control_description]" class="form-control" rows="2"></textarea>
        </div>
        
        <div class="form-group">
            <label class="form-label">Implementation Notes</label>
            <input type="text" name="controls[${controlIndex}][implementation_notes]" class="form-control" 
                   placeholder="How to implement this control">
        </div>
    `;
    
    container.appendChild(controlDiv);
    controlIndex++;
}
</script>

<?php include '../includes/footer.php'; ?>
