<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';

$auth = new Auth();
$auth->requireLogin();
$pageTitle = 'Settings Link Diagnostic';

// Get user info
$db = Database::getInstance();
$user = $db->fetchOne("SELECT * FROM users WHERE id = ?", [$_SESSION['user_id']]);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings Link Diagnostic</title>
    <style>
        body {
            font-family: system-ui, -apple-system, sans-serif;
            padding: 2rem;
            max-width: 1200px;
            margin: 0 auto;
            background: #f3f4f6;
        }
        .diagnostic-box {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        h1 {
            color: #1f2937;
            margin-bottom: 2rem;
        }
        h2 {
            color: #374151;
            font-size: 1.25rem;
            margin-bottom: 1rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid #e5e7eb;
        }
        .status {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 6px;
            font-weight: 600;
            font-size: 0.875rem;
        }
        .status-ok { background: #d1fae5; color: #065f46; }
        .status-error { background: #fee2e2; color: #991b1b; }
        .status-warning { background: #fef3c7; color: #92400e; }
        .info-grid {
            display: grid;
            grid-template-columns: 200px 1fr;
            gap: 0.75rem;
            margin-top: 1rem;
        }
        .info-label {
            font-weight: 600;
            color: #6b7280;
        }
        .info-value {
            color: #1f2937;
            font-family: 'Courier New', monospace;
            background: #f9fafb;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
        }
        .test-link {
            display: inline-block;
            padding: 0.75rem 1.5rem;
            background: #6366f1;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
            margin: 0.5rem 0.5rem 0.5rem 0;
            transition: background 0.2s;
        }
        .test-link:hover {
            background: #4f46e5;
        }
        code {
            background: #f3f4f6;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 0.875rem;
        }
        .nav-preview {
            background: #f9fafb;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
        }
        .nav-preview a {
            display: inline-block;
            padding: 0.5rem 1rem;
            margin: 0.25rem;
            background: white;
            color: #374151;
            text-decoration: none;
            border-radius: 6px;
            border: 1px solid #d1d5db;
        }
        .nav-preview a:hover {
            background: #eef2ff;
            color: #6366f1;
            border-color: #6366f1;
        }
        .back-link {
            display: inline-block;
            margin-top: 2rem;
            padding: 0.5rem 1rem;
            background: #e5e7eb;
            color: #374151;
            text-decoration: none;
            border-radius: 6px;
        }
    </style>
</head>
<body>
    <h1>🔧 Settings Link Diagnostic Tool</h1>

    <!-- User Information -->
    <div class="diagnostic-box">
        <h2>👤 User Information</h2>
        <div class="info-grid">
            <div class="info-label">Username:</div>
            <div class="info-value"><?php echo htmlspecialchars($user['username']); ?></div>
            
            <div class="info-label">Full Name:</div>
            <div class="info-value"><?php echo htmlspecialchars($user['full_name']); ?></div>
            
            <div class="info-label">Role:</div>
            <div class="info-value">
                <?php echo htmlspecialchars($user['role']); ?>
                <?php if ($user['role'] === 'admin'): ?>
                    <span class="status status-ok">ADMIN</span>
                <?php else: ?>
                    <span class="status status-error">NOT ADMIN</span>
                <?php endif; ?>
            </div>
            
            <div class="info-label">Session Role:</div>
            <div class="info-value">
                <?php echo htmlspecialchars($_SESSION['role'] ?? 'not set'); ?>
                <?php if (($_SESSION['role'] ?? '') === 'admin'): ?>
                    <span class="status status-ok">ADMIN</span>
                <?php else: ?>
                    <span class="status status-error">NOT ADMIN</span>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if ($user['role'] !== 'admin'): ?>
            <div style="margin-top: 1rem; padding: 1rem; background: #fee2e2; border-radius: 6px; color: #991b1b;">
                ⚠️ <strong>Issue Identified:</strong> Your account role is not 'admin'. Settings link will not appear in navigation.
            </div>
        <?php endif; ?>
    </div>

    <!-- Configuration Check -->
    <div class="diagnostic-box">
        <h2>⚙️ Configuration Check</h2>
        <div class="info-grid">
            <div class="info-label">BASE_URL:</div>
            <div class="info-value"><?php echo htmlspecialchars(BASE_URL); ?></div>
            
            <div class="info-label">SITE_URL:</div>
            <div class="info-value"><?php echo htmlspecialchars(SITE_URL); ?></div>
            
            <div class="info-label">Settings URL:</div>
            <div class="info-value"><?php echo htmlspecialchars(SITE_URL . 'settings.php'); ?></div>
            
            <div class="info-label">Settings File:</div>
            <div class="info-value">
                <?php if (file_exists(__DIR__ . '/settings.php')): ?>
                    <span class="status status-ok">EXISTS</span>
                <?php else: ?>
                    <span class="status status-error">MISSING</span>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Navigation Preview -->
    <div class="diagnostic-box">
        <h2>🔗 Navigation Preview (Admin Menu)</h2>
        
        <?php if (($_SESSION['role'] ?? '') === 'admin'): ?>
            <p>Based on your admin role, these links should appear in your navigation:</p>
            <div class="nav-preview">
                <a href="<?php echo SITE_URL; ?>users.php">Users</a>
                <a href="<?php echo SITE_URL; ?>settings.php">Settings</a>
                <a href="<?php echo SITE_URL; ?>audit-logs.php">Audit Logs</a>
            </div>
            
            <p style="margin-top: 1rem;">Try clicking the Settings link above. If it works here but not in the main navigation, we have a CSS or JavaScript issue.</p>
        <?php else: ?>
            <p style="color: #991b1b;">❌ Admin menu will not display because your role is not 'admin'.</p>
        <?php endif; ?>
    </div>

    <!-- Direct Links Test -->
    <div class="diagnostic-box">
        <h2>🧪 Direct Link Tests</h2>
        <p>Try accessing Settings through these direct links:</p>
        
        <a href="<?php echo SITE_URL; ?>settings.php" class="test-link">
            📝 Go to Settings (Normal Link)
        </a>
        
        <a href="<?php echo SITE_URL; ?>settings.php" class="test-link" target="_blank">
            🔗 Open Settings in New Tab
        </a>
        
        <a href="#" onclick="window.location.href='<?php echo SITE_URL; ?>settings.php'; return false;" class="test-link">
            ⚡ Go to Settings (JavaScript)
        </a>
    </div>

    <!-- Browser Information -->
    <div class="diagnostic-box">
        <h2>🌐 Browser Information</h2>
        <div class="info-grid">
            <div class="info-label">User Agent:</div>
            <div class="info-value" style="word-break: break-all;">
                <?php echo htmlspecialchars($_SERVER['HTTP_USER_AGENT'] ?? 'Not available'); ?>
            </div>
            
            <div class="info-label">Protocol:</div>
            <div class="info-value">
                <?php echo isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'HTTPS' : 'HTTP'; ?>
            </div>
            
            <div class="info-label">Server Software:</div>
            <div class="info-value">
                <?php echo htmlspecialchars($_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'); ?>
            </div>
        </div>
    </div>

    <!-- File Verification -->
    <div class="diagnostic-box">
        <h2>📁 File Verification</h2>
        <div class="info-grid">
            <div class="info-label">header.php:</div>
            <div class="info-value">
                <?php if (file_exists('../includes/header.php')): ?>
                    <span class="status status-ok">EXISTS</span>
                    (<?php echo filesize('../includes/header.php'); ?> bytes)
                <?php else: ?>
                    <span class="status status-error">MISSING</span>
                <?php endif; ?>
            </div>
            
            <div class="info-label">settings.php:</div>
            <div class="info-value">
                <?php if (file_exists(__DIR__ . '/settings.php')): ?>
                    <span class="status status-ok">EXISTS</span>
                    (<?php echo filesize(__DIR__ . '/settings.php'); ?> bytes)
                <?php else: ?>
                    <span class="status status-error">MISSING</span>
                <?php endif; ?>
            </div>
            
            <div class="info-label">style.css:</div>
            <div class="info-value">
                <?php if (file_exists(__DIR__ . '/assets/css/style.css')): ?>
                    <span class="status status-ok">EXISTS</span>
                    (<?php echo filesize(__DIR__ . '/assets/css/style.css'); ?> bytes)
                <?php else: ?>
                    <span class="status status-error">MISSING</span>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Header.php Content Check -->
    <div class="diagnostic-box">
        <h2>📄 Header.php Navigation Code</h2>
        <p>Current admin navigation code from header.php:</p>
        <pre style="background: #1f2937; color: #f3f4f6; padding: 1rem; border-radius: 8px; overflow-x: auto; font-size: 0.875rem;"><?php
        $headerContent = file_get_contents('../includes/header.php');
        // Extract the admin menu section
        preg_match('/if.*role.*admin.*?\?>(.*?)endif/s', $headerContent, $matches);
        if (isset($matches[1])) {
            echo htmlspecialchars(trim($matches[1]));
        } else {
            echo "Could not extract admin menu section";
        }
        ?></pre>
    </div>

    <!-- Recommendations -->
    <div class="diagnostic-box">
        <h2>💡 Troubleshooting Steps</h2>
        <ol style="line-height: 1.8;">
            <li><strong>Verify Role:</strong> Ensure your account has role = 'admin' in database</li>
            <li><strong>Clear Cache:</strong> Press Ctrl+Shift+R (or Cmd+Shift+R on Mac) to hard refresh</li>
            <li><strong>Check Files:</strong> Ensure you uploaded the latest header.php to server</li>
            <li><strong>Test Direct Link:</strong> Try the "Go to Settings" buttons above</li>
            <li><strong>Check Console:</strong> Open browser developer tools (F12) and check for JavaScript errors</li>
            <li><strong>Check CSS:</strong> Inspect the Settings link with developer tools to see applied styles</li>
        </ol>
    </div>

    <a href="<?php echo SITE_URL; ?>dashboard.php" class="back-link">← Back to Dashboard</a>

    <script>
        console.log('Diagnostic Page Loaded');
        console.log('User Role:', '<?php echo htmlspecialchars($_SESSION['role'] ?? 'none'); ?>');
        console.log('SITE_URL:', '<?php echo SITE_URL; ?>');
        console.log('Settings URL:', '<?php echo SITE_URL; ?>settings.php');
        
        // Test if JavaScript is blocking navigation
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM Ready - checking for click blockers');
            
            // Find all links in nav
            const navLinks = document.querySelectorAll('.nav-links a, a[href*="settings"]');
            console.log('Found', navLinks.length, 'navigation links');
            
            navLinks.forEach((link, index) => {
                const href = link.getAttribute('href');
                const text = link.textContent.trim();
                console.log(`Link ${index + 1}:`, text, '→', href);
                
                // Check computed styles
                const styles = window.getComputedStyle(link);
                if (styles.pointerEvents === 'none') {
                    console.warn('⚠️ Link has pointer-events: none!', link);
                }
                if (styles.display === 'none' || styles.visibility === 'hidden') {
                    console.warn('⚠️ Link is hidden!', link);
                }
            });
        });
    </script>
</body>
</html>
