<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$pageTitle = 'Dashboard';

// Get statistics with error handling
try {
    $stats = [
        'total_assessments' => $db->fetchOne("SELECT COUNT(*) as count FROM assessments")['count'] ?? 0,
        'pending_assessments' => $db->fetchOne("SELECT COUNT(*) as count FROM assessments WHERE status IN ('draft', 'in_progress')")['count'] ?? 0,
        'total_locations' => $db->fetchOne("SELECT COUNT(*) as count FROM locations WHERE active = 1")['count'] ?? 0,
        'high_risks' => 0
    ];
    
    // Try to get high risks count (table may not exist yet)
    try {
        $highRiskResult = $db->fetchOne("SELECT COUNT(*) as count FROM assessment_risks WHERE risk_level IN ('high', 'critical')");
        $stats['high_risks'] = $highRiskResult['count'] ?? 0;
    } catch (Exception $e) {
        // Table doesn't exist yet, use alternative
        $stats['high_risks'] = $db->fetchOne("SELECT COUNT(*) as count FROM assessments WHERE overall_risk_level IN ('high', 'critical')")['count'] ?? 0;
    }
} catch (Exception $e) {
    $stats = [
        'total_assessments' => 0,
        'pending_assessments' => 0,
        'total_locations' => 0,
        'high_risks' => 0
    ];
}

// Recent assessments
$recentAssessments = $db->fetchAll("
    SELECT a.*, l.name as location_name, u.full_name as assessor_name
    FROM assessments a
    LEFT JOIN locations l ON a.location_id = l.id
    LEFT JOIN users u ON a.assessor_id = u.id
    ORDER BY a.created_at DESC
    LIMIT 5
");

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">Dashboard</h1>
    <a href="assessment-create.php" class="btn btn-primary">+ New Assessment</a>
</div>

<div class="stats-grid">
    <div class="stat-card">
        <h3>Total Assessments</h3>
        <div class="value"><?php echo $stats['total_assessments']; ?></div>
    </div>
    <div class="stat-card">
        <h3>Pending</h3>
        <div class="value"><?php echo $stats['pending_assessments']; ?></div>
    </div>
    <div class="stat-card">
        <h3>Active Locations</h3>
        <div class="value"><?php echo $stats['total_locations']; ?></div>
    </div>
    <div class="stat-card">
        <h3>High/Critical Risks</h3>
        <div class="value" style="color: #ef4444;"><?php echo $stats['high_risks']; ?></div>
    </div>
</div>

<div class="card">
    <div class="card-header">
        <h2 class="card-title">Recent Assessments</h2>
    </div>
    <div class="card-body">
        <table>
            <thead>
                <tr>
                    <th>Assessment #</th>
                    <th>Location</th>
                    <th>Visit Date</th>
                    <th>Assessor</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($recentAssessments)): ?>
                    <tr>
                        <td colspan="6" class="table-empty">
                            No assessments yet. <a href="assessment-create.php">Create your first assessment!</a>
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($recentAssessments as $assessment): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($assessment['assessment_number']); ?></td>
                            <td><?php echo htmlspecialchars($assessment['location_name']); ?></td>
                            <td><?php echo formatDate($assessment['visit_date'], 'd M Y'); ?></td>
                            <td><?php echo htmlspecialchars($assessment['assessor_name']); ?></td>
                            <td>
                                <span class="badge badge-<?php echo $assessment['status']; ?>">
                                    <?php echo ucfirst(str_replace('_', ' ', $assessment['status'])); ?>
                                </span>
                            </td>
                            <td class="table-actions">
                                <a href="assessment-view.php?id=<?php echo $assessment['id']; ?>" class="btn btn-sm btn-secondary">View</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
