<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/AuditManager.php';

$auth = new Auth();
$auth->requireLogin();

// Check if user has permission to view audits
$canViewAudits = $auth->hasPermission('audit', 'read') || $_SESSION['role'] === 'admin';

if (!$canViewAudits) {
    setFlashMessage('You do not have permission to view audit logs', 'danger');
    header('Location: dashboard.php');
    exit;
}

$db = Database::getInstance();
$auditManager = new AuditManager();
$pageTitle = 'Audit Logs';

// Handle export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    $filters = [
        'user_id' => $_GET['user_id'] ?? null,
        'action' => $_GET['action'] ?? null,
        'entity_type' => $_GET['entity_type'] ?? null,
        'date_from' => $_GET['date_from'] ?? null,
        'date_to' => $_GET['date_to'] ?? null
    ];
    
    $csv = $auditManager->exportToCSV($filters);
    
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="audit_logs_' . date('Y-m-d') . '.csv"');
    echo $csv;
    exit;
}

// Get filters
$filters = [
    'user_id' => $_GET['user_id'] ?? null,
    'action' => $_GET['action'] ?? null,
    'entity_type' => $_GET['entity_type'] ?? null,
    'date_from' => $_GET['date_from'] ?? null,
    'date_to' => $_GET['date_to'] ?? null,
    'search' => $_GET['search'] ?? null
];

// Pagination
$page = $_GET['page'] ?? 1;
$perPage = 50;
$offset = ($page - 1) * $perPage;

// Get logs
$logs = $auditManager->getLogs($filters, $perPage, $offset);
$totalLogs = $auditManager->getLogsCount($filters);
$totalPages = ceil($totalLogs / $perPage);

// Get statistics
$stats = $auditManager->getActivityStatistics(7);

// Get unique values for filters
$users = $db->fetchAll("SELECT id, full_name, username FROM users ORDER BY full_name");
$actions = $db->fetchAll("SELECT DISTINCT action FROM audit_log ORDER BY action");
$entityTypes = $db->fetchAll("SELECT DISTINCT entity_type FROM audit_log ORDER BY entity_type");

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">📋 Audit Logs</h1>
    <div style="display: flex; gap: 0.5rem;">
        <a href="?export=csv<?php echo http_build_query($filters) ? '&' . http_build_query($filters) : ''; ?>" class="btn btn-secondary">
            📥 Export CSV
        </a>
    </div>
</div>

<!-- Statistics Cards -->
<div class="stats-grid" style="margin-bottom: 2rem;">
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(99, 102, 241, 0.1), rgba(139, 92, 246, 0.1));">
            📊
        </div>
        <h3>Total Actions (7 days)</h3>
        <div class="value"><?php echo number_format($stats['total_actions']); ?></div>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.1), rgba(5, 150, 105, 0.1));">
            👥
        </div>
        <h3>Active Users</h3>
        <div class="value"><?php echo number_format($stats['unique_users']); ?></div>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(245, 158, 11, 0.1), rgba(217, 119, 6, 0.1));">
            🔥
        </div>
        <h3>Top Action</h3>
        <div class="value" style="font-size: 1.5rem;">
            <?php 
            if (!empty($stats['top_actions'])) {
                echo htmlspecialchars($stats['top_actions'][0]['action']);
            } else {
                echo 'N/A';
            }
            ?>
        </div>
    </div>
</div>

<!-- Filters -->
<div class="card" style="margin-bottom: 2rem;">
    <div class="card-header">
        <h2 class="card-title">🔍 Filters</h2>
    </div>
    <div class="card-body">
        <form method="GET" class="filters">
            <div class="filter-item">
                <label class="form-label">Search</label>
                <input type="text" name="search" class="form-control" 
                       placeholder="Search..." value="<?php echo htmlspecialchars($filters['search'] ?? ''); ?>">
            </div>
            
            <div class="filter-item">
                <label class="form-label">User</label>
                <select name="user_id" class="form-select">
                    <option value="">All Users</option>
                    <?php foreach ($users as $user): ?>
                        <option value="<?php echo $user['id']; ?>" 
                                <?php echo ($filters['user_id'] ?? '') == $user['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($user['full_name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="filter-item">
                <label class="form-label">Action</label>
                <select name="action" class="form-select">
                    <option value="">All Actions</option>
                    <?php foreach ($actions as $action): ?>
                        <option value="<?php echo $action['action']; ?>" 
                                <?php echo ($filters['action'] ?? '') == $action['action'] ? 'selected' : ''; ?>>
                            <?php echo ucwords(str_replace('_', ' ', $action['action'])); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="filter-item">
                <label class="form-label">Entity Type</label>
                <select name="entity_type" class="form-select">
                    <option value="">All Types</option>
                    <?php foreach ($entityTypes as $type): ?>
                        <option value="<?php echo $type['entity_type']; ?>" 
                                <?php echo ($filters['entity_type'] ?? '') == $type['entity_type'] ? 'selected' : ''; ?>>
                            <?php echo ucwords(str_replace('_', ' ', $type['entity_type'])); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="filter-item">
                <label class="form-label">Date From</label>
                <input type="date" name="date_from" class="form-control" 
                       value="<?php echo htmlspecialchars($filters['date_from'] ?? ''); ?>">
            </div>
            
            <div class="filter-item">
                <label class="form-label">Date To</label>
                <input type="date" name="date_to" class="form-control" 
                       value="<?php echo htmlspecialchars($filters['date_to'] ?? ''); ?>">
            </div>
            
            <div class="filter-item" style="display: flex; align-items: flex-end; gap: 0.5rem;">
                <button type="submit" class="btn btn-primary">Apply</button>
                <a href="audit-logs.php" class="btn btn-secondary">Clear</a>
            </div>
        </form>
    </div>
</div>

<!-- Audit Logs Table -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">Audit Trail</h2>
        <span class="badge badge-primary"><?php echo number_format($totalLogs); ?> records</span>
    </div>
    <div class="card-body">
        <?php if (empty($logs)): ?>
            <p style="text-align: center; padding: 2rem; color: var(--gray-500);">
                No audit logs found matching your criteria.
            </p>
        <?php else: ?>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th style="width: 150px;">Date/Time</th>
                            <th>User</th>
                            <th style="width: 120px;">Action</th>
                            <th>Entity</th>
                            <th>Changes</th>
                            <th style="width: 120px;">IP Address</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($logs as $log): ?>
                            <tr>
                                <td>
                                    <small style="display: block;"><?php echo date('d/m/Y', strtotime($log['created_at'])); ?></small>
                                    <small style="color: var(--gray-500);"><?php echo date('H:i:s', strtotime($log['created_at'])); ?></small>
                                </td>
                                <td>
                                    <strong><?php echo htmlspecialchars($log['user_full_name'] ?? $log['username']); ?></strong>
                                </td>
                                <td>
                                    <span class="badge badge-<?php 
                                        echo match($log['action']) {
                                            'create' => 'success',
                                            'update' => 'primary',
                                            'delete' => 'danger',
                                            'login' => 'success',
                                            'logout' => 'gray',
                                            default => 'secondary'
                                        };
                                    ?>">
                                        <?php echo ucwords(str_replace('_', ' ', $log['action'])); ?>
                                    </span>
                                </td>
                                <td>
                                    <strong><?php echo ucwords(str_replace('_', ' ', $log['entity_type'])); ?></strong>
                                    <?php if ($log['entity_name']): ?>
                                        <br><small style="color: var(--gray-600);"><?php echo htmlspecialchars($log['entity_name']); ?></small>
                                    <?php elseif ($log['entity_id']): ?>
                                        <br><small style="color: var(--gray-500);">ID: <?php echo $log['entity_id']; ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($log['changes_summary']): ?>
                                        <small><?php echo htmlspecialchars($log['changes_summary']); ?></small>
                                    <?php else: ?>
                                        <small style="color: var(--gray-400);">No details</small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <small style="font-family: monospace;"><?php echo htmlspecialchars($log['ip_address']); ?></small>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <div style="display: flex; justify-content: center; gap: 0.5rem; margin-top: 1.5rem;">
                    <?php if ($page > 1): ?>
                        <a href="?page=<?php echo $page - 1; ?>&<?php echo http_build_query($filters); ?>" class="btn btn-secondary btn-sm">
                            ← Previous
                        </a>
                    <?php endif; ?>
                    
                    <span style="padding: 0.5rem 1rem; background: var(--gray-100); border-radius: var(--radius-lg);">
                        Page <?php echo $page; ?> of <?php echo $totalPages; ?>
                    </span>
                    
                    <?php if ($page < $totalPages): ?>
                        <a href="?page=<?php echo $page + 1; ?>&<?php echo http_build_query($filters); ?>" class="btn btn-secondary btn-sm">
                            Next →
                        </a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
