/**
 * Risk Assessment System - Main JavaScript
 * Version: 3.0.0
 */

// Utility functions
const utils = {
    // Show confirmation dialog
    confirm: function(message, callback) {
        if (confirm(message)) {
            callback();
        }
    },
    
    // Show alert
    alert: function(message, type = 'info') {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.textContent = message;
        alertDiv.style.position = 'fixed';
        alertDiv.style.top = '20px';
        alertDiv.style.right = '20px';
        alertDiv.style.zIndex = '9999';
        alertDiv.style.minWidth = '300px';
        
        document.body.appendChild(alertDiv);
        
        setTimeout(() => {
            alertDiv.style.opacity = '0';
            alertDiv.style.transition = 'opacity 0.3s';
            setTimeout(() => alertDiv.remove(), 300);
        }, 3000);
    },
    
    // Format date
    formatDate: function(date) {
        if (!date) return '';
        const d = new Date(date);
        return d.toLocaleDateString('en-GB');
    },
    
    // Calculate risk score
    calculateRisk: function(likelihood, severity) {
        const score = likelihood * severity;
        let level = 'low';
        
        if (score <= 4) level = 'low';
        else if (score <= 9) level = 'medium';
        else if (score <= 16) level = 'high';
        else level = 'critical';
        
        return { score, level };
    }
};

// Photo upload handler
const photoUpload = {
    init: function() {
        const uploadInputs = document.querySelectorAll('.photo-upload-input');
        uploadInputs.forEach(input => {
            input.addEventListener('change', function(e) {
                photoUpload.handleUpload(e.target);
            });
        });
        
        const deleteButtons = document.querySelectorAll('.photo-delete');
        deleteButtons.forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                photoUpload.handleDelete(this);
            });
        });
    },
    
    handleUpload: function(input) {
        const file = input.files[0];
        if (!file) return;
        
        // Validate file type
        const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        if (!allowedTypes.includes(file.type)) {
            utils.alert('Please select a valid image file (JPEG, PNG, GIF, WebP)', 'danger');
            input.value = '';
            return;
        }
        
        // Validate file size (5MB)
        if (file.size > 5242880) {
            utils.alert('File size must be less than 5MB', 'danger');
            input.value = '';
            return;
        }
        
        const formData = new FormData();
        formData.append('photo', file);
        formData.append('assessment_id', input.dataset.assessmentId);
        formData.append('item_id', input.dataset.itemId);
        
        // Show loading
        const container = input.closest('.photo-upload-container');
        const loading = document.createElement('div');
        loading.className = 'loading';
        loading.innerHTML = '<div class="spinner"></div>';
        container.appendChild(loading);
        
        fetch('api/upload-photo.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            loading.remove();
            if (data.success) {
                utils.alert('Photo uploaded successfully', 'success');
                location.reload();
            } else {
                utils.alert(data.message || 'Upload failed', 'danger');
            }
        })
        .catch(error => {
            loading.remove();
            utils.alert('Upload failed: ' + error.message, 'danger');
        });
    },
    
    handleDelete: function(button) {
        utils.confirm('Are you sure you want to delete this photo?', function() {
            const filename = button.dataset.filename;
            const itemId = button.dataset.itemId;
            
            fetch('api/delete-photo.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    filename: filename,
                    item_id: itemId
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    utils.alert('Photo deleted successfully', 'success');
                    button.closest('.photo-item').remove();
                } else {
                    utils.alert(data.message || 'Delete failed', 'danger');
                }
            })
            .catch(error => {
                utils.alert('Delete failed: ' + error.message, 'danger');
            });
        });
    }
};

// Risk calculator
const riskCalculator = {
    init: function() {
        const likelihoodInputs = document.querySelectorAll('.likelihood-input');
        const severityInputs = document.querySelectorAll('.severity-input');
        
        likelihoodInputs.forEach(input => {
            input.addEventListener('change', function() {
                riskCalculator.calculate(this);
            });
        });
        
        severityInputs.forEach(input => {
            input.addEventListener('change', function() {
                riskCalculator.calculate(this);
            });
        });
    },
    
    calculate: function(input) {
        const row = input.closest('.risk-row');
        if (!row) return;
        
        const likelihood = parseInt(row.querySelector('.likelihood-input').value) || 0;
        const severity = parseInt(row.querySelector('.severity-input').value) || 0;
        
        const result = utils.calculateRisk(likelihood, severity);
        
        const scoreDisplay = row.querySelector('.risk-score');
        const levelDisplay = row.querySelector('.risk-level');
        
        if (scoreDisplay) {
            scoreDisplay.textContent = result.score;
        }
        
        if (levelDisplay) {
            levelDisplay.textContent = result.level.toUpperCase();
            levelDisplay.className = `badge badge-${result.level}`;
        }
    }
};

// Form validation
const formValidation = {
    init: function() {
        const forms = document.querySelectorAll('.needs-validation');
        forms.forEach(form => {
            form.addEventListener('submit', function(e) {
                if (!formValidation.validate(this)) {
                    e.preventDefault();
                    e.stopPropagation();
                }
            });
        });
    },
    
    validate: function(form) {
        let isValid = true;
        const required = form.querySelectorAll('[required]');
        
        required.forEach(field => {
            if (!field.value.trim()) {
                isValid = false;
                field.classList.add('error');
                
                let error = field.nextElementSibling;
                if (!error || !error.classList.contains('form-error')) {
                    error = document.createElement('div');
                    error.className = 'form-error';
                    error.textContent = 'This field is required';
                    field.parentNode.insertBefore(error, field.nextSibling);
                }
            } else {
                field.classList.remove('error');
                const error = field.nextElementSibling;
                if (error && error.classList.contains('form-error')) {
                    error.remove();
                }
            }
        });
        
        if (!isValid) {
            utils.alert('Please fill in all required fields', 'warning');
        }
        
        return isValid;
    }
};

// Delete confirmation
function confirmDelete(entityName) {
    return confirm(`Are you sure you want to delete this ${entityName}? This action cannot be undone.`);
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    photoUpload.init();
    riskCalculator.init();
    formValidation.init();
    
    // Active nav link
    const currentPage = window.location.pathname.split('/').pop();
    const navLinks = document.querySelectorAll('.nav-links a');
    navLinks.forEach(link => {
        if (link.getAttribute('href') === currentPage) {
            link.classList.add('active');
        }
    });
    
    // Auto-hide alerts after 5 seconds
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.opacity = '0';
            alert.style.transition = 'opacity 0.3s';
            setTimeout(() => alert.remove(), 300);
        }, 5000);
    });
});
