<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$pageTitle = 'Assessments';

// Handle filters
$status = $_GET['status'] ?? '';
$location = $_GET['location'] ?? '';
$search = $_GET['search'] ?? '';

// Build query
$sql = "SELECT a.*, l.name as location_name, l.city, u.full_name as assessor_name,
        COUNT(ar.id) as risk_count
        FROM assessments a
        LEFT JOIN locations l ON a.location_id = l.id
        LEFT JOIN users u ON a.assessor_id = u.id
        LEFT JOIN assessment_risks ar ON a.id = ar.assessment_id
        WHERE 1=1";

$params = [];

if ($status) {
    $sql .= " AND a.status = ?";
    $params[] = $status;
}

if ($location) {
    $sql .= " AND a.location_id = ?";
    $params[] = $location;
}

if ($search) {
    $sql .= " AND (a.assessment_number LIKE ? OR a.title LIKE ? OR l.name LIKE ?)";
    $searchParam = "%{$search}%";
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
}

$sql .= " GROUP BY a.id ORDER BY a.created_at DESC";

$assessments = $db->fetchAll($sql, $params);

// Get locations for filter
$locations = $db->fetchAll("SELECT id, name FROM locations WHERE active = 1 ORDER BY name");

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">Risk Assessments</h1>
    <?php if ($auth->hasPermission('assessments', 'create')): ?>
        <a href="assessment-create.php" class="btn btn-primary">+ New Assessment</a>
    <?php endif; ?>
</div>

<div class="card">
    <div class="card-body">
        <form method="GET" class="filters">
            <div class="filter-item">
                <input type="text" name="search" class="form-control" 
                       placeholder="Search..." 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            <div class="filter-item">
                <select name="status" class="form-select">
                    <option value="">All Statuses</option>
                    <option value="draft" <?php echo $status === 'draft' ? 'selected' : ''; ?>>Draft</option>
                    <option value="in_progress" <?php echo $status === 'in_progress' ? 'selected' : ''; ?>>In Progress</option>
                    <option value="completed" <?php echo $status === 'completed' ? 'selected' : ''; ?>>Completed</option>
                    <option value="approved" <?php echo $status === 'approved' ? 'selected' : ''; ?>>Approved</option>
                </select>
            </div>
            <div class="filter-item">
                <select name="location" class="form-select">
                    <option value="">All Locations</option>
                    <?php foreach ($locations as $loc): ?>
                        <option value="<?php echo $loc['id']; ?>" <?php echo $location == $loc['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($loc['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="btn btn-primary">Filter</button>
            <a href="assessments.php" class="btn btn-secondary">Clear</a>
        </form>
    </div>
</div>

<div class="table-container">
    <table>
        <thead>
            <tr>
                <th>Assessment #</th>
                <th>Title</th>
                <th>Location</th>
                <th>Visit Date</th>
                <th>Assessor</th>
                <th>Status</th>
                <th>Risk Level</th>
                <th>Risks</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($assessments)): ?>
                <tr>
                    <td colspan="9" class="table-empty">
                        No assessments found. <?php if ($auth->hasPermission('assessments', 'create')): ?>
                            <a href="assessment-create.php">Create one now!</a>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php else: ?>
                <?php foreach ($assessments as $assessment): ?>
                    <tr>
                        <td><strong><?php echo htmlspecialchars($assessment['assessment_number']); ?></strong></td>
                        <td><?php echo htmlspecialchars($assessment['title']); ?></td>
                        <td>
                            <?php echo htmlspecialchars($assessment['location_name']); ?><br>
                            <small class="text-muted"><?php echo htmlspecialchars($assessment['city']); ?></small>
                        </td>
                        <td><?php echo formatDate($assessment['visit_date'], 'd M Y'); ?></td>
                        <td><?php echo htmlspecialchars($assessment['assessor_name']); ?></td>
                        <td>
                            <span class="badge badge-<?php echo $assessment['status']; ?>">
                                <?php echo ucfirst(str_replace('_', ' ', $assessment['status'])); ?>
                            </span>
                        </td>
                        <td>
                            <?php if ($assessment['overall_risk_level']): ?>
                                <span class="badge badge-<?php echo $assessment['overall_risk_level']; ?>">
                                    <?php echo ucfirst($assessment['overall_risk_level']); ?>
                                </span>
                            <?php else: ?>
                                <span class="text-muted">-</span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo $assessment['risk_count']; ?></td>
                        <td class="table-actions">
                            <a href="assessment-view.php?id=<?php echo $assessment['id']; ?>" 
                               class="btn btn-sm btn-secondary">View</a>
                            <?php if ($auth->hasPermission('assessments', 'update') && !$assessment['is_locked']): ?>
                                <a href="assessment-edit.php?id=<?php echo $assessment['id']; ?>" 
                                   class="btn btn-sm btn-primary">Edit</a>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<?php include '../includes/footer.php'; ?>
