<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/VersionControl.php';

$auth = new Auth();
$auth->requireLogin();
$versionControl = new VersionControl();

$db = Database::getInstance();
$pageTitle = 'View Assessment';

$id = $_GET['id'] ?? 0;

// Get assessment with all details
$sql = "SELECT a.*, l.*, 
        a.id as assessment_id, a.created_at as assessment_created_at,
        l.name as location_name,
        u.full_name as assessor_name,
        u.email as assessor_email
        FROM assessments a
        LEFT JOIN locations l ON a.location_id = l.id
        LEFT JOIN users u ON a.assessor_id = u.id
        WHERE a.id = ?";

$assessment = $db->fetchOne($sql, [$id]);

if (!$assessment) {
    setFlashMessage('Assessment not found', 'danger');
    header('Location: assessments.php');
    exit;
}

// Get risks
$risks = $db->fetchAll("
    SELECT ar.*, rc.name as category_name
    FROM assessment_risks ar
    LEFT JOIN risk_items ri ON ar.risk_item_id = ri.id
    LEFT JOIN risk_categories rc ON ri.category_id = rc.id
    WHERE ar.assessment_id = ?
    ORDER BY ar.risk_score DESC, ar.id
", [$id]);

// Get action items
$actions = $db->fetchAll("
    SELECT ai.*, u.full_name as assigned_to_name
    FROM action_items ai
    LEFT JOIN users u ON ai.assigned_to = u.id
    WHERE ai.assessment_id = ?
    ORDER BY ai.due_date, ai.priority DESC
", [$id]);

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">Assessment Details</h1>
    <div class="flex gap-2">
        <a href="assessments.php" class="btn btn-secondary">← Back</a>
<?php
require_once '../includes/PDFTemplateManager.php';
$pdfManager = new PDFTemplateManager();
$pdfTemplates = $pdfManager->getTemplates(true);
?>
        <?php if ($auth->hasPermission('assessments', 'update') && !$assessment['is_locked']): ?>
            <a href="assessment-edit.php?id=<?php echo $id; ?>" class="btn btn-primary">Edit</a>
        <?php endif; ?>
        <button onclick="window.print()" class="btn btn-secondary">Print / PDF</button>
    </div>
</div>

<div class="card">
    <div class="card-header">
        <div class="flex justify-between align-center">
            <div>
                <h2 class="card-title"><?php echo htmlspecialchars($assessment['title']); ?></h2>
                <p class="text-muted">Assessment #<?php echo htmlspecialchars($assessment['assessment_number']); ?></p>
            </div>
            <div>
                <span class="badge badge-<?php echo $assessment['status']; ?>" style="font-size: 1rem;">
                    <?php echo ucfirst(str_replace('_', ' ', $assessment['status'])); ?>
                </span>
                <?php if ($assessment['overall_risk_level']): ?>
                    <span class="badge badge-<?php echo $assessment['overall_risk_level']; ?>" style="font-size: 1rem;">
                        <?php echo ucfirst($assessment['overall_risk_level']); ?> Risk
                    </span>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <div class="card-body">
        <div class="form-row">
            <div>
                <strong>Location:</strong><br>
                <?php echo htmlspecialchars($assessment['location_name']); ?><br>
                <small class="text-muted">
                    <?php echo htmlspecialchars($assessment['address']); ?>, 
                    <?php echo htmlspecialchars($assessment['city']); ?>, 
                    <?php echo htmlspecialchars($assessment['postcode']); ?>
                </small>
            </div>
            <div>
                <strong>Visit Date/Time:</strong><br>
                <?php echo formatDate($assessment['visit_date'], 'd M Y'); ?>
                <?php if ($assessment['visit_time']): ?>
                    at <?php echo date('H:i', strtotime($assessment['visit_time'])); ?>
                <?php endif; ?>
            </div>
            <div>
                <strong>Assessor:</strong><br>
                <?php echo htmlspecialchars($assessment['assessor_name']); ?>
            </div>
        </div>
        
        <?php if ($assessment['delivery_type'] || $assessment['vehicle_type'] || $assessment['weather_conditions']): ?>
            <hr style="margin: 1.5rem 0;">
            <div class="form-row">
                <?php if ($assessment['delivery_type']): ?>
                    <div>
                        <strong>Delivery Type:</strong><br>
                        <?php echo htmlspecialchars($assessment['delivery_type']); ?>
                    </div>
                <?php endif; ?>
                <?php if ($assessment['vehicle_type']): ?>
                    <div>
                        <strong>Vehicle Type:</strong><br>
                        <?php echo htmlspecialchars($assessment['vehicle_type']); ?>
                    </div>
                <?php endif; ?>
                <?php if ($assessment['weather_conditions']): ?>
                    <div>
                        <strong>Weather:</strong><br>
                        <?php echo htmlspecialchars($assessment['weather_conditions']); ?>
                    </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($assessment['description']): ?>
            <hr style="margin: 1.5rem 0;">
            <div>
                <strong>Description:</strong><br>
                <?php echo nl2br(htmlspecialchars($assessment['description'])); ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<div class="card">
    <div class="card-header">
        <h2 class="card-title">Risk Assessment (<?php echo count($risks); ?> risks identified)</h2>
    </div>
    <div class="card-body">
        <?php if (empty($risks)): ?>
            <p class="text-muted text-center" style="padding: 2rem;">
                No risks have been added to this assessment yet.
<?php
require_once '../includes/PDFTemplateManager.php';
$pdfManager = new PDFTemplateManager();
$pdfTemplates = $pdfManager->getTemplates(true);
?>
                <?php if ($auth->hasPermission('assessments', 'update')): ?>
                    <a href="assessment-edit.php?id=<?php echo $id; ?>">Add risks now</a>
                <?php endif; ?>
            </p>
        <?php else: ?>
            <?php 
            require_once '../includes/PhotoUpload.php';
            $photoUpload = new PhotoUpload();
            ?>
            
            <div class="risks-grid" style="display: grid; gap: 20px;">
                <?php foreach ($risks as $risk): ?>
                    <div class="risk-card" style="border: 1px solid #e5e7eb; border-radius: 8px; padding: 20px; background: white;">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div style="flex: 1;">
                                <h3 style="margin: 0 0 8px 0; font-size: 16px; font-weight: 600;">
                                    <?php 
                                    $hazardText = htmlspecialchars($risk['hazard_description']);
                                    // Show first 100 chars as title
                                    echo strlen($hazardText) > 100 ? substr($hazardText, 0, 100) . '...' : $hazardText; 
                                    ?>
                                </h3>
                                <?php if ($risk['persons_at_risk']): ?>
                                    <p style="margin: 0; font-size: 13px; color: #6b7280;">
                                        <strong>Persons at Risk:</strong> <?php echo htmlspecialchars($risk['persons_at_risk']); ?>
                                    </p>
                                <?php endif; ?>
                            </div>
                            <div style="display: flex; gap: 8px; align-items: center; margin-left: 15px;">
                                <span class="badge badge-<?php echo $risk['risk_level']; ?>" style="white-space: nowrap;">
                                    <?php echo ucfirst($risk['risk_level']); ?>
                                </span>
                                <span style="background: #f3f4f6; padding: 4px 12px; border-radius: 6px; font-weight: 600; white-space: nowrap;">
                                    Score: <?php echo $risk['risk_score']; ?>
                                </span>
                            </div>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 15px; margin-bottom: 15px; padding: 12px; background: #f9fafb; border-radius: 6px;">
                            <div>
                                <div style="font-size: 12px; color: #6b7280; margin-bottom: 4px;">Likelihood</div>
                                <div style="font-size: 20px; font-weight: 700; color: #1f2937;"><?php echo $risk['likelihood']; ?><span style="font-size: 14px; color: #9ca3af;">/5</span></div>
                            </div>
                            <div>
                                <div style="font-size: 12px; color: #6b7280; margin-bottom: 4px;">Severity</div>
                                <div style="font-size: 20px; font-weight: 700; color: #1f2937;"><?php echo $risk['severity']; ?><span style="font-size: 14px; color: #9ca3af;">/5</span></div>
                            </div>
                        </div>
                        
                        <?php if ($risk['existing_controls']): ?>
                            <div style="margin-bottom: 15px;">
                                <strong style="font-size: 13px; color: #374151;">Existing Controls:</strong>
                                <p style="margin: 5px 0 0 0; font-size: 14px; color: #4b5563;">
                                    <?php echo nl2br(htmlspecialchars($risk['existing_controls'])); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($risk['additional_controls']): ?>
                            <div style="margin-bottom: 15px;">
                                <strong style="font-size: 13px; color: #374151;">Additional Controls:</strong>
                                <p style="margin: 5px 0 0 0; font-size: 14px; color: #4b5563;">
                                    <?php echo nl2br(htmlspecialchars($risk['additional_controls'])); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                        
                        <?php 
                        // Show photo upload component
                        $itemId = $risk['id'];
                        $assessmentId = $id;
                        $photos = $photoUpload->getPhotos($itemId);
                        include '../includes/partials/photo-upload.php';
                        ?>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php if (!empty($actions)): ?>
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Action Items (<?php echo count($actions); ?>)</h2>
        </div>
        <div class="card-body">
            <table>
                <thead>
                    <tr>
                        <th>Action</th>
                        <th>Assigned To</th>
                        <th>Due Date</th>
                        <th>Priority</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($actions as $action): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($action['action_description']); ?></td>
                            <td><?php echo htmlspecialchars($action['assigned_to_name'] ?? 'Unassigned'); ?></td>
                            <td><?php echo $action['due_date'] ? formatDate($action['due_date'], 'd M Y') : '-'; ?></td>
                            <td>
                                <span class="badge badge-<?php echo $action['priority']; ?>">
                                    <?php echo ucfirst($action['priority']); ?>
                                </span>
                            </td>
                            <td>
                                <span class="badge badge-<?php echo $action['status']; ?>">
                                    <?php echo ucfirst(str_replace('_', ' ', $action['status'])); ?>
                                </span>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
<?php endif; ?>

<?php include '../includes/footer.php'; ?>

<script>
// Load PDF templates into select
const templates = <?php echo json_encode($pdfTemplates); ?>;
const select = document.getElementById('pdfTemplateSelect');

if (select && templates) {
    templates.forEach(template => {
        const option = document.createElement('option');
        option.value = template.id;
        option.textContent = template.template_name;
        if (template.is_default == 1) {
            option.textContent += ' (Default)';
        }
        select.appendChild(option);
    });
}

function exportPDF() {
    const templateId = document.getElementById('pdfTemplateSelect')?.value || '';
    const assessmentId = <?php echo $id; ?>;
    let url = 'assessment-export-pdf.php?id=' + assessmentId;
    if (templateId) {
        url += '&template=' + templateId;
    }
    window.open(url, '_blank');
}

// Initialize photo upload functionality
document.addEventListener('DOMContentLoaded', function() {
    if (typeof photoUpload !== 'undefined') {
        photoUpload.init();
    }
});
</script>
<!-- Include hazard library modal -->
<?php
$assessmentId = $id;
include '../includes/partials/hazard-library-modal.php';
?>
