<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';
require_once '../includes/VersionControl.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$versionControl = new VersionControl();
$pageTitle = 'Version History';

$assessmentId = $_GET['id'] ?? 0;

// Get assessment
$assessment = $db->fetchOne("SELECT * FROM assessments WHERE id = ?", [$assessmentId]);

if (!$assessment) {
    setFlashMessage('Assessment not found', 'danger');
    header('Location: assessments.php');
    exit;
}

// Check permission
if (!$auth->hasPermission('assessments', 'read')) {
    setFlashMessage('No permission to view versions', 'danger');
    header('Location: dashboard.php');
    exit;
}

// Handle create version
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_version'])) {
    if ($auth->hasPermission('assessments', 'update')) {
        try {
            $versionControl->createVersion(
                $assessmentId,
                'manual',
                $_POST['version_notes'] ?? null
            );
            setFlashMessage('Version created successfully', 'success');
            header('Location: assessment-versions.php?id=' . $assessmentId);
            exit;
        } catch (Exception $e) {
            setFlashMessage('Error: ' . $e->getMessage(), 'danger');
        }
    }
}

// Handle restore
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['restore_version'])) {
    if ($auth->hasPermission('assessments', 'update')) {
        try {
            $versionControl->restoreVersion($_POST['version_id']);
            setFlashMessage('Version restored successfully', 'success');
            header('Location: assessment-view.php?id=' . $assessmentId);
            exit;
        } catch (Exception $e) {
            setFlashMessage('Error: ' . $e->getMessage(), 'danger');
        }
    }
}

// Get versions
$versions = $versionControl->getVersions($assessmentId);
$stats = $versionControl->getVersionStatistics($assessmentId);

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">📜 Version History</h1>
    <div style="display: flex; gap: 0.5rem;">
        <a href="assessment-view.php?id=<?php echo $assessmentId; ?>" class="btn btn-secondary">← Back to Assessment</a>
        <?php if ($auth->hasPermission('assessments', 'update')): ?>
            <button onclick="showCreateVersionModal()" class="btn btn-primary">💾 Create Version</button>
        <?php endif; ?>
    </div>
</div>

<!-- Assessment Info -->
<div class="card" style="margin-bottom: 2rem;">
    <div class="card-body">
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem;">
            <div>
                <strong>Assessment:</strong><br>
                <?php echo htmlspecialchars($assessment['assessment_number']); ?>
            </div>
            <div>
                <strong>Title:</strong><br>
                <?php echo htmlspecialchars($assessment['title']); ?>
            </div>
            <div>
                <strong>Status:</strong><br>
                <span class="badge badge-<?php echo $assessment['status']; ?>">
                    <?php echo strtoupper($assessment['status']); ?>
                </span>
            </div>
            <div>
                <strong>Total Versions:</strong><br>
                <?php echo $stats['total_versions']; ?>
            </div>
        </div>
    </div>
</div>

<!-- Version Statistics -->
<div class="stats-grid" style="margin-bottom: 2rem;">
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(99, 102, 241, 0.1), rgba(139, 92, 246, 0.1));">
            📊
        </div>
        <h3>Total Versions</h3>
        <div class="value"><?php echo $stats['total_versions']; ?></div>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.1), rgba(5, 150, 105, 0.1));">
            ✋
        </div>
        <h3>Manual Saves</h3>
        <div class="value"><?php echo $stats['manual_versions']; ?></div>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-icon" style="background: linear-gradient(135deg, rgba(245, 158, 11, 0.1), rgba(217, 119, 6, 0.1));">
            🤖
        </div>
        <h3>Auto Saves</h3>
        <div class="value"><?php echo $stats['auto_versions']; ?></div>
    </div>
</div>

<!-- Version Timeline -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">Version Timeline</h2>
    </div>
    <div class="card-body">
        <?php if (empty($versions)): ?>
            <p style="text-align: center; padding: 2rem; color: var(--gray-500);">
                No versions yet. Create your first version to track changes.
            </p>
        <?php else: ?>
            <div class="timeline">
                <?php foreach ($versions as $version): ?>
                    <div class="timeline-item" style="padding: 1.5rem; margin-bottom: 1rem; background: var(--gray-50); border-left: 4px solid var(--primary); border-radius: var(--radius-lg);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1rem;">
                            <div>
                                <h3 style="margin: 0; font-size: 1.25rem;">
                                    Version <?php echo $version['version_number']; ?>
                                    <span class="badge badge-<?php echo $version['version_type'] === 'manual' ? 'success' : 'secondary'; ?>">
                                        <?php echo strtoupper($version['version_type']); ?>
                                    </span>
                                </h3>
                                <small style="color: var(--gray-600);">
                                    <?php echo date('d/m/Y H:i:s', strtotime($version['created_at'])); ?>
                                </small>
                            </div>
                            <div style="display: flex; gap: 0.5rem;">
                                <a href="version-compare.php?id1=<?php echo $version['id']; ?>&id2=<?php echo $versions[0]['id'] ?? $version['id']; ?>" 
                                   class="btn btn-sm btn-secondary">
                                    🔍 Compare
                                </a>
                                <?php if ($auth->hasPermission('assessments', 'update')): ?>
                                    <button onclick="confirmRestore(<?php echo $version['id']; ?>, <?php echo $version['version_number']; ?>)" 
                                            class="btn btn-sm btn-primary">
                                        ↩️ Restore
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 1rem; margin-bottom: 1rem;">
                            <div>
                                <strong>Created By:</strong><br>
                                <?php echo htmlspecialchars($version['created_by_name'] ?? 'System'); ?>
                            </div>
                            <div>
                                <strong>Status:</strong><br>
                                <span class="badge badge-<?php echo $version['status']; ?>">
                                    <?php echo strtoupper($version['status']); ?>
                                </span>
                            </div>
                        </div>
                        
                        <?php if ($version['version_notes']): ?>
                            <div style="padding: 0.75rem; background: rgba(255, 255, 255, 0.5); border-radius: var(--radius-md); margin-top: 0.5rem;">
                                <strong>Notes:</strong><br>
                                <?php echo nl2br(htmlspecialchars($version['version_notes'])); ?>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Create Version Modal -->
<div id="createVersionModal" class="modal-overlay" style="display: none;">
    <div class="modal" style="max-width: 600px;">
        <div class="modal-header">
            <h2 class="modal-title">💾 Create New Version</h2>
            <button onclick="closeCreateVersionModal()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">&times;</button>
        </div>
        <form method="POST">
            <div class="modal-body">
                <div class="form-group">
                    <label class="form-label">Version Notes (Optional)</label>
                    <textarea name="version_notes" class="form-textarea" rows="4" 
                              placeholder="Describe what changed in this version..."></textarea>
                    <small class="form-help">Add a description to help identify this version later</small>
                </div>
                
                <div style="padding: 1rem; background: rgba(59, 130, 246, 0.1); border-radius: var(--radius-lg); margin-top: 1rem;">
                    <strong>💡 What gets saved:</strong><br>
                    <ul style="margin: 0.5rem 0 0 1.5rem; font-size: 0.875rem;">
                        <li>Assessment details and status</li>
                        <li>All risk items and ratings</li>
                        <li>Action items and due dates</li>
                        <li>Current timestamp</li>
                    </ul>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" onclick="closeCreateVersionModal()" class="btn btn-secondary">Cancel</button>
                <button type="submit" name="create_version" class="btn btn-primary">Create Version</button>
            </div>
        </form>
    </div>
</div>

<!-- Restore Confirmation Form -->
<form id="restoreForm" method="POST" style="display: none;">
    <input type="hidden" name="restore_version" value="1">
    <input type="hidden" name="version_id" id="restoreVersionId">
</form>

<script>
function showCreateVersionModal() {
    document.getElementById('createVersionModal').style.display = 'flex';
}

function closeCreateVersionModal() {
    document.getElementById('createVersionModal').style.display = 'none';
}

function confirmRestore(versionId, versionNumber) {
    if (confirm(`Are you sure you want to restore Version ${versionNumber}?\n\nThis will:\n• Create a backup of the current state\n• Restore all data from Version ${versionNumber}\n• This action cannot be easily undone`)) {
        document.getElementById('restoreVersionId').value = versionId;
        document.getElementById('restoreForm').submit();
    }
}

// Close modal on overlay click
document.getElementById('createVersionModal')?.addEventListener('click', function(e) {
    if (e.target === this) closeCreateVersionModal();
});
</script>

<style>
.timeline-item {
    position: relative;
    transition: all 0.2s;
}

.timeline-item:hover {
    transform: translateX(4px);
    box-shadow: var(--shadow-lg);
}

.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
    padding: 1rem;
}

.modal {
    background: white;
    border-radius: var(--radius-xl);
    box-shadow: var(--shadow-2xl);
    width: 100%;
}

.modal-header {
    padding: 1.5rem 2rem;
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-body {
    padding: 2rem;
}

.modal-footer {
    padding: 1.5rem 2rem;
    border-top: 1px solid var(--gray-200);
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
}
</style>

<?php include '../includes/footer.php'; ?>
