<?php
require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Auth.php';
require_once '../includes/functions.php';

$auth = new Auth();
$auth->requireLogin();

if (!$auth->hasPermission('assessments', 'create')) {
    setFlashMessage('You do not have permission to create assessments', 'danger');
    header('Location: assessments.php');
    exit;
}

$db = Database::getInstance();
$pageTitle = 'Create Assessment';

// Get locations
$locations = $db->fetchAll("SELECT * FROM locations WHERE active = 1 ORDER BY name");

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $locationId = $_POST['location_id'] ?? null;
    $visitDate = $_POST['visit_date'] ?? date('Y-m-d');
    $visitTime = $_POST['visit_time'] ?? null;
    $title = $_POST['title'] ?? '';
    $description = $_POST['description'] ?? '';
    $deliveryType = $_POST['delivery_type'] ?? '';
    $vehicleType = $_POST['vehicle_type'] ?? '';
    $weatherConditions = $_POST['weather_conditions'] ?? '';
    
    $errors = [];
    
    if (!$locationId) $errors[] = 'Location is required';
    if (!$visitDate) $errors[] = 'Visit date is required';
    if (!$title) $errors[] = 'Title is required';
    
    if (empty($errors)) {
        try {
            $assessmentNumber = generateAssessmentNumber();
            
            $sql = "INSERT INTO assessments (
                assessment_number, title, location_id, visit_date, visit_time,
                assessor_id, status, delivery_type, vehicle_type, weather_conditions,
                description, created_by, updated_by
            ) VALUES (?, ?, ?, ?, ?, ?, 'draft', ?, ?, ?, ?, ?, ?)";
            
            $db->query($sql, [
                $assessmentNumber,
                $title,
                $locationId,
                $visitDate,
                $visitTime,
                $_SESSION['user_id'],
                $deliveryType,
                $vehicleType,
                $weatherConditions,
                $description,
                $_SESSION['user_id'],  // created_by
                $_SESSION['user_id']   // updated_by
            ]);
            
            $assessmentId = $db->lastInsertId();
            
            logAudit($_SESSION['user_id'], 'create', 'assessment', $assessmentId);
            
            setFlashMessage('Assessment created successfully', 'success');
            header('Location: assessment-view.php?id=' . $assessmentId);
            exit;
        } catch (Exception $e) {
            $errors[] = 'Failed to create assessment: ' . $e->getMessage();
        }
    }
}

include '../includes/header.php';
?>

<div class="page-header">
    <h1 class="page-title">Create New Assessment</h1>
    <a href="assessments.php" class="btn btn-secondary">← Back to Assessments</a>
</div>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger">
        <strong>Error:</strong>
        <ul style="margin: 0.5rem 0 0 1.5rem;">
            <?php foreach ($errors as $error): ?>
                <li><?php echo htmlspecialchars($error); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<form method="POST" class="needs-validation">
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Assessment Details</h2>
        </div>
        <div class="card-body">
            <div class="form-row">
                <div class="form-group">
                    <label for="location_id" class="form-label required">Location</label>
                    <select id="location_id" name="location_id" class="form-select" required>
                        <option value="">Select location...</option>
                        <?php foreach ($locations as $location): ?>
                            <option value="<?php echo $location['id']; ?>" 
                                    <?php echo (isset($_POST['location_id']) && $_POST['location_id'] == $location['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($location['name'] . ' - ' . $location['city']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="visit_date" class="form-label required">Visit Date</label>
                    <input type="date" id="visit_date" name="visit_date" class="form-control" 
                           value="<?php echo $_POST['visit_date'] ?? date('Y-m-d'); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="visit_time" class="form-label">Visit Time</label>
                    <input type="time" id="visit_time" name="visit_time" class="form-control" 
                           value="<?php echo $_POST['visit_time'] ?? ''; ?>">
                </div>
            </div>
            
            <div class="form-group">
                <label for="title" class="form-label required">Assessment Title</label>
                <input type="text" id="title" name="title" class="form-control" 
                       placeholder="e.g., Morning Delivery Assessment - Tesco Manchester"
                       value="<?php echo htmlspecialchars($_POST['title'] ?? ''); ?>" required>
            </div>
            
            <div class="form-group">
                <label for="description" class="form-label">Description</label>
                <textarea id="description" name="description" class="form-textarea" rows="4"
                          placeholder="Provide details about the assessment..."><?php echo htmlspecialchars($_POST['description'] ?? ''); ?></textarea>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label for="delivery_type" class="form-label">Delivery Type</label>
                    <select id="delivery_type" name="delivery_type" class="form-select">
                        <option value="">Select...</option>
                        <option value="Standard delivery">Standard delivery</option>
                        <option value="Ambient">Ambient</option>
                        <option value="Chilled">Chilled</option>
                        <option value="Frozen">Frozen</option>
                        <option value="Multi-drop">Multi-drop</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="vehicle_type" class="form-label">Vehicle Type</label>
                    <select id="vehicle_type" name="vehicle_type" class="form-select">
                        <option value="">Select...</option>
                        <option value="Articulated HGV">Articulated HGV</option>
                        <option value="Rigid HGV">Rigid HGV</option>
                        <option value="7.5T Truck">7.5T Truck</option>
                        <option value="Box Van">Box Van</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="weather_conditions" class="form-label">Weather Conditions</label>
                    <select id="weather_conditions" name="weather_conditions" class="form-select">
                        <option value="">Select...</option>
                        <option value="Clear">Clear</option>
                        <option value="Rain">Rain</option>
                        <option value="Heavy Rain">Heavy Rain</option>
                        <option value="Snow">Snow</option>
                        <option value="Ice">Ice</option>
                        <option value="Fog">Fog</option>
                        <option value="Wind">Wind</option>
                    </select>
                </div>
            </div>
        </div>
        
        <div class="card-footer">
            <button type="submit" class="btn btn-primary">Create Assessment</button>
            <a href="assessments.php" class="btn btn-secondary">Cancel</a>
        </div>
    </div>
</form>

<?php include '../includes/footer.php'; ?>
