<?php
/**
 * Automatic Audit Logging Integration
 * Include this file in CRUD operations to automatically log changes
 */

require_once __DIR__ . '/../../includes/AuditManager.php';

class AutoAudit {
    private static $auditManager = null;
    
    private static function getAuditManager() {
        if (self::$auditManager === null) {
            self::$auditManager = new AuditManager();
        }
        return self::$auditManager;
    }
    
    /**
     * Log assessment action
     */
    public static function logAssessment($action, $assessmentId, $oldData = null, $newData = null) {
        $db = Database::getInstance();
        $assessment = $db->fetchOne("SELECT assessment_number FROM assessments WHERE id = ?", [$assessmentId]);
        $entityName = $assessment['assessment_number'] ?? "Assessment #{$assessmentId}";
        
        self::getAuditManager()->log($action, 'assessment', $assessmentId, $oldData, $newData, $entityName);
    }
    
    /**
     * Log location action
     */
    public static function logLocation($action, $locationId, $oldData = null, $newData = null) {
        $db = Database::getInstance();
        $location = $db->fetchOne("SELECT name FROM locations WHERE id = ?", [$locationId]);
        $entityName = $location['name'] ?? "Location #{$locationId}";
        
        self::getAuditManager()->log($action, 'location', $locationId, $oldData, $newData, $entityName);
    }
    
    /**
     * Log hazard action
     */
    public static function logHazard($action, $hazardId, $oldData = null, $newData = null) {
        $db = Database::getInstance();
        $hazard = $db->fetchOne("SELECT name FROM risk_items WHERE id = ?", [$hazardId]);
        $entityName = $hazard['name'] ?? "Hazard #{$hazardId}";
        
        self::getAuditManager()->log($action, 'hazard', $hazardId, $oldData, $newData, $entityName);
    }
    
    /**
     * Log user action
     */
    public static function logUser($action, $userId, $oldData = null, $newData = null) {
        $db = Database::getInstance();
        $user = $db->fetchOne("SELECT username FROM users WHERE id = ?", [$userId]);
        $entityName = $user['username'] ?? "User #{$userId}";
        
        self::getAuditManager()->log($action, 'user', $userId, $oldData, $newData, $entityName);
    }
    
    /**
     * Log login attempt
     */
    public static function logLogin($username, $success = true) {
        $action = $success ? 'login_success' : 'login_failed';
        self::getAuditManager()->log($action, 'auth', null, null, ['username' => $username], $username);
    }
    
    /**
     * Log logout
     */
    public static function logLogout() {
        $username = $_SESSION['username'] ?? 'Unknown';
        self::getAuditManager()->log('logout', 'auth', null, null, null, $username);
    }
    
    /**
     * Log export action
     */
    public static function logExport($entityType, $entityId, $format = 'pdf') {
        $db = Database::getInstance();
        
        switch ($entityType) {
            case 'assessment':
                $entity = $db->fetchOne("SELECT assessment_number as name FROM assessments WHERE id = ?", [$entityId]);
                break;
            default:
                $entity = ['name' => "{$entityType} #{$entityId}"];
        }
        
        $entityName = $entity['name'] ?? "{$entityType} #{$entityId}";
        
        self::getAuditManager()->log(
            'export',
            $entityType,
            $entityId,
            null,
            ['format' => $format],
            $entityName
        );
    }
}
