<!-- Hazard Library Modal -->
<div id="hazardLibraryModal" class="modal-overlay" style="display: none;">
    <div class="modal" style="max-width: 900px; max-height: 90vh; overflow-y: auto;">
        <div class="modal-header">
            <h2 class="modal-title">📚 Add from Hazard Library</h2>
            <button onclick="closeHazardLibraryModal()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">&times;</button>
        </div>
        
        <div class="modal-body">
            <!-- Search and Filter -->
            <div style="margin-bottom: 1.5rem;">
                <div style="display: flex; gap: 1rem; margin-bottom: 1rem;">
                    <input type="text" id="hazardSearch" class="form-control" placeholder="Search hazards..." 
                           onkeyup="searchHazards()">
                    
                    <select id="hazardCategoryFilter" class="form-select" onchange="searchHazards()">
                        <option value="">All Categories</option>
                    </select>
                </div>
            </div>
            
            <!-- Hazards Grid -->
            <div id="hazardsGrid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(280px, 1fr)); gap: 1rem; max-height: 500px; overflow-y: auto;">
                <div style="grid-column: 1 / -1; text-align: center; padding: 2rem; color: var(--gray-500);">
                    Loading hazards...
                </div>
            </div>
        </div>
        
        <div class="modal-footer">
            <button onclick="closeHazardLibraryModal()" class="btn btn-secondary">Close</button>
        </div>
    </div>
</div>

<script>
let allHazards = [];
let hazardCategories = [];

function openHazardLibraryModal() {
    document.getElementById('hazardLibraryModal').style.display = 'flex';
    loadHazards();
}

function closeHazardLibraryModal() {
    document.getElementById('hazardLibraryModal').style.display = 'none';
}

function loadHazards() {
    fetch('api/search-hazards.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                allHazards = data.hazards;
                
                // Extract categories
                const categorySet = new Set();
                allHazards.forEach(h => {
                    if (h.category_name) categorySet.add(JSON.stringify({id: h.category_id, name: h.category_name}));
                });
                
                hazardCategories = Array.from(categorySet).map(c => JSON.parse(c));
                
                // Populate category filter
                const select = document.getElementById('hazardCategoryFilter');
                select.innerHTML = '<option value="">All Categories</option>';
                hazardCategories.forEach(cat => {
                    const option = document.createElement('option');
                    option.value = cat.id;
                    option.textContent = cat.name;
                    select.appendChild(option);
                });
                
                displayHazards(allHazards);
            } else {
                document.getElementById('hazardsGrid').innerHTML = 
                    '<div style="grid-column: 1 / -1; text-align: center; padding: 2rem; color: var(--danger);">Error loading hazards</div>';
            }
        })
        .catch(error => {
            console.error('Error:', error);
            document.getElementById('hazardsGrid').innerHTML = 
                '<div style="grid-column: 1 / -1; text-align: center; padding: 2rem; color: var(--danger);">Error loading hazards</div>';
        });
}

function searchHazards() {
    const searchTerm = document.getElementById('hazardSearch').value.toLowerCase();
    const categoryId = document.getElementById('hazardCategoryFilter').value;
    
    let filtered = allHazards;
    
    if (searchTerm) {
        filtered = filtered.filter(h => 
            h.name.toLowerCase().includes(searchTerm) || 
            (h.description && h.description.toLowerCase().includes(searchTerm))
        );
    }
    
    if (categoryId) {
        filtered = filtered.filter(h => h.category_id == categoryId);
    }
    
    displayHazards(filtered);
}

function displayHazards(hazards) {
    const grid = document.getElementById('hazardsGrid');
    
    if (hazards.length === 0) {
        grid.innerHTML = '<div style="grid-column: 1 / -1; text-align: center; padding: 2rem; color: var(--gray-500);">No hazards found</div>';
        return;
    }
    
    grid.innerHTML = hazards.map(hazard => {
        const primaryPhoto = hazard.photos && hazard.photos.length > 0 
            ? hazard.photos.find(p => p.is_primary) || hazard.photos[0]
            : null;
        
        return `
            <div class="hazard-card" style="border: 2px solid var(--gray-200); border-radius: var(--radius-lg); overflow: hidden; cursor: pointer; transition: all 0.2s;"
                 onclick="addHazardToAssessment(${hazard.id})"
                 onmouseover="this.style.borderColor='var(--primary)'; this.style.transform='translateY(-2px)'"
                 onmouseout="this.style.borderColor='var(--gray-200)'; this.style.transform='translateY(0)'">
                
                ${primaryPhoto ? `
                    <div style="height: 120px; overflow: hidden; background: var(--gray-100);">
                        <img src="${primaryPhoto.thumbnail}" alt="${hazard.name}" 
                             style="width: 100%; height: 100%; object-fit: cover;">
                    </div>
                ` : `
                    <div style="height: 120px; background: linear-gradient(135deg, var(--gray-100), var(--gray-200)); 
                         display: flex; align-items: center; justify-content: center; font-size: 3rem;">
                        ⚠️
                    </div>
                `}
                
                <div style="padding: 1rem;">
                    <h4 style="margin: 0 0 0.5rem 0; font-size: 1rem; font-weight: 700;">
                        ${hazard.name}
                    </h4>
                    
                    <div style="margin-bottom: 0.5rem;">
                        <span class="badge badge-${hazard.residual_risk}" style="font-size: 0.65rem;">
                            ${hazard.residual_risk.toUpperCase()}
                        </span>
                    </div>
                    
                    <p style="font-size: 0.75rem; color: var(--gray-600); margin: 0; line-height: 1.4;">
                        ${hazard.description.substring(0, 80)}${hazard.description.length > 80 ? '...' : ''}
                    </p>
                    
                    <div style="margin-top: 0.75rem; display: flex; gap: 0.5rem; font-size: 0.7rem; color: var(--gray-500);">
                        <span>L:${hazard.default_likelihood}</span>
                        <span>S:${hazard.default_severity}</span>
                        <span>Score:${hazard.default_likelihood * hazard.default_severity}</span>
                    </div>
                </div>
            </div>
        `;
    }).join('');
}

function addHazardToAssessment(hazardId) {
    const assessmentId = <?php echo $assessmentId ?? 'null'; ?>;
    
    if (!assessmentId) {
        alert('Assessment ID not found');
        return;
    }
    
    fetch('api/add-hazard-to-assessment.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
            hazard_id: hazardId,
            assessment_id: assessmentId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            closeHazardLibraryModal();
            location.reload();
        } else {
            alert('Error: ' + data.message);
        }
    })
    .catch(error => {
        alert('Error adding hazard to assessment');
        console.error(error);
    });
}

// Close modal on overlay click
document.getElementById('hazardLibraryModal')?.addEventListener('click', function(e) {
    if (e.target === this) closeHazardLibraryModal();
});
</script>

<style>
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
    padding: 1rem;
}

.modal {
    background: white;
    border-radius: var(--radius-xl);
    box-shadow: var(--shadow-2xl);
    width: 100%;
}

.modal-header {
    padding: 1.5rem 2rem;
    border-bottom: 1px solid var(--gray-200);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-title {
    font-size: 1.5rem;
    font-weight: 700;
    margin: 0;
}

.modal-body {
    padding: 2rem;
}

.modal-footer {
    padding: 1.5rem 2rem;
    border-top: 1px solid var(--gray-200);
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
}
</style>
