<?php
/**
 * Helper Functions
 */

function sanitize($data) {
    return htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
}

function generateAssessmentNumber() {
    $year = date('Y');
    $db = Database::getInstance();
    
    $sql = "SELECT MAX(CAST(SUBSTRING(assessment_number, -6) AS UNSIGNED)) as max_num 
            FROM assessments 
            WHERE assessment_number LIKE ?";
    $result = $db->fetchOne($sql, ["RA-{$year}-%"]);
    
    $nextNum = ($result && $result['max_num']) ? $result['max_num'] + 1 : 1;
    
    return sprintf("RA-%s-%06d", $year, $nextNum);
}

function calculateRiskLevel($likelihood, $severity) {
    $score = $likelihood * $severity;
    
    if ($score <= 4) return 'low';
    if ($score <= 9) return 'medium';
    if ($score <= 16) return 'high';
    return 'critical';
}

function getRiskColor($level) {
    $colors = [
        'low' => '#10b981',
        'medium' => '#f59e0b',
        'high' => '#ef4444',
        'critical' => '#991b1b'
    ];
    return $colors[$level] ?? '#6b7280';
}

function formatDate($date, $format = 'Y-m-d') {
    if (!$date) return '';
    return date($format, strtotime($date));
}

function setFlashMessage($message, $type = 'info') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
}

function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = [
            'message' => $_SESSION['flash_message'],
            'type' => $_SESSION['flash_type']
        ];
        unset($_SESSION['flash_message'], $_SESSION['flash_type']);
        return $message;
    }
    return null;
}

/**
 * Enhanced audit logging function
 * Automatically logs actions with detailed information
 */
function logAudit($action, $entityType, $entityId = null, $oldValues = null, $newValues = null, $entityName = null) {
    require_once __DIR__ . '/AuditManager.php';
    $auditManager = new AuditManager();
    return $auditManager->log($action, $entityType, $entityId, $oldValues, $newValues, $entityName);
}

/**
 * Log user activity
 */
function logActivity($activityType, $description = null, $metadata = null) {
    require_once __DIR__ . '/AuditManager.php';
    $auditManager = new AuditManager();
    return $auditManager->logActivity($activityType, $description, $metadata);
}

/**
 * Check if user has audit permission
 */
function canViewAudits() {
    global $auth;
    if (!isset($auth)) {
        require_once __DIR__ . '/Auth.php';
        $auth = new Auth();
    }
    return $auth->hasPermission('audit', 'read') || ($_SESSION['role'] ?? '') === 'admin';
}

/**
 * Check if user can create versions
 */
function canCreateVersions() {
    global $auth;
    if (!isset($auth)) {
        require_once __DIR__ . '/Auth.php';
        $auth = new Auth();
    }
    return $auth->hasPermission('assessments', 'update');
}
