<?php
/**
 * Version Control Manager
 * Handles versioning for assessments and other entities
 */

class VersionControl {
    private $db;
    private $auditManager;
    
    public function __construct() {
        $this->db = Database::getInstance();
        require_once __DIR__ . '/AuditManager.php';
        $this->auditManager = new AuditManager();
    }
    
    /**
     * Create a new version of an assessment
     */
    public function createVersion($assessmentId, $versionType = 'manual', $notes = null) {
        // Get current assessment data
        $assessment = $this->getAssessmentWithDetails($assessmentId);
        
        if (!$assessment) {
            throw new Exception('Assessment not found');
        }
        
        // Get current version number
        $currentVersion = $this->getCurrentVersionNumber($assessmentId);
        $newVersionNumber = $currentVersion + 1;
        
        // Create snapshot
        $snapshot = $this->createSnapshot($assessment);
        
        // Insert version
        $sql = "INSERT INTO assessment_versions (
            assessment_id, version_number, created_by, version_type, version_notes, snapshot_data, status
        ) VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $this->db->query($sql, [
            $assessmentId,
            $newVersionNumber,
            $_SESSION['user_id'] ?? null,
            $versionType,
            $notes,
            json_encode($snapshot),
            $assessment['status']
        ]);
        
        $versionId = $this->db->lastInsertId();
        
        // Log to audit
        $this->auditManager->log(
            'version_created',
            'assessment',
            $assessmentId,
            null,
            ['version_number' => $newVersionNumber, 'type' => $versionType],
            $assessment['assessment_number']
        );
        
        return $versionId;
    }
    
    /**
     * Get all versions of an assessment
     */
    public function getVersions($assessmentId) {
        $sql = "SELECT av.*, u.full_name as created_by_name
                FROM assessment_versions av
                LEFT JOIN users u ON av.created_by = u.id
                WHERE av.assessment_id = ?
                ORDER BY av.version_number DESC";
        
        return $this->db->fetchAll($sql, [$assessmentId]);
    }
    
    /**
     * Get a specific version
     */
    public function getVersion($versionId) {
        $sql = "SELECT av.*, u.full_name as created_by_name
                FROM assessment_versions av
                LEFT JOIN users u ON av.created_by = u.id
                WHERE av.id = ?";
        
        $version = $this->db->fetchOne($sql, [$versionId]);
        
        if ($version && $version['snapshot_data']) {
            $version['snapshot'] = json_decode($version['snapshot_data'], true);
        }
        
        return $version;
    }
    
    /**
     * Compare two versions
     */
    public function compareVersions($versionId1, $versionId2) {
        $version1 = $this->getVersion($versionId1);
        $version2 = $this->getVersion($versionId2);
        
        if (!$version1 || !$version2) {
            throw new Exception('Version not found');
        }
        
        $snapshot1 = $version1['snapshot'];
        $snapshot2 = $version2['snapshot'];
        
        $differences = $this->findDifferences($snapshot1, $snapshot2);
        
        return [
            'version1' => $version1,
            'version2' => $version2,
            'differences' => $differences
        ];
    }
    
    /**
     * Restore an assessment from a version
     */
    public function restoreVersion($versionId) {
        $version = $this->getVersion($versionId);
        
        if (!$version) {
            throw new Exception('Version not found');
        }
        
        $snapshot = $version['snapshot'];
        $assessmentId = $version['assessment_id'];
        
        // Create a backup version before restoring
        $this->createVersion($assessmentId, 'auto', 'Backup before restore');
        
        // Restore main assessment data
        $sql = "UPDATE assessments SET
                title = ?,
                location_id = ?,
                visit_date = ?,
                visit_time = ?,
                assessor_id = ?,
                status = ?,
                overall_risk_level = ?,
                vehicle_type = ?,
                delivery_type = ?,
                weather_conditions = ?,
                additional_notes = ?,
                description = ?,
                updated_by = ?,
                updated_at = NOW()
                WHERE id = ?";
        
        $this->db->query($sql, [
            $snapshot['assessment']['title'] ?? '',
            $snapshot['assessment']['location_id'] ?? null,
            $snapshot['assessment']['visit_date'] ?? null,
            $snapshot['assessment']['visit_time'] ?? null,
            $snapshot['assessment']['assessor_id'] ?? null,
            $snapshot['assessment']['status'] ?? 'draft',
            $snapshot['assessment']['overall_risk_level'] ?? null,
            $snapshot['assessment']['vehicle_type'] ?? null,
            $snapshot['assessment']['delivery_type'] ?? null,
            $snapshot['assessment']['weather_conditions'] ?? null,
            $snapshot['assessment']['additional_notes'] ?? null,
            $snapshot['assessment']['description'] ?? null,
            $_SESSION['user_id'] ?? null,
            $assessmentId
        ]);
        
        // Log restore action
        $this->auditManager->log(
            'version_restored',
            'assessment',
            $assessmentId,
            null,
            ['restored_version' => $version['version_number']],
            $snapshot['assessment']['assessment_number'] ?? "Assessment #{$assessmentId}"
        );
        
        return true;
    }
    
    /**
     * Delete old versions based on retention policy
     */
    public function cleanupOldVersions() {
        $policy = $this->db->fetchOne(
            "SELECT retention_days FROM retention_policies 
             WHERE entity_type = 'assessment_versions' AND is_active = 1"
        );
        
        if ($policy) {
            $sql = "DELETE FROM assessment_versions 
                    WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
                    AND version_type = 'auto'";
            $this->db->query($sql, [$policy['retention_days']]);
            
            return $this->db->rowCount();
        }
        
        return 0;
    }
    
    /**
     * Get version statistics
     */
    public function getVersionStatistics($assessmentId = null) {
        if ($assessmentId) {
            return [
                'total_versions' => $this->getVersionCount($assessmentId),
                'manual_versions' => $this->getVersionCount($assessmentId, 'manual'),
                'auto_versions' => $this->getVersionCount($assessmentId, 'auto'),
                'latest_version' => $this->getLatestVersion($assessmentId)
            ];
        }
        
        // System-wide statistics
        return [
            'total_versions' => $this->getTotalVersions(),
            'assessments_with_versions' => $this->getAssessmentsWithVersions(),
            'avg_versions_per_assessment' => $this->getAverageVersionsPerAssessment()
        ];
    }
    
    // Private helper methods
    
    private function getAssessmentWithDetails($assessmentId) {
        // Get main assessment
        $assessment = $this->db->fetchOne(
            "SELECT a.*, l.name as location_name, u.full_name as assessor_name
             FROM assessments a
             LEFT JOIN locations l ON a.location_id = l.id
             LEFT JOIN users u ON a.assessor_id = u.id
             WHERE a.id = ?",
            [$assessmentId]
        );
        
        if (!$assessment) return null;
        
        // Get risks
        $assessment['risks'] = $this->db->fetchAll(
            "SELECT * FROM assessment_risks WHERE assessment_id = ? ORDER BY id",
            [$assessmentId]
        );
        
        // Get action items
        $assessment['actions'] = $this->db->fetchAll(
            "SELECT * FROM action_items WHERE assessment_id = ? ORDER BY id",
            [$assessmentId]
        );
        
        return $assessment;
    }
    
    private function createSnapshot($assessment) {
        return [
            'assessment' => [
                'id' => $assessment['id'],
                'assessment_number' => $assessment['assessment_number'],
                'title' => $assessment['title'],
                'location_id' => $assessment['location_id'],
                'location_name' => $assessment['location_name'],
                'visit_date' => $assessment['visit_date'],
                'assessor_id' => $assessment['assessor_id'],
                'assessor_name' => $assessment['assessor_name'],
                'status' => $assessment['status'],
                'overall_risk_level' => $assessment['overall_risk_level'],
                'vehicle_type' => $assessment['vehicle_type'],
                'delivery_type' => $assessment['delivery_type'],
                'weather_conditions' => $assessment['weather_conditions'],
                'additional_notes' => $assessment['additional_notes'],
                'created_at' => $assessment['created_at'],
                'updated_at' => $assessment['updated_at']
            ],
            'risks' => $assessment['risks'] ?? [],
            'actions' => $assessment['actions'] ?? [],
            'snapshot_created_at' => date('Y-m-d H:i:s')
        ];
    }
    
    private function getCurrentVersionNumber($assessmentId) {
        $result = $this->db->fetchOne(
            "SELECT MAX(version_number) as max_version 
             FROM assessment_versions 
             WHERE assessment_id = ?",
            [$assessmentId]
        );
        
        return $result['max_version'] ?? 0;
    }
    
    private function findDifferences($snapshot1, $snapshot2) {
        $differences = [];
        
        // Compare assessment fields
        foreach ($snapshot1['assessment'] as $key => $value1) {
            $value2 = $snapshot2['assessment'][$key] ?? null;
            if ($value1 != $value2) {
                $differences['assessment'][$key] = [
                    'old' => $value1,
                    'new' => $value2
                ];
            }
        }
        
        // Compare risks count
        $differences['risks_count'] = [
            'old' => count($snapshot1['risks'] ?? []),
            'new' => count($snapshot2['risks'] ?? [])
        ];
        
        // Compare actions count
        $differences['actions_count'] = [
            'old' => count($snapshot1['actions'] ?? []),
            'new' => count($snapshot2['actions'] ?? [])
        ];
        
        return $differences;
    }
    
    private function getVersionCount($assessmentId, $type = null) {
        $sql = "SELECT COUNT(*) as count FROM assessment_versions WHERE assessment_id = ?";
        $params = [$assessmentId];
        
        if ($type) {
            $sql .= " AND version_type = ?";
            $params[] = $type;
        }
        
        $result = $this->db->fetchOne($sql, $params);
        return $result['count'] ?? 0;
    }
    
    private function getLatestVersion($assessmentId) {
        return $this->db->fetchOne(
            "SELECT * FROM assessment_versions 
             WHERE assessment_id = ? 
             ORDER BY version_number DESC 
             LIMIT 1",
            [$assessmentId]
        );
    }
    
    private function getTotalVersions() {
        $result = $this->db->fetchOne("SELECT COUNT(*) as count FROM assessment_versions");
        return $result['count'] ?? 0;
    }
    
    private function getAssessmentsWithVersions() {
        $result = $this->db->fetchOne("SELECT COUNT(DISTINCT assessment_id) as count FROM assessment_versions");
        return $result['count'] ?? 0;
    }
    
    private function getAverageVersionsPerAssessment() {
        $result = $this->db->fetchOne(
            "SELECT AVG(version_count) as avg FROM (
                SELECT assessment_id, COUNT(*) as version_count 
                FROM assessment_versions 
                GROUP BY assessment_id
            ) as counts"
        );
        return round($result['avg'] ?? 0, 1);
    }
}
