<?php
/**
 * Theme Manager
 * Handles color scheme customization
 */

class ThemeManager {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Get all themes
     */
    public function getThemes($activeOnly = false) {
        $sql = "SELECT * FROM color_themes";
        if ($activeOnly) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY is_default DESC, theme_name ASC";
        
        return $this->db->fetchAll($sql);
    }
    
    /**
     * Get theme by ID
     */
    public function getTheme($themeId) {
        return $this->db->fetchOne("SELECT * FROM color_themes WHERE id = ?", [$themeId]);
    }
    
    /**
     * Get default theme
     */
    public function getDefaultTheme() {
        $theme = $this->db->fetchOne("SELECT * FROM color_themes WHERE is_default = 1");
        if (!$theme) {
            // Fallback to first active theme
            $theme = $this->db->fetchOne("SELECT * FROM color_themes WHERE is_active = 1 LIMIT 1");
        }
        return $theme;
    }
    
    /**
     * Get user's preferred theme
     */
    public function getUserTheme($userId) {
        $sql = "SELECT ct.* FROM color_themes ct
                JOIN user_preferences up ON ct.id = up.theme_id
                WHERE up.user_id = ? AND ct.is_active = 1";
        
        $theme = $this->db->fetchOne($sql, [$userId]);
        
        if (!$theme) {
            $theme = $this->getDefaultTheme();
        }
        
        return $theme;
    }
    
    /**
     * Create new theme
     */
    public function createTheme($data) {
        $sql = "INSERT INTO color_themes (
            theme_name, primary_color, secondary_color, accent_color,
            success_color, warning_color, danger_color,
            background_gradient_start, background_gradient_end,
            created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $this->db->query($sql, [
            $data['theme_name'],
            $data['primary_color'],
            $data['secondary_color'],
            $data['accent_color'],
            $data['success_color'] ?? '#10b981',
            $data['warning_color'] ?? '#f59e0b',
            $data['danger_color'] ?? '#ef4444',
            $data['background_gradient_start'],
            $data['background_gradient_end'],
            $_SESSION['user_id'] ?? null
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Update theme
     */
    public function updateTheme($themeId, $data) {
        $sql = "UPDATE color_themes SET
            theme_name = ?,
            primary_color = ?,
            secondary_color = ?,
            accent_color = ?,
            success_color = ?,
            warning_color = ?,
            danger_color = ?,
            background_gradient_start = ?,
            background_gradient_end = ?
            WHERE id = ?";
        
        $this->db->query($sql, [
            $data['theme_name'],
            $data['primary_color'],
            $data['secondary_color'],
            $data['accent_color'],
            $data['success_color'] ?? '#10b981',
            $data['warning_color'] ?? '#f59e0b',
            $data['danger_color'] ?? '#ef4444',
            $data['background_gradient_start'],
            $data['background_gradient_end'],
            $themeId
        ]);
    }
    
    /**
     * Set default theme
     */
    public function setDefaultTheme($themeId) {
        // Remove default from all themes
        $this->db->query("UPDATE color_themes SET is_default = 0");
        
        // Set new default
        $this->db->query("UPDATE color_themes SET is_default = 1 WHERE id = ?", [$themeId]);
    }
    
    /**
     * Set user theme preference
     */
    public function setUserTheme($userId, $themeId) {
        $sql = "INSERT INTO user_preferences (user_id, theme_id) 
                VALUES (?, ?)
                ON DUPLICATE KEY UPDATE theme_id = ?";
        
        $this->db->query($sql, [$userId, $themeId, $themeId]);
    }
    
    /**
     * Delete theme
     */
    public function deleteTheme($themeId) {
        // Don't allow deleting default theme
        $theme = $this->getTheme($themeId);
        if ($theme && $theme['is_default']) {
            throw new Exception('Cannot delete default theme');
        }
        
        $this->db->query("DELETE FROM color_themes WHERE id = ?", [$themeId]);
    }
    
    /**
     * Generate CSS from theme
     */
    public function generateCSS($theme) {
        if (!$theme) {
            return '';
        }
        
        return "
        :root {
            --primary: {$theme['primary_color']};
            --secondary: {$theme['secondary_color']};
            --accent: {$theme['accent_color']};
            --success: {$theme['success_color']};
            --warning: {$theme['warning_color']};
            --danger: {$theme['danger_color']};
        }
        
        body {
            background: linear-gradient(135deg, {$theme['background_gradient_start']} 0%, {$theme['background_gradient_end']} 100%);
        }
        
        .navbar h1,
        .stat-card .value,
        .page-title {
            background: linear-gradient(135deg, {$theme['primary_color']}, {$theme['secondary_color']});
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, {$theme['primary_color']}, {$theme['secondary_color']});
        }
        
        .stat-card::before {
            background: linear-gradient(90deg, {$theme['primary_color']}, {$theme['secondary_color']});
        }
        ";
    }
}
