<?php
/**
 * Photo Upload Handler
 * Manages photo uploads for risk assessment items
 */

class PhotoUpload {
    private $db;
    private $uploadPath;
    private $maxFileSize = 5242880; // 5MB
    private $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    private $maxWidth = 2000;
    private $maxHeight = 2000;
    private $thumbWidth = 300;
    private $thumbHeight = 300;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->uploadPath = ROOT_PATH . '/public/uploads/assessments/';
        
        // Ensure upload directory exists
        if (!is_dir($this->uploadPath)) {
            mkdir($this->uploadPath, 0755, true);
        }
    }
    
    /**
     * Upload a photo
     */
    public function upload($file, $assessmentId, $itemId) {
        // Validate file
        $validation = $this->validateFile($file);
        if ($validation !== true) {
            return ['success' => false, 'message' => $validation];
        }
        
        // Generate unique filename
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $filename = sprintf(
            'assessment_%d_item_%d_%d_%s.%s',
            $assessmentId,
            $itemId,
            time(),
            substr(md5(uniqid()), 0, 8),
            $extension
        );
        
        $filepath = $this->uploadPath . $filename;
        
        // Process and save image
        if (!$this->processImage($file['tmp_name'], $filepath, $file['type'])) {
            return ['success' => false, 'message' => 'Failed to process image'];
        }
        
        // Create thumbnail
        $this->createThumbnail($filepath, $filename);
        
        // Add to database
        if (!$this->addPhotoToItem($itemId, $filename)) {
            // Clean up files if database update fails
            @unlink($filepath);
            @unlink($this->uploadPath . 'thumb_' . $filename);
            return ['success' => false, 'message' => 'Failed to update database'];
        }
        
        return [
            'success' => true,
            'photo' => [
                'filename' => $filename,
                'url' => SITE_URL . 'uploads/assessments/' . $filename,
                'thumbnail' => SITE_URL . 'uploads/assessments/thumb_' . $filename
            ]
        ];
    }
    
    /**
     * Delete a photo
     */
    public function delete($filename) {
        $filepath = $this->uploadPath . $filename;
        $thumbpath = $this->uploadPath . 'thumb_' . $filename;
        
        $deleted = false;
        if (file_exists($filepath)) {
            $deleted = @unlink($filepath);
        }
        
        if (file_exists($thumbpath)) {
            @unlink($thumbpath);
        }
        
        return $deleted;
    }
    
    /**
     * Get photos for an item
     */
    public function getPhotos($itemId) {
        $sql = "SELECT photos FROM assessment_risks WHERE id = ?";
        $result = $this->db->fetchOne($sql, [$itemId]);
        
        if (!$result || !$result['photos']) {
            return [];
        }
        
        $photos = json_decode($result['photos'], true);
        if (!is_array($photos)) {
            return [];
        }
        
        return array_map(function($filename) {
            return [
                'filename' => $filename,
                'url' => SITE_URL . 'uploads/assessments/' . $filename,
                'thumbnail' => SITE_URL . 'uploads/assessments/thumb_' . $filename
            ];
        }, $photos);
    }
    
    /**
     * Add photo to item in database
     */
    public function addPhotoToItem($itemId, $filename) {
        // Get current photos
        $sql = "SELECT photos FROM assessment_risks WHERE id = ?";
        $result = $this->db->fetchOne($sql, [$itemId]);
        
        if (!$result) {
            return false;
        }
        
        $photos = $result['photos'] ? json_decode($result['photos'], true) : [];
        if (!is_array($photos)) {
            $photos = [];
        }
        
        $photos[] = $filename;
        
        // Update database
        $sql = "UPDATE assessment_risks SET photos = ? WHERE id = ?";
        try {
            $this->db->query($sql, [json_encode($photos), $itemId]);
            return true;
        } catch (Exception $e) {
            error_log('Failed to add photo to item: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Remove photo from item in database
     */
    public function removePhotoFromItem($itemId, $filename) {
        // Get current photos
        $sql = "SELECT photos FROM assessment_risks WHERE id = ?";
        $result = $this->db->fetchOne($sql, [$itemId]);
        
        if (!$result) {
            return false;
        }
        
        $photos = $result['photos'] ? json_decode($result['photos'], true) : [];
        if (!is_array($photos)) {
            return false;
        }
        
        // Remove filename from array
        $photos = array_values(array_filter($photos, function($p) use ($filename) {
            return $p !== $filename;
        }));
        
        // Update database
        $sql = "UPDATE assessment_risks SET photos = ? WHERE id = ?";
        try {
            $this->db->query($sql, [json_encode($photos), $itemId]);
            return true;
        } catch (Exception $e) {
            error_log('Failed to remove photo from item: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Validate uploaded file
     */
    private function validateFile($file) {
        if (!isset($file['error']) || is_array($file['error'])) {
            return 'Invalid file upload';
        }
        
        switch ($file['error']) {
            case UPLOAD_ERR_OK:
                break;
            case UPLOAD_ERR_INI_SIZE:
            case UPLOAD_ERR_FORM_SIZE:
                return 'File size exceeds limit (5MB)';
            case UPLOAD_ERR_NO_FILE:
                return 'No file uploaded';
            default:
                return 'Upload error occurred';
        }
        
        if ($file['size'] > $this->maxFileSize) {
            return 'File size exceeds 5MB limit';
        }
        
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($file['tmp_name']);
        
        if (!in_array($mimeType, $this->allowedTypes)) {
            return 'Invalid file type. Only JPEG, PNG, GIF, and WebP images allowed';
        }
        
        return true;
    }
    
    /**
     * Process and resize image
     */
    private function processImage($source, $destination, $mimeType) {
        // Load image
        switch ($mimeType) {
            case 'image/jpeg':
            case 'image/jpg':
                $image = @imagecreatefromjpeg($source);
                break;
            case 'image/png':
                $image = @imagecreatefrompng($source);
                break;
            case 'image/gif':
                $image = @imagecreatefromgif($source);
                break;
            case 'image/webp':
                $image = @imagecreatefromwebp($source);
                break;
            default:
                return false;
        }
        
        if (!$image) {
            return false;
        }
        
        // Get dimensions
        $width = imagesx($image);
        $height = imagesy($image);
        
        // Resize if needed
        if ($width > $this->maxWidth || $height > $this->maxHeight) {
            $ratio = min($this->maxWidth / $width, $this->maxHeight / $height);
            $newWidth = round($width * $ratio);
            $newHeight = round($height * $ratio);
            
            $resized = imagecreatetruecolor($newWidth, $newHeight);
            
            // Preserve transparency for PNG and GIF
            if ($mimeType === 'image/png' || $mimeType === 'image/gif') {
                imagealphablending($resized, false);
                imagesavealpha($resized, true);
                $transparent = imagecolorallocatealpha($resized, 255, 255, 255, 127);
                imagefilledrectangle($resized, 0, 0, $newWidth, $newHeight, $transparent);
            }
            
            imagecopyresampled($resized, $image, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
            imagedestroy($image);
            $image = $resized;
        }
        
        // Save image
        $result = false;
        switch ($mimeType) {
            case 'image/jpeg':
            case 'image/jpg':
                $result = imagejpeg($image, $destination, 85);
                break;
            case 'image/png':
                $result = imagepng($image, $destination, 8);
                break;
            case 'image/gif':
                $result = imagegif($image, $destination);
                break;
            case 'image/webp':
                $result = imagewebp($image, $destination, 85);
                break;
        }
        
        imagedestroy($image);
        return $result;
    }
    
    /**
     * Create thumbnail
     */
    private function createThumbnail($filepath, $filename) {
        $image = @imagecreatefromstring(file_get_contents($filepath));
        if (!$image) {
            return false;
        }
        
        $width = imagesx($image);
        $height = imagesy($image);
        
        // Calculate thumbnail dimensions (square crop from center)
        $size = min($width, $height);
        $x = ($width - $size) / 2;
        $y = ($height - $size) / 2;
        
        $thumb = imagecreatetruecolor($this->thumbWidth, $this->thumbHeight);
        
        // Preserve transparency
        imagealphablending($thumb, false);
        imagesavealpha($thumb, true);
        
        imagecopyresampled(
            $thumb, $image,
            0, 0, $x, $y,
            $this->thumbWidth, $this->thumbHeight,
            $size, $size
        );
        
        $thumbPath = $this->uploadPath . 'thumb_' . $filename;
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        switch ($extension) {
            case 'jpg':
            case 'jpeg':
                imagejpeg($thumb, $thumbPath, 85);
                break;
            case 'png':
                imagepng($thumb, $thumbPath, 8);
                break;
            case 'gif':
                imagegif($thumb, $thumbPath);
                break;
            case 'webp':
                imagewebp($thumb, $thumbPath, 85);
                break;
        }
        
        imagedestroy($image);
        imagedestroy($thumb);
        
        return true;
    }
}
