<?php
/**
 * PDF Template Manager
 * Handles customizable PDF export layouts
 */

class PDFTemplateManager {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Get all templates
     */
    public function getTemplates($activeOnly = false) {
        $sql = "SELECT * FROM pdf_templates";
        if ($activeOnly) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY is_default DESC, template_name ASC";
        
        return $this->db->fetchAll($sql);
    }
    
    /**
     * Get template by ID
     */
    public function getTemplate($templateId) {
        return $this->db->fetchOne("SELECT * FROM pdf_templates WHERE id = ?", [$templateId]);
    }
    
    /**
     * Get default template
     */
    public function getDefaultTemplate() {
        $template = $this->db->fetchOne("SELECT * FROM pdf_templates WHERE is_default = 1");
        if (!$template) {
            $template = $this->db->fetchOne("SELECT * FROM pdf_templates WHERE is_active = 1 LIMIT 1");
        }
        return $template;
    }
    
    /**
     * Create new template
     */
    public function createTemplate($data) {
        $sql = "INSERT INTO pdf_templates (
            template_name, header_logo_url, header_text, footer_text,
            show_risk_matrix, show_photos, show_action_items, show_signatures,
            page_orientation, font_family, font_size, include_fields, custom_css,
            created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $includeFields = isset($data['include_fields']) ? json_encode($data['include_fields']) : null;
        
        $this->db->query($sql, [
            $data['template_name'],
            $data['header_logo_url'] ?? null,
            $data['header_text'] ?? '',
            $data['footer_text'] ?? '',
            $data['show_risk_matrix'] ?? 1,
            $data['show_photos'] ?? 1,
            $data['show_action_items'] ?? 1,
            $data['show_signatures'] ?? 0,
            $data['page_orientation'] ?? 'portrait',
            $data['font_family'] ?? 'Arial',
            $data['font_size'] ?? 12,
            $includeFields,
            $data['custom_css'] ?? null,
            $_SESSION['user_id'] ?? null
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Update template
     */
    public function updateTemplate($templateId, $data) {
        $sql = "UPDATE pdf_templates SET
            template_name = ?,
            header_logo_url = ?,
            header_text = ?,
            footer_text = ?,
            show_risk_matrix = ?,
            show_photos = ?,
            show_action_items = ?,
            show_signatures = ?,
            page_orientation = ?,
            font_family = ?,
            font_size = ?,
            include_fields = ?,
            custom_css = ?
            WHERE id = ?";
        
        $includeFields = isset($data['include_fields']) ? json_encode($data['include_fields']) : null;
        
        $this->db->query($sql, [
            $data['template_name'],
            $data['header_logo_url'] ?? null,
            $data['header_text'] ?? '',
            $data['footer_text'] ?? '',
            $data['show_risk_matrix'] ?? 1,
            $data['show_photos'] ?? 1,
            $data['show_action_items'] ?? 1,
            $data['show_signatures'] ?? 0,
            $data['page_orientation'] ?? 'portrait',
            $data['font_family'] ?? 'Arial',
            $data['font_size'] ?? 12,
            $includeFields,
            $data['custom_css'] ?? null,
            $templateId
        ]);
    }
    
    /**
     * Set default template
     */
    public function setDefaultTemplate($templateId) {
        $this->db->query("UPDATE pdf_templates SET is_default = 0");
        $this->db->query("UPDATE pdf_templates SET is_default = 1 WHERE id = ?", [$templateId]);
    }
    
    /**
     * Delete template
     */
    public function deleteTemplate($templateId) {
        $template = $this->getTemplate($templateId);
        if ($template && $template['is_default']) {
            throw new Exception('Cannot delete default template');
        }
        
        $this->db->query("DELETE FROM pdf_templates WHERE id = ?", [$templateId]);
    }
    
    /**
     * Generate PDF HTML
     */
    public function generatePDFHTML($assessment, $risks, $actions, $templateId = null) {
        if (!$templateId) {
            $template = $this->getDefaultTemplate();
        } else {
            $template = $this->getTemplate($templateId);
        }
        
        if (!$template) {
            throw new Exception('No PDF template found');
        }
        
        $html = $this->buildPDFHTML($assessment, $risks, $actions, $template);
        return $html;
    }
    
    /**
     * Build PDF HTML
     */
    private function buildPDFHTML($assessment, $risks, $actions, $template) {
        $includeFields = $template['include_fields'] ? json_decode($template['include_fields'], true) : [];
        
        $html = "<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Assessment Report - {$assessment['assessment_number']}</title>
    <style>
        @page { size: {$template['page_orientation']}; margin: 20mm; }
        body {
            font-family: {$template['font_family']}, sans-serif;
            font-size: {$template['font_size']}pt;
            line-height: 1.6;
            color: #333;
        }
        .header {
            text-align: center;
            border-bottom: 3px solid #6366f1;
            padding-bottom: 20px;
            margin-bottom: 30px;
        }
        .header h1 {
            color: #6366f1;
            margin: 0;
            font-size: 24pt;
        }
        .logo {
            max-width: 200px;
            margin-bottom: 10px;
        }
        .section {
            margin-bottom: 30px;
            page-break-inside: avoid;
        }
        .section-title {
            background: #f3f4f6;
            padding: 10px 15px;
            border-left: 4px solid #6366f1;
            font-weight: bold;
            font-size: 14pt;
            margin-bottom: 15px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        th, td {
            border: 1px solid #e5e7eb;
            padding: 8px;
            text-align: left;
        }
        th {
            background: #f9fafb;
            font-weight: bold;
        }
        .risk-low { background: #d1fae5; }
        .risk-medium { background: #fef3c7; }
        .risk-high { background: #fee2e2; }
        .risk-critical { background: #dc2626; color: white; }
        .info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
            margin-bottom: 20px;
        }
        .info-item {
            padding: 10px;
            background: #f9fafb;
            border-radius: 5px;
        }
        .info-label {
            font-weight: bold;
            color: #6b7280;
            font-size: 10pt;
        }
        .footer {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            text-align: center;
            font-size: 9pt;
            color: #6b7280;
            border-top: 1px solid #e5e7eb;
            padding-top: 10px;
        }
        .page-break { page-break-after: always; }
        .signature-box {
            border: 1px solid #e5e7eb;
            padding: 20px;
            margin-top: 40px;
            min-height: 80px;
        }
        {$template['custom_css']}
    </style>
</head>
<body>";
        
        // Header
        $html .= "<div class='header'>";
        if ($template['header_logo_url']) {
            $html .= "<img src='{$template['header_logo_url']}' class='logo'>";
        }
        $headerText = str_replace('{{date}}', date('d/m/Y'), $template['header_text']);
        $html .= "<h1>{$headerText}</h1>";
        $html .= "<p><strong>{$assessment['assessment_number']}</strong></p>";
        $html .= "</div>";
        
        // Assessment Info
        $html .= "<div class='section'>";
        $html .= "<div class='section-title'>Assessment Information</div>";
        $html .= "<div class='info-grid'>";
        
        if (empty($includeFields) || in_array('title', $includeFields)) {
            $html .= "<div class='info-item'><div class='info-label'>Title</div>{$assessment['title']}</div>";
        }
        if (empty($includeFields) || in_array('location', $includeFields)) {
            $html .= "<div class='info-item'><div class='info-label'>Location</div>{$assessment['location_name']}</div>";
        }
        if (empty($includeFields) || in_array('visit_date', $includeFields)) {
            $html .= "<div class='info-item'><div class='info-label'>Visit Date</div>" . date('d/m/Y', strtotime($assessment['visit_date'])) . "</div>";
        }
        if (empty($includeFields) || in_array('assessor', $includeFields)) {
            $html .= "<div class='info-item'><div class='info-label'>Assessor</div>{$assessment['assessor_name']}</div>";
        }
        if ($assessment['vehicle_type'] && (empty($includeFields) || in_array('vehicle_type', $includeFields))) {
            $html .= "<div class='info-item'><div class='info-label'>Vehicle Type</div>{$assessment['vehicle_type']}</div>";
        }
        if ($assessment['weather_conditions'] && (empty($includeFields) || in_array('weather', $includeFields))) {
            $html .= "<div class='info-item'><div class='info-label'>Weather</div>{$assessment['weather_conditions']}</div>";
        }
        
        $html .= "</div></div>";
        
        // Risks
        if (!empty($risks)) {
            $html .= "<div class='section'>";
            $html .= "<div class='section-title'>Risk Assessment</div>";
            $html .= "<table>";
            $html .= "<thead><tr>";
            $html .= "<th>Hazard</th>";
            $html .= "<th style='width: 10%;'>L</th>";
            $html .= "<th style='width: 10%;'>S</th>";
            $html .= "<th style='width: 10%;'>Score</th>";
            $html .= "<th style='width: 15%;'>Level</th>";
            $html .= "<th>Controls</th>";
            $html .= "</tr></thead><tbody>";
            
            foreach ($risks as $risk) {
                $html .= "<tr>";
                $html .= "<td><strong>{$risk['hazard_name']}</strong>";
                if ($risk['hazard_description']) {
                    $html .= "<br><small>{$risk['hazard_description']}</small>";
                }
                $html .= "</td>";
                $html .= "<td style='text-align:center;'>{$risk['likelihood']}</td>";
                $html .= "<td style='text-align:center;'>{$risk['severity']}</td>";
                $html .= "<td style='text-align:center;'><strong>{$risk['risk_score']}</strong></td>";
                $html .= "<td class='risk-{$risk['risk_level']}'>" . strtoupper($risk['risk_level']) . "</td>";
                $html .= "<td>" . nl2br(htmlspecialchars($risk['existing_controls'] ?? 'None')) . "</td>";
                $html .= "</tr>";
            }
            
            $html .= "</tbody></table></div>";
        }
        
        // Action Items
        if ($template['show_action_items'] && !empty($actions)) {
            $html .= "<div class='section'>";
            $html .= "<div class='section-title'>Action Items</div>";
            $html .= "<table>";
            $html .= "<thead><tr><th>Action</th><th>Assigned To</th><th>Due Date</th><th>Priority</th></tr></thead>";
            $html .= "<tbody>";
            
            foreach ($actions as $action) {
                $html .= "<tr>";
                $html .= "<td>{$action['action_description']}</td>";
                $html .= "<td>" . ($action['assigned_to_name'] ?? 'Unassigned') . "</td>";
                $html .= "<td>" . ($action['due_date'] ? date('d/m/Y', strtotime($action['due_date'])) : '-') . "</td>";
                $html .= "<td>" . ucfirst($action['priority']) . "</td>";
                $html .= "</tr>";
            }
            
            $html .= "</tbody></table></div>";
        }
        
        // Signatures
        if ($template['show_signatures']) {
            $html .= "<div class='section'>";
            $html .= "<div class='section-title'>Signatures</div>";
            $html .= "<div class='signature-box'>";
            $html .= "<strong>Assessor Signature:</strong> _____________________________ Date: ___________";
            $html .= "</div>";
            $html .= "<div class='signature-box'>";
            $html .= "<strong>Reviewed By:</strong> _____________________________ Date: ___________";
            $html .= "</div>";
            $html .= "</div>";
        }
        
        // Footer
        $footerText = str_replace('{{date}}', date('d/m/Y H:i'), $template['footer_text']);
        $html .= "<div class='footer'>{$footerText}</div>";
        
        $html .= "</body></html>";
        
        return $html;
    }
}
