<?php
/**
 * Hazard Library Manager
 * Manages predefined hazards, controls, and reference photos
 */

class HazardLibrary {
    private $db;
    private $uploadPath;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->uploadPath = ROOT_PATH . '/uploads/hazard-library/';
        
        if (!is_dir($this->uploadPath)) {
            mkdir($this->uploadPath, 0755, true);
        }
    }
    
    /**
     * Get all hazards with optional filtering
     */
    public function getHazards($categoryId = null, $searchTerm = null, $includePhotos = false) {
        $sql = "SELECT ri.*, rc.name as category_name,
                u.full_name as created_by_name,
                (SELECT COUNT(*) FROM control_measures WHERE risk_item_id = ri.id) as control_count
                FROM risk_items ri
                LEFT JOIN risk_categories rc ON ri.category_id = rc.id
                LEFT JOIN users u ON ri.created_by = u.id
                WHERE ri.is_template = 1";
        
        $params = [];
        
        if ($categoryId) {
            $sql .= " AND ri.category_id = ?";
            $params[] = $categoryId;
        }
        
        if ($searchTerm) {
            $sql .= " AND (ri.name LIKE ? OR ri.description LIKE ?)";
            $searchParam = "%{$searchTerm}%";
            $params[] = $searchParam;
            $params[] = $searchParam;
        }
        
        $sql .= " ORDER BY rc.display_order, ri.name";
        
        $hazards = $this->db->fetchAll($sql, $params);
        
        if ($includePhotos) {
            foreach ($hazards as &$hazard) {
                $hazard['photos'] = $this->getHazardPhotos($hazard['id']);
            }
        }
        
        return $hazards;
    }
    
    /**
     * Get single hazard by ID
     */
    public function getHazard($id) {
        $hazard = $this->db->fetchOne("
            SELECT ri.*, rc.name as category_name
            FROM risk_items ri
            LEFT JOIN risk_categories rc ON ri.category_id = rc.id
            WHERE ri.id = ?
        ", [$id]);
        
        if ($hazard) {
            $hazard['controls'] = $this->getControls($id);
            $hazard['photos'] = $this->getHazardPhotos($id);
        }
        
        return $hazard;
    }
    
    /**
     * Create new hazard template
     */
    public function createHazard($data) {
        $sql = "INSERT INTO risk_items (
            category_id, name, description, default_likelihood, default_severity,
            typical_controls, affected_persons, residual_risk, is_template, created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1, ?)";
        
        $this->db->query($sql, [
            $data['category_id'],
            $data['name'],
            $data['description'],
            $data['default_likelihood'],
            $data['default_severity'],
            $data['typical_controls'] ?? null,
            $data['affected_persons'] ?? null,
            $data['residual_risk'] ?? 'medium',
            $_SESSION['user_id'] ?? null
        ]);
        
        $hazardId = $this->db->lastInsertId();
        
        // Add control measures if provided
        if (isset($data['controls']) && is_array($data['controls'])) {
            foreach ($data['controls'] as $control) {
                $this->addControl($hazardId, $control);
            }
        }
        
        logAudit($_SESSION['user_id'], 'create', 'hazard_template', $hazardId);
        
        return $hazardId;
    }
    
    /**
     * Update hazard template
     */
    public function updateHazard($id, $data) {
        $sql = "UPDATE risk_items SET
            category_id = ?,
            name = ?,
            description = ?,
            default_likelihood = ?,
            default_severity = ?,
            typical_controls = ?,
            affected_persons = ?,
            residual_risk = ?
            WHERE id = ?";
        
        $this->db->query($sql, [
            $data['category_id'],
            $data['name'],
            $data['description'],
            $data['default_likelihood'],
            $data['default_severity'],
            $data['typical_controls'] ?? null,
            $data['affected_persons'] ?? null,
            $data['residual_risk'] ?? 'medium',
            $id
        ]);
        
        logAudit($_SESSION['user_id'], 'update', 'hazard_template', $id);
    }
    
    /**
     * Delete hazard template
     */
    public function deleteHazard($id) {
        // Delete photos first
        $photos = $this->getHazardPhotos($id);
        foreach ($photos as $photo) {
            $this->deletePhoto($photo['id']);
        }
        
        // Delete hazard (cascade will handle controls)
        $this->db->query("DELETE FROM risk_items WHERE id = ?", [$id]);
        
        logAudit($_SESSION['user_id'], 'delete', 'hazard_template', $id);
    }
    
    /**
     * Get control measures for hazard
     */
    public function getControls($hazardId) {
        return $this->db->fetchAll("
            SELECT * FROM control_measures
            WHERE risk_item_id = ?
            ORDER BY display_order, id
        ", [$hazardId]);
    }
    
    /**
     * Add control measure
     */
    public function addControl($hazardId, $data) {
        $sql = "INSERT INTO control_measures (
            risk_item_id, control_type, control_description,
            implementation_notes, cost_estimate, effectiveness, display_order
        ) VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $this->db->query($sql, [
            $hazardId,
            $data['control_type'],
            $data['control_description'],
            $data['implementation_notes'] ?? null,
            $data['cost_estimate'] ?? null,
            $data['effectiveness'] ?? 'high',
            $data['display_order'] ?? 0
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Update control measure
     */
    public function updateControl($controlId, $data) {
        $sql = "UPDATE control_measures SET
            control_type = ?,
            control_description = ?,
            implementation_notes = ?,
            cost_estimate = ?,
            effectiveness = ?,
            display_order = ?
            WHERE id = ?";
        
        $this->db->query($sql, [
            $data['control_type'],
            $data['control_description'],
            $data['implementation_notes'] ?? null,
            $data['cost_estimate'] ?? null,
            $data['effectiveness'] ?? 'high',
            $data['display_order'] ?? 0,
            $controlId
        ]);
    }
    
    /**
     * Delete control measure
     */
    public function deleteControl($controlId) {
        $this->db->query("DELETE FROM control_measures WHERE id = ?", [$controlId]);
    }
    
    /**
     * Upload photo for hazard
     */
    public function uploadPhoto($file, $hazardId, $caption = null, $isPrimary = false) {
        // Validate file
        if ($file['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('File upload error');
        }
        
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($file['tmp_name']);
        
        if (!in_array($mimeType, $allowedTypes)) {
            throw new Exception('Invalid file type');
        }
        
        if ($file['size'] > 5242880) { // 5MB
            throw new Exception('File too large (max 5MB)');
        }
        
        // Generate filename
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $filename = sprintf(
            'hazard_%d_%d_%s.%s',
            $hazardId,
            time(),
            substr(md5(uniqid()), 0, 8),
            $extension
        );
        
        $filepath = $this->uploadPath . $filename;
        
        // Process and save image
        if (!$this->processImage($file['tmp_name'], $filepath, $mimeType)) {
            throw new Exception('Failed to process image');
        }
        
        // Create thumbnail
        $this->createThumbnail($filepath, $filename);
        
        // If this is primary, unset other primary photos
        if ($isPrimary) {
            $this->db->query("UPDATE hazard_library_photos SET is_primary = 0 WHERE risk_item_id = ?", [$hazardId]);
        }
        
        // Save to database
        $sql = "INSERT INTO hazard_library_photos (risk_item_id, filename, caption, is_primary, uploaded_by)
                VALUES (?, ?, ?, ?, ?)";
        
        $this->db->query($sql, [
            $hazardId,
            $filename,
            $caption,
            $isPrimary ? 1 : 0,
            $_SESSION['user_id'] ?? null
        ]);
        
        return [
            'id' => $this->db->lastInsertId(),
            'filename' => $filename,
            'url' => SITE_URL . 'uploads/hazard-library/' . $filename,
            'thumbnail' => SITE_URL . 'uploads/hazard-library/thumb_' . $filename
        ];
    }
    
    /**
     * Get photos for hazard
     */
    public function getHazardPhotos($hazardId) {
        $photos = $this->db->fetchAll("
            SELECT hlp.*, u.full_name as uploaded_by_name
            FROM hazard_library_photos hlp
            LEFT JOIN users u ON hlp.uploaded_by = u.id
            WHERE hlp.risk_item_id = ?
            ORDER BY hlp.is_primary DESC, hlp.display_order, hlp.id
        ", [$hazardId]);
        
        return array_map(function($photo) {
            $photo['url'] = SITE_URL . 'uploads/hazard-library/' . $photo['filename'];
            $photo['thumbnail'] = SITE_URL . 'uploads/hazard-library/thumb_' . $photo['filename'];
            return $photo;
        }, $photos);
    }
    
    /**
     * Delete photo
     */
    public function deletePhoto($photoId) {
        $photo = $this->db->fetchOne("SELECT * FROM hazard_library_photos WHERE id = ?", [$photoId]);
        
        if ($photo) {
            // Delete files
            @unlink($this->uploadPath . $photo['filename']);
            @unlink($this->uploadPath . 'thumb_' . $photo['filename']);
            
            // Delete from database
            $this->db->query("DELETE FROM hazard_library_photos WHERE id = ?", [$photoId]);
        }
    }
    
    /**
     * Add hazard to assessment
     */
    public function addToAssessment($hazardId, $assessmentId, $customData = []) {
        $hazard = $this->getHazard($hazardId);
        
        if (!$hazard) {
            throw new Exception('Hazard not found');
        }
        
        // Insert into assessment_risks
        $sql = "INSERT INTO assessment_risks (
            assessment_id, template_id, hazard_name, hazard_description,
            likelihood, severity, existing_controls
        ) VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $this->db->query($sql, [
            $assessmentId,
            $hazardId,
            $customData['hazard_name'] ?? $hazard['name'],
            $customData['hazard_description'] ?? $hazard['description'],
            $customData['likelihood'] ?? $hazard['default_likelihood'],
            $customData['severity'] ?? $hazard['default_severity'],
            $customData['existing_controls'] ?? $hazard['typical_controls']
        ]);
        
        $riskId = $this->db->lastInsertId();
        
        // Update usage count
        $this->db->query("UPDATE risk_items SET usage_count = usage_count + 1, last_used = NOW() WHERE id = ?", [$hazardId]);
        
        return $riskId;
    }
    
    /**
     * Get categories for filter
     */
    public function getCategories() {
        return $this->db->fetchAll("SELECT * FROM risk_categories WHERE active = 1 ORDER BY display_order");
    }
    
    /**
     * Get usage statistics
     */
    public function getStatistics() {
        return [
            'total_hazards' => $this->db->fetchOne("SELECT COUNT(*) as count FROM risk_items WHERE is_template = 1")['count'],
            'total_controls' => $this->db->fetchOne("SELECT COUNT(*) as count FROM control_measures")['count'],
            'total_photos' => $this->db->fetchOne("SELECT COUNT(*) as count FROM hazard_library_photos")['count'],
            'most_used' => $this->db->fetchAll("
                SELECT id, name, usage_count FROM risk_items 
                WHERE is_template = 1 
                ORDER BY usage_count DESC 
                LIMIT 5
            ")
        ];
    }
    
    /**
     * Process image
     */
    private function processImage($source, $destination, $mimeType) {
        $image = null;
        
        switch ($mimeType) {
            case 'image/jpeg':
            case 'image/jpg':
                $image = @imagecreatefromjpeg($source);
                break;
            case 'image/png':
                $image = @imagecreatefrompng($source);
                break;
            case 'image/gif':
                $image = @imagecreatefromgif($source);
                break;
            case 'image/webp':
                $image = @imagecreatefromwebp($source);
                break;
        }
        
        if (!$image) return false;
        
        $width = imagesx($image);
        $height = imagesy($image);
        
        // Resize if needed
        $maxWidth = 1600;
        $maxHeight = 1600;
        
        if ($width > $maxWidth || $height > $maxHeight) {
            $ratio = min($maxWidth / $width, $maxHeight / $height);
            $newWidth = round($width * $ratio);
            $newHeight = round($height * $ratio);
            
            $resized = imagecreatetruecolor($newWidth, $newHeight);
            
            if ($mimeType === 'image/png' || $mimeType === 'image/gif') {
                imagealphablending($resized, false);
                imagesavealpha($resized, true);
            }
            
            imagecopyresampled($resized, $image, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
            imagedestroy($image);
            $image = $resized;
        }
        
        // Save
        $result = false;
        switch ($mimeType) {
            case 'image/jpeg':
            case 'image/jpg':
                $result = imagejpeg($image, $destination, 85);
                break;
            case 'image/png':
                $result = imagepng($image, $destination, 8);
                break;
            case 'image/gif':
                $result = imagegif($image, $destination);
                break;
            case 'image/webp':
                $result = imagewebp($image, $destination, 85);
                break;
        }
        
        imagedestroy($image);
        return $result;
    }
    
    /**
     * Create thumbnail
     */
    private function createThumbnail($filepath, $filename) {
        $image = @imagecreatefromstring(file_get_contents($filepath));
        if (!$image) return false;
        
        $width = imagesx($image);
        $height = imagesy($image);
        $size = min($width, $height);
        $x = ($width - $size) / 2;
        $y = ($height - $size) / 2;
        
        $thumb = imagecreatetruecolor(300, 300);
        imagealphablending($thumb, false);
        imagesavealpha($thumb, true);
        
        imagecopyresampled($thumb, $image, 0, 0, $x, $y, 300, 300, $size, $size);
        
        $thumbPath = $this->uploadPath . 'thumb_' . $filename;
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        switch ($extension) {
            case 'jpg':
            case 'jpeg':
                imagejpeg($thumb, $thumbPath, 85);
                break;
            case 'png':
                imagepng($thumb, $thumbPath, 8);
                break;
            case 'gif':
                imagegif($thumb, $thumbPath);
                break;
            case 'webp':
                imagewebp($thumb, $thumbPath, 85);
                break;
        }
        
        imagedestroy($image);
        imagedestroy($thumb);
        
        return true;
    }
}
