<?php
/**
 * Audit Manager
 * Comprehensive audit logging and tracking system
 */

class AuditManager {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Log an audit event
     */
    public function log($action, $entityType, $entityId = null, $oldValues = null, $newValues = null, $entityName = null) {
        $userId = $_SESSION['user_id'] ?? null;
        $username = $_SESSION['username'] ?? 'System';
        $ipAddress = $this->getIpAddress();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
        $sessionId = session_id();
        
        $changesSummary = $this->generateChangesSummary($oldValues, $newValues);
        
        $sql = "INSERT INTO audit_log (
            user_id, username, action, entity_type, entity_id, entity_name,
            old_values, new_values, changes_summary, ip_address, user_agent, session_id
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $this->db->query($sql, [
            $userId,
            $username,
            $action,
            $entityType,
            $entityId,
            $entityName,
            $oldValues ? json_encode($oldValues) : null,
            $newValues ? json_encode($newValues) : null,
            $changesSummary,
            $ipAddress,
            $userAgent,
            $sessionId
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Get audit logs with filtering
     */
    public function getLogs($filters = [], $limit = 50, $offset = 0) {
        $sql = "SELECT al.*, u.full_name as user_full_name
                FROM audit_log al
                LEFT JOIN users u ON al.user_id = u.id
                WHERE 1=1";
        
        $params = [];
        
        if (!empty($filters['user_id'])) {
            $sql .= " AND al.user_id = ?";
            $params[] = $filters['user_id'];
        }
        
        if (!empty($filters['action'])) {
            $sql .= " AND al.action = ?";
            $params[] = $filters['action'];
        }
        
        if (!empty($filters['entity_type'])) {
            $sql .= " AND al.entity_type = ?";
            $params[] = $filters['entity_type'];
        }
        
        if (!empty($filters['entity_id'])) {
            $sql .= " AND al.entity_id = ?";
            $params[] = $filters['entity_id'];
        }
        
        if (!empty($filters['date_from'])) {
            $sql .= " AND al.created_at >= ?";
            $params[] = $filters['date_from'];
        }
        
        if (!empty($filters['date_to'])) {
            $sql .= " AND al.created_at <= ?";
            $params[] = $filters['date_to'] . ' 23:59:59';
        }
        
        if (!empty($filters['search'])) {
            $sql .= " AND (al.entity_name LIKE ? OR al.changes_summary LIKE ? OR al.username LIKE ?)";
            $searchTerm = "%{$filters['search']}%";
            $params[] = $searchTerm;
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        $sql .= " ORDER BY al.created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        return $this->db->fetchAll($sql, $params);
    }
    
    /**
     * Get total count of logs (for pagination)
     */
    public function getLogsCount($filters = []) {
        $sql = "SELECT COUNT(*) as total FROM audit_log WHERE 1=1";
        $params = [];
        
        if (!empty($filters['user_id'])) {
            $sql .= " AND user_id = ?";
            $params[] = $filters['user_id'];
        }
        
        if (!empty($filters['action'])) {
            $sql .= " AND action = ?";
            $params[] = $filters['action'];
        }
        
        if (!empty($filters['entity_type'])) {
            $sql .= " AND entity_type = ?";
            $params[] = $filters['entity_type'];
        }
        
        $result = $this->db->fetchOne($sql, $params);
        return $result['total'] ?? 0;
    }
    
    /**
     * Get entity audit history
     */
    public function getEntityHistory($entityType, $entityId) {
        $sql = "SELECT al.*, u.full_name as user_full_name
                FROM audit_log al
                LEFT JOIN users u ON al.user_id = u.id
                WHERE al.entity_type = ? AND al.entity_id = ?
                ORDER BY al.created_at DESC";
        
        return $this->db->fetchAll($sql, [$entityType, $entityId]);
    }
    
    /**
     * Get user activity summary
     */
    public function getUserActivitySummary($userId, $days = 30) {
        $sql = "SELECT 
                action,
                entity_type,
                COUNT(*) as count,
                MAX(created_at) as last_activity
                FROM audit_log
                WHERE user_id = ?
                AND created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY action, entity_type
                ORDER BY count DESC";
        
        return $this->db->fetchAll($sql, [$userId, $days]);
    }
    
    /**
     * Get system activity statistics
     */
    public function getActivityStatistics($days = 7) {
        return [
            'total_actions' => $this->getTotalActions($days),
            'unique_users' => $this->getUniqueUsers($days),
            'top_actions' => $this->getTopActions($days),
            'top_users' => $this->getTopUsers($days),
            'actions_by_day' => $this->getActionsByDay($days),
            'actions_by_type' => $this->getActionsByType($days)
        ];
    }
    
    /**
     * Log user activity (page views, searches, etc)
     */
    public function logActivity($activityType, $description = null, $metadata = null) {
        $userId = $_SESSION['user_id'] ?? null;
        if (!$userId) return;
        
        $sql = "INSERT INTO user_activity (
            user_id, activity_type, activity_description, page_url, metadata, ip_address
        ) VALUES (?, ?, ?, ?, ?, ?)";
        
        $this->db->query($sql, [
            $userId,
            $activityType,
            $description,
            $_SERVER['REQUEST_URI'] ?? null,
            $metadata ? json_encode($metadata) : null,
            $this->getIpAddress()
        ]);
    }
    
    /**
     * Export audit logs to CSV
     */
    public function exportToCSV($filters = []) {
        $logs = $this->getLogs($filters, 10000, 0); // Max 10k records
        
        $output = fopen('php://temp', 'r+');
        
        // Headers
        fputcsv($output, [
            'Date/Time', 'User', 'Action', 'Entity Type', 'Entity Name', 
            'Changes Summary', 'IP Address'
        ]);
        
        // Data
        foreach ($logs as $log) {
            fputcsv($output, [
                $log['created_at'],
                $log['user_full_name'] ?? $log['username'],
                $log['action'],
                $log['entity_type'],
                $log['entity_name'] ?? "ID: {$log['entity_id']}",
                $log['changes_summary'],
                $log['ip_address']
            ]);
        }
        
        rewind($output);
        $csv = stream_get_contents($output);
        fclose($output);
        
        return $csv;
    }
    
    /**
     * Clean up old audit logs based on retention policy
     */
    public function cleanupOldLogs() {
        $policy = $this->db->fetchOne(
            "SELECT retention_days FROM retention_policies WHERE entity_type = 'audit_log' AND is_active = 1"
        );
        
        if ($policy) {
            $sql = "DELETE FROM audit_log 
                    WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)";
            $this->db->query($sql, [$policy['retention_days']]);
            
            return $this->db->rowCount();
        }
        
        return 0;
    }
    
    /**
     * Generate human-readable changes summary
     */
    private function generateChangesSummary($oldValues, $newValues) {
        if (!$oldValues && !$newValues) {
            return null;
        }
        
        if (!$oldValues && $newValues) {
            return "Created new record";
        }
        
        if ($oldValues && !$newValues) {
            return "Deleted record";
        }
        
        $changes = [];
        $old = is_array($oldValues) ? $oldValues : json_decode($oldValues, true);
        $new = is_array($newValues) ? $newValues : json_decode($newValues, true);
        
        if (!$old || !$new) return null;
        
        foreach ($new as $key => $value) {
            if (!isset($old[$key]) || $old[$key] != $value) {
                $oldVal = $old[$key] ?? 'null';
                $changes[] = ucfirst(str_replace('_', ' ', $key)) . ": '{$oldVal}' → '{$value}'";
            }
        }
        
        return implode('; ', $changes);
    }
    
    /**
     * Get IP address
     */
    private function getIpAddress() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        }
    }
    
    // Statistical helper methods
    private function getTotalActions($days) {
        $result = $this->db->fetchOne(
            "SELECT COUNT(*) as total FROM audit_log 
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$days]
        );
        return $result['total'] ?? 0;
    }
    
    private function getUniqueUsers($days) {
        $result = $this->db->fetchOne(
            "SELECT COUNT(DISTINCT user_id) as total FROM audit_log 
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$days]
        );
        return $result['total'] ?? 0;
    }
    
    private function getTopActions($days, $limit = 5) {
        return $this->db->fetchAll(
            "SELECT action, COUNT(*) as count 
             FROM audit_log 
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
             GROUP BY action 
             ORDER BY count DESC 
             LIMIT ?",
            [$days, $limit]
        );
    }
    
    private function getTopUsers($days, $limit = 5) {
        return $this->db->fetchAll(
            "SELECT u.full_name, al.username, COUNT(*) as count 
             FROM audit_log al
             LEFT JOIN users u ON al.user_id = u.id
             WHERE al.created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
             GROUP BY al.user_id, al.username, u.full_name
             ORDER BY count DESC 
             LIMIT ?",
            [$days, $limit]
        );
    }
    
    private function getActionsByDay($days) {
        return $this->db->fetchAll(
            "SELECT DATE(created_at) as date, COUNT(*) as count 
             FROM audit_log 
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
             GROUP BY DATE(created_at) 
             ORDER BY date",
            [$days]
        );
    }
    
    private function getActionsByType($days) {
        return $this->db->fetchAll(
            "SELECT entity_type, COUNT(*) as count 
             FROM audit_log 
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
             GROUP BY entity_type 
             ORDER BY count DESC",
            [$days]
        );
    }
}
