# Database Schema Documentation v3.4.0

## 📊 Overview

The Risk Assessment System uses a single comprehensive SQL file that creates the entire database structure, sample data, and configuration in one operation.

**File:** `complete_schema.sql`  
**Size:** ~45KB  
**Tables:** 23  
**Indexes:** 60+  
**Sample Data:** Yes  
**Triggers:** 1  

---

## 🗂️ Database Structure

### Total Tables: 23

#### Core Tables (4)
1. **users** - User accounts and authentication
2. **locations** - Assessment locations with enhanced scheduling
3. **risk_categories** - Risk categorization (10 default categories)
4. **risk_items** - Risk library and hazard templates

#### Assessment Tables (3)
5. **assessments** - Risk assessments
6. **assessment_risks** - Individual risk records
7. **action_items** - Corrective actions
8. **assessment_versions** - Version history

#### Location Enhancement Tables (2)
9. **location_hours** - Detailed weekly schedules
10. **delivery_slots** - Specific delivery time windows

#### Hazard Library Tables (2)
11. **control_measures** - Control measure hierarchy
12. **hazard_library_photos** - Reference images

#### Audit & Activity Tables (4)
13. **audit_log** - Complete action history
14. **user_activity** - User activity tracking
15. **change_requests** - Approval workflows
16. **retention_policies** - Data retention configuration

#### Permission & Customization Tables (4)
17. **permissions** - Role-based access control
18. **custom_fields** - Extensible field definitions
19. **custom_field_values** - Custom field data
20. **system_settings** - Configuration storage

#### Support Tables (3)
21. **email_queue** - Email scheduling
22. **sessions** - Session management
23. **sync_queue** - Offline sync support

---

## 📋 Table Details

### 1. users
**Purpose:** Store user accounts and authentication

| Column | Type | Description |
|--------|------|-------------|
| id | INT | Primary key |
| username | VARCHAR(100) | Unique username |
| password | VARCHAR(255) | Hashed password (bcrypt) |
| full_name | VARCHAR(200) | User's full name |
| email | VARCHAR(200) | Email address |
| role | ENUM | admin, assessor, viewer |
| active | TINYINT(1) | Active status |
| last_login | DATETIME | Last login timestamp |

**Indexes:** username, email, role, active

---

### 2. locations (Enhanced)
**Purpose:** Store location information with opening hours and delivery windows

| Column | Type | Description |
|--------|------|-------------|
| id | INT | Primary key |
| name | VARCHAR(200) | Location name |
| chain | VARCHAR(200) | Chain/group name |
| address | TEXT | Full address |
| city | VARCHAR(100) | City |
| postcode | VARCHAR(20) | Postal code |
| opening_hours | JSON | Hours by day of week |
| delivery_windows | JSON | Delivery slots by day |
| contact_name | VARCHAR(200) | Site contact |
| contact_phone | VARCHAR(50) | Contact phone |
| contact_email | VARCHAR(200) | Contact email |
| parking_info | TEXT | Parking instructions |
| delivery_restrictions | TEXT | Vehicle restrictions |

**JSON Format - opening_hours:**
```json
{
  "monday": {"open": "08:00", "close": "18:00", "closed": false},
  "tuesday": {"open": "08:00", "close": "18:00", "closed": false},
  ...
}
```

**JSON Format - delivery_windows:**
```json
{
  "monday": [
    {"start": "08:00", "end": "12:00", "name": "Morning"},
    {"start": "13:00", "end": "17:00", "name": "Afternoon"}
  ],
  ...
}
```

---

### 3. assessments
**Purpose:** Store risk assessments

| Column | Type | Description |
|--------|------|-------------|
| id | INT | Primary key |
| assessment_number | VARCHAR(50) | Unique number (RA-YYYY-NNNNNN) |
| title | VARCHAR(200) | Assessment title |
| location_id | INT | FK to locations |
| visit_date | DATE | Assessment date |
| assessor_id | INT | FK to users |
| status | ENUM | draft, in_progress, completed, reviewed, archived |
| overall_risk_level | VARCHAR(50) | Overall risk rating |

**Automatic Fields:**
- `created_at` - Auto-generated
- `updated_at` - Auto-updated
- `created_by` - User who created
- `updated_by` - Last user to update

---

### 4. assessment_risks
**Purpose:** Individual risk items within assessments

| Column | Type | Description |
|--------|------|-------------|
| id | INT | Primary key |
| assessment_id | INT | FK to assessments |
| template_id | INT | FK to risk_items (if from library) |
| hazard_description | TEXT | Description of hazard |
| likelihood | INT | 1-5 scale |
| severity | INT | 1-5 scale |
| risk_score | INT | Calculated: likelihood × severity |
| risk_level | VARCHAR(50) | Calculated: critical/high/medium/low |

**Calculated Fields:**
- `risk_score` = likelihood × severity
- `risk_level` = Based on score (≥20: critical, ≥15: high, ≥10: medium, <10: low)

---

### 5. audit_log
**Purpose:** Track all user actions

| Column | Type | Description |
|--------|------|-------------|
| id | INT | Primary key |
| user_id | INT | FK to users |
| action | VARCHAR(50) | create, update, delete, login, etc. |
| entity_type | VARCHAR(50) | assessment, location, user, etc. |
| entity_id | INT | ID of affected entity |
| old_values | JSON | Previous state |
| new_values | JSON | New state |
| changes_summary | TEXT | Human-readable summary |
| ip_address | VARCHAR(45) | User's IP |
| session_id | VARCHAR(100) | Session identifier |

**Retention:** 730 days (2 years) by default

---

### 6. assessment_versions
**Purpose:** Store version snapshots of assessments

| Column | Type | Description |
|--------|------|-------------|
| id | INT | Primary key |
| assessment_id | INT | FK to assessments |
| version_number | INT | Sequential version number |
| version_type | VARCHAR(50) | auto, manual, scheduled |
| version_notes | TEXT | User notes |
| snapshot_data | JSON | Complete assessment state |

**Trigger:** Auto-creates version when status changes

---

### 7. permissions
**Purpose:** Role-based access control

| Column | Type | Description |
|--------|------|-------------|
| id | INT | Primary key |
| role | ENUM | admin, assessor, viewer |
| resource | VARCHAR(50) | assessments, locations, etc. |
| can_read | TINYINT(1) | Read permission |
| can_create | TINYINT(1) | Create permission |
| can_update | TINYINT(1) | Update permission |
| can_delete | TINYINT(1) | Delete permission |

**Default Permissions:**
- Admin: Full access (CRUD on all resources)
- Assessor: Create/Read/Update on assessments, locations, hazards
- Viewer: Read-only on assessments, locations

---

## 🔧 Triggers

### after_assessment_update
**Purpose:** Automatically create version snapshots

**When:** After UPDATE on assessments table

**Condition:** Status changes OR title changes OR location changes

**Action:** Creates new version in assessment_versions with:
- Incremented version_number
- Type: 'auto'
- Notes: "Auto-saved: Status changed from X to Y"
- Complete snapshot in JSON format

---

## 📊 Indexes

### Primary Indexes (23)
Every table has a primary key on `id`

### Foreign Key Indexes (30+)
All foreign key relationships are indexed

### Performance Indexes (10+)
- `idx_assessment_location_date` - Assessment queries by location and date
- `idx_assessment_status_date` - Status-based filtering
- `idx_audit_user_date` - Audit log filtering
- `idx_action_items_status_date` - Action item queries

---

## 🎯 Default Data

### Users (1)
- **Username:** admin
- **Password:** admin123 (hashed)
- **Role:** admin
- **Email:** admin@example.com

⚠️ Change this password immediately after installation!

### Risk Categories (10)
1. Vehicle Operations
2. Loading/Unloading
3. Access & Egress
4. Environmental
5. People & Pedestrians
6. Equipment & Tools
7. Manual Handling
8. Site Specific
9. Traffic Management
10. Other Hazards

### Hazard Templates (15+)
Pre-loaded HGV delivery hazards:
- Reversing Collisions
- Vehicle Rollover
- Blind Spot Incidents
- Manual Handling Injuries
- Falls from Height
- Crushing Injuries
- Falling Objects
- Slips and Trips
- Unsafe Access Points
- Door Strike Injuries
- Poor Visibility
- Ice and Snow Hazards
- Extreme Weather
- Vehicle-Pedestrian Collision
- Contractor Coordination
- Lone Working

Each includes:
- Complete description (100+ words)
- Default likelihood and severity ratings
- Typical control measures
- Affected persons
- Residual risk level

### Permissions (24)
8 resources × 3 roles = 24 permission records
- assessments, locations, users, hazards
- audit, settings, permissions, reports

### Retention Policies (4)
- Audit logs: 730 days
- Assessment versions: 1095 days
- User activity: 365 days
- Assessments: 1825 days

### System Settings (6)
- site_name, items_per_page, enable_email_notifications
- assessment_number_prefix, date_format, timezone

---

## 🔄 Migration from Multiple Files

### Previous Structure (v3.3.0 and earlier)
Required importing 5 separate files:
1. schema.sql
2. schema_additions.sql
3. hazard_library_schema.sql
4. audit_version_schema.sql
5. location_enhancements_schema.sql

### New Structure (v3.4.0+)
**Single file:** complete_schema.sql

**Benefits:**
- ✅ One command installation
- ✅ No risk of missing files
- ✅ Guaranteed correct order
- ✅ Faster setup
- ✅ Easier to maintain
- ✅ Better version control

### Backward Compatibility
Old schema files are preserved in `database/backup_schemas/` for reference but should not be used for new installations.

---

## 📝 Maintenance Queries

### Check Table Sizes
```sql
SELECT 
    table_name,
    ROUND(((data_length + index_length) / 1024 / 1024), 2) AS "Size (MB)"
FROM information_schema.TABLES 
WHERE table_schema = 'risk_assessment_db'
ORDER BY (data_length + index_length) DESC;
```

### Count Records by Table
```sql
SELECT 'users' as table_name, COUNT(*) as count FROM users
UNION ALL SELECT 'locations', COUNT(*) FROM locations
UNION ALL SELECT 'assessments', COUNT(*) FROM assessments
UNION ALL SELECT 'audit_log', COUNT(*) FROM audit_log
UNION ALL SELECT 'assessment_versions', COUNT(*) FROM assessment_versions;
```

### Find Orphaned Records
```sql
-- Assessment risks without assessments
SELECT COUNT(*) FROM assessment_risks ar
LEFT JOIN assessments a ON ar.assessment_id = a.id
WHERE a.id IS NULL;

-- Versions without assessments
SELECT COUNT(*) FROM assessment_versions av
LEFT JOIN assessments a ON av.assessment_id = a.id
WHERE a.id IS NULL;
```

### Cleanup Old Audit Logs
```sql
DELETE FROM audit_log 
WHERE created_at < DATE_SUB(NOW(), INTERVAL 730 DAY);
```

---

## 🔐 Security Notes

### Password Storage
- Uses PHP's `password_hash()` with bcrypt
- Default cost factor: 10
- Never store plain text passwords

### SQL Injection Protection
- All queries use PDO prepared statements
- Never concatenate user input into SQL

### Session Security
- Sessions stored in database (sessions table)
- HttpOnly cookies enabled
- Secure flag for HTTPS connections

---

## 📈 Performance Tuning

### Recommended MySQL Configuration
```ini
innodb_buffer_pool_size = 256M
query_cache_size = 64M
query_cache_limit = 2M
max_connections = 100
```

### Index Maintenance
```sql
-- Analyze tables monthly
ANALYZE TABLE assessments, assessment_risks, audit_log;

-- Optimize tables quarterly
OPTIMIZE TABLE audit_log, user_activity;
```

### Backup Recommendation
```bash
# Daily backup
mysqldump -u root -p risk_assessment_db > backup_$(date +%Y%m%d).sql

# With compression
mysqldump -u root -p risk_assessment_db | gzip > backup_$(date +%Y%m%d).sql.gz
```

---

## ✅ Verification

### After Import, Verify:
```sql
-- Check table count
SELECT COUNT(*) as table_count 
FROM information_schema.tables 
WHERE table_schema = 'risk_assessment_db';
-- Should return: 23

-- Check admin user exists
SELECT username, role FROM users WHERE username = 'admin';
-- Should return: admin, admin role

-- Check hazard templates
SELECT COUNT(*) as hazard_count 
FROM risk_items 
WHERE is_template = 1;
-- Should return: 15+

-- Check permissions
SELECT COUNT(*) as permission_count FROM permissions;
-- Should return: 24
```

---

**Schema Version:** 3.4.0  
**Last Updated:** February 2026  
**Status:** Production Ready

For installation instructions, see QUICK_INSTALLATION.md
