#!/bin/bash

echo "========================================"
echo "COMPREHENSIVE HTML/LINK ERROR CHECKER"
echo "========================================"
echo ""

ERRORS=0

# 1. Check for missing closing tags
echo "1. Checking for mismatched <a> tags..."
for file in public/*.php includes/*.php; do
    if [ -f "$file" ]; then
        open_tags=$(grep -o "<a " "$file" | wc -l)
        close_tags=$(grep -o "</a>" "$file" | wc -l)
        if [ $open_tags -ne $close_tags ]; then
            echo "  ⚠️  $(basename $file): $open_tags opening tags, $close_tags closing tags"
            ERRORS=$((ERRORS+1))
        fi
    fi
done

echo ""
echo "2. Checking for empty href attributes..."
grep -n 'href=""' --include="*.php" -r public/ includes/ | while read line; do
    echo "  ⚠️  $line"
    ERRORS=$((ERRORS+1))
done

echo ""
echo "3. Checking for broken form actions..."
grep -n 'action=""' --include="*.php" -r public/ | while read line; do
    echo "  ⚠️  $line"
    ERRORS=$((ERRORS+1))
done

echo ""
echo "4. Checking for unclosed divs in main pages..."
for file in public/dashboard.php public/assessments.php public/settings.php; do
    if [ -f "$file" ]; then
        open_divs=$(grep -o "<div" "$file" | wc -l)
        close_divs=$(grep -o "</div>" "$file" | wc -l)
        if [ $open_divs -ne $close_divs ]; then
            echo "  ⚠️  $(basename $file): $open_divs opening divs, $close_divs closing divs"
        fi
    fi
done

echo ""
echo "5. Checking for proper Auth includes..."
for file in public/*.php; do
    if [ -f "$file" ] && [ "$(basename $file)" != "login.php" ]; then
        if ! grep -q "requireLogin\|requireRole" "$file"; then
            basename $file | grep -v "^api-\|^test-\|^diagnose\|^show-errors" > /dev/null
            if [ $? -eq 0 ]; then
                echo "  ⚠️  $(basename $file): Missing authentication check"
            fi
        fi
    fi
done

echo ""
echo "6. Checking for duplicate IDs in forms..."
for file in public/*.php; do
    if [ -f "$file" ]; then
        ids=$(grep -o 'id="[^"]*"' "$file" | cut -d'"' -f2 | sort)
        duplicates=$(echo "$ids" | uniq -d)
        if [ -n "$duplicates" ]; then
            echo "  ⚠️  $(basename $file): Duplicate IDs: $duplicates"
        fi
    fi
done

echo ""
echo "7. Checking navigation consistency across main pages..."
for file in public/dashboard.php public/assessments.php public/locations.php; do
    if [ -f "$file" ]; then
        if ! grep -q "include.*header.php" "$file"; then
            echo "  ⚠️  $(basename $file): Missing header include"
            ERRORS=$((ERRORS+1))
        fi
        if ! grep -q "include.*footer.php" "$file"; then
            echo "  ⚠️  $(basename $file): Missing footer include"
            ERRORS=$((ERRORS+1))
        fi
    fi
done

echo ""
echo "========================================"
if [ $ERRORS -eq 0 ]; then
    echo "✅ NO CRITICAL ERRORS FOUND"
else
    echo "⚠️  FOUND $ERRORS POTENTIAL ISSUES"
fi
echo "========================================"

