<?php
/**
 * System Checker - Comprehensive Testing Tool
 * Tests all endpoints, permissions, and database connectivity
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config/config.php';
require_once 'includes/Database.php';
require_once 'includes/Auth.php';

echo "<!DOCTYPE html><html><head><title>System Checker</title>";
echo "<style>
body { font-family: monospace; padding: 20px; background: #f5f5f5; }
.test { padding: 10px; margin: 5px 0; border-radius: 5px; }
.pass { background: #d1fae5; border-left: 4px solid #10b981; }
.fail { background: #fee2e2; border-left: 4px solid #ef4444; }
.warn { background: #fef3c7; border-left: 4px solid #f59e0b; }
.info { background: #dbeafe; border-left: 4px solid #3b82f6; }
h2 { color: #1f2937; margin-top: 30px; }
.summary { padding: 20px; background: white; border-radius: 10px; margin: 20px 0; }
.count { font-size: 24px; font-weight: bold; margin: 10px 0; }
</style></head><body>";

echo "<h1>🔍 System Checker v3.3.0</h1>";
echo "<p>Testing all components, endpoints, and permissions...</p>";

$results = [
    'pass' => 0,
    'fail' => 0,
    'warn' => 0,
    'total' => 0
];

function test($name, $result, $message = '') {
    global $results;
    $results['total']++;
    
    if ($result === true) {
        $results['pass']++;
        echo "<div class='test pass'>✅ PASS: $name $message</div>";
    } elseif ($result === 'warn') {
        $results['warn']++;
        echo "<div class='test warn'>⚠️  WARN: $name $message</div>";
    } else {
        $results['fail']++;
        echo "<div class='test fail'>❌ FAIL: $name $message</div>";
    }
}

// Test 1: Database Connection
echo "<h2>1. Database Tests</h2>";
try {
    $db = Database::getInstance();
    test('Database Connection', true, '- Connected successfully');
} catch (Exception $e) {
    test('Database Connection', false, '- ' . $e->getMessage());
    die('Cannot continue without database');
}

// Test 2: Required Tables
$requiredTables = [
    'users', 'locations', 'assessments', 'risk_categories', 'risk_items',
    'assessment_risks', 'action_items', 'permissions', 'audit_log',
    'assessment_versions', 'user_activity', 'hazard_library_photos'
];

foreach ($requiredTables as $table) {
    try {
        $result = $db->query("SELECT 1 FROM $table LIMIT 1");
        test("Table: $table", true, '- Exists and accessible');
    } catch (Exception $e) {
        test("Table: $table", false, '- ' . $e->getMessage());
    }
}

// Test 3: Essential Files
echo "<h2>2. File Structure Tests</h2>";
$essentialFiles = [
    'config/config.php' => 'Configuration',
    'includes/Database.php' => 'Database class',
    'includes/Auth.php' => 'Authentication class',
    'includes/functions.php' => 'Helper functions',
    'includes/AuditManager.php' => 'Audit Manager',
    'includes/VersionControl.php' => 'Version Control',
    'includes/HazardLibrary.php' => 'Hazard Library',
    'public/index.php' => 'Entry point',
    'public/login.php' => 'Login page',
    'public/dashboard.php' => 'Dashboard',
    'public/assets/css/style.css' => 'Main stylesheet',
    'public/assets/js/main.js' => 'Main JavaScript'
];

foreach ($essentialFiles as $file => $desc) {
    $exists = file_exists($file);
    test("File: $desc", $exists, $exists ? "- Found at $file" : "- Missing: $file");
}

// Test 4: Public Pages
echo "<h2>3. Public Page Tests</h2>";
$publicPages = [
    'index.php', 'login.php', 'dashboard.php', 'assessments.php', 
    'assessment-create.php', 'assessment-view.php', 'assessment-edit.php',
    'locations.php', 'location-create.php', 'location-edit.php', 'location-view.php',
    'hazard-library.php', 'hazard-create.php', 'hazard-view.php', 'hazard-edit.php',
    'users.php', 'user-create.php', 'user-edit.php',
    'settings.php', 'themes.php', 'permissions.php', 'pdf-templates.php',
    'audit-logs.php', 'assessment-versions.php', 'version-compare.php', 'user-activity.php',
    'logout.php'
];

foreach ($publicPages as $page) {
    $path = "public/$page";
    if (file_exists($path)) {
        // Check for syntax errors
        $output = [];
        $return = 0;
        exec("php -l $path 2>&1", $output, $return);
        if ($return === 0) {
            test("Page: $page", true, '- Syntax OK');
        } else {
            test("Page: $page", false, '- Syntax error: ' . implode(' ', $output));
        }
    } else {
        test("Page: $page", false, "- File not found");
    }
}

// Test 5: API Endpoints
echo "<h2>4. API Endpoint Tests</h2>";
$apiEndpoints = [
    'upload-hazard-photo.php',
    'delete-hazard-photo.php',
    'add-hazard-to-assessment.php',
    'search-hazards.php',
    'delete-hazard.php',
    'auto-audit.php'
];

foreach ($apiEndpoints as $endpoint) {
    $path = "public/api/$endpoint";
    $exists = file_exists($path);
    test("API: $endpoint", $exists ? 'warn' : false, 
         $exists ? '- Exists (no runtime test)' : '- Not found');
}

// Test 6: Permissions Setup
echo "<h2>5. Permission Tests</h2>";
try {
    $permissions = $db->fetchAll("SELECT DISTINCT role, resource FROM permissions ORDER BY role, resource");
    test('Permissions Table', !empty($permissions), '- ' . count($permissions) . ' permission rules found');
    
    $roles = $db->fetchAll("SELECT DISTINCT role FROM permissions");
    foreach ($roles as $role) {
        $count = $db->fetchOne("SELECT COUNT(*) as count FROM permissions WHERE role = ?", [$role['role']]);
        test("Role: " . $role['role'], true, "- {$count['count']} permissions");
    }
} catch (Exception $e) {
    test('Permissions System', false, '- ' . $e->getMessage());
}

// Test 7: Default Admin User
echo "<h2>6. User Tests</h2>";
try {
    $admin = $db->fetchOne("SELECT * FROM users WHERE username = 'admin'");
    test('Default Admin User', !empty($admin), $admin ? '- Exists' : '- Not found (should create)');
    
    $userCount = $db->fetchOne("SELECT COUNT(*) as count FROM users");
    test('User Count', true, "- {$userCount['count']} users in system");
} catch (Exception $e) {
    test('User System', false, '- ' . $e->getMessage());
}

// Test 8: Location Enhancements
echo "<h2>7. Location Enhancement Tests</h2>";
try {
    $columns = $db->fetchAll("SHOW COLUMNS FROM locations");
    $columnNames = array_column($columns, 'Field');
    
    $requiredColumns = ['opening_hours', 'delivery_windows', 'contact_name', 'contact_phone', 'parking_info'];
    foreach ($requiredColumns as $col) {
        $exists = in_array($col, $columnNames);
        test("Location column: $col", $exists, $exists ? '- Exists' : '- Missing (run location_enhancements_schema.sql)');
    }
} catch (Exception $e) {
    test('Location Schema', false, '- ' . $e->getMessage());
}

// Test 9: Sample Data
echo "<h2>8. Data Validation Tests</h2>";
try {
    $assessmentCount = $db->fetchOne("SELECT COUNT(*) as count FROM assessments");
    test('Assessments', 'warn', "- {$assessmentCount['count']} assessments");
    
    $locationCount = $db->fetchOne("SELECT COUNT(*) as count FROM locations");
    test('Locations', 'warn', "- {$locationCount['count']} locations");
    
    $hazardCount = $db->fetchOne("SELECT COUNT(*) as count FROM risk_items WHERE is_template = 1");
    test('Hazard Templates', !empty($hazardCount['count']), "- {$hazardCount['count']} templates");
} catch (Exception $e) {
    test('Data Check', false, '- ' . $e->getMessage());
}

// Test 10: Audit System
echo "<h2>9. Audit System Tests</h2>";
try {
    $auditCount = $db->fetchOne("SELECT COUNT(*) as count FROM audit_log");
    test('Audit Logs', 'warn', "- {$auditCount['count']} entries");
    
    $versionCount = $db->fetchOne("SELECT COUNT(*) as count FROM assessment_versions");
    test('Assessment Versions', 'warn', "- {$versionCount['count']} versions");
} catch (Exception $e) {
    test('Audit System', false, '- ' . $e->getMessage());
}

// Test 11: File Permissions
echo "<h2>10. File Permission Tests</h2>";
$writeableDirs = ['public/uploads/assessments', 'public/uploads/hazard-library', 'logs'];
foreach ($writeableDirs as $dir) {
    $writable = is_writable($dir);
    test("Directory writable: $dir", $writable, $writable ? '- OK' : '- Not writable (chmod 755)');
}

// Test 12: Configuration
echo "<h2>11. Configuration Tests</h2>";
test('DB_HOST defined', defined('DB_HOST'), defined('DB_HOST') ? '- ' . DB_HOST : '- Not defined');
test('DB_NAME defined', defined('DB_NAME'), defined('DB_NAME') ? '- ' . DB_NAME : '- Not defined');
test('BASE_URL defined', defined('BASE_URL'), defined('BASE_URL') ? '- ' . BASE_URL : '- Not defined');
test('SITE_URL defined', defined('SITE_URL'), defined('SITE_URL') ? '- ' . SITE_URL : '- Not defined');

// Summary
echo "<div class='summary'>";
echo "<h2>📊 Test Summary</h2>";
echo "<div class='count'>Total Tests: {$results['total']}</div>";
echo "<div class='count' style='color: #10b981;'>✅ Passed: {$results['pass']}</div>";
echo "<div class='count' style='color: #f59e0b;'>⚠️  Warnings: {$results['warn']}</div>";
echo "<div class='count' style='color: #ef4444;'>❌ Failed: {$results['fail']}</div>";

$passRate = round(($results['pass'] / $results['total']) * 100, 1);
echo "<div class='count'>Pass Rate: {$passRate}%</div>";

if ($results['fail'] == 0) {
    echo "<h3 style='color: #10b981;'>✅ All critical tests passed!</h3>";
} else {
    echo "<h3 style='color: #ef4444;'>❌ {$results['fail']} test(s) failed - review above</h3>";
}

if ($results['warn'] > 0) {
    echo "<p style='color: #f59e0b;'>⚠️ {$results['warn']} warning(s) - check configuration</p>";
}
echo "</div>";

echo "<hr><p><small>System Checker v3.3.0 | " . date('Y-m-d H:i:s') . "</small></p>";
echo "</body></html>";
