# Risk Assessment System v2.0.0 (PHP Edition)

A comprehensive, production-ready Risk Assessment Management System built with **pure PHP and MySQL** (no frameworks, no React). Features include complete CRUD operations, role-based access control, dark mode, and a modern responsive design.

## ðŸŒŸ Features

### Core Features
- âœ… Complete Assessment Management (Create, Read, Update, Delete)
- âœ… **Version Control** - Track changes, compare, and restore
- âœ… **Assessment Locking** - Prevent accidental edits
- âœ… **Error Reporting & Monitoring** - Comprehensive error tracking
- âœ… **Automated Email Notifications** for new assessments
- âœ… **Configurable Email List** for notifications
- âœ… **Offline Mode with PWA** - Work without internet
- âœ… **Auto-Sync** when connection returns
- âœ… **PDF Export** - Professional assessment reports
- âœ… **Location Coordinates** - Latitude/Longitude tracking
- âœ… **what3words Integration** - 3-word location addresses
- âœ… **Google Maps Integration** - Interactive maps
- âœ… Location Management with mapping
- âœ… User Management with 3 Roles (Admin, Assessor, Viewer)
- âœ… Risk Item Library (10 categories, 30+ items)
- âœ… Action Items Tracking
- âœ… Audit Logging
- âœ… Assessment Numbering (RA-YYYY-NNNNNN)
- âœ… Risk Level Calculation (Automatic)

### UI/UX Features
- âœ… Modern, Professional Design
- âœ… Dark/Light Theme Toggle
- âœ… Fully Responsive (Mobile, Tablet, Desktop)
- âœ… Clean, Intuitive Interface
- âœ… Flash Messages for User Feedback
- âœ… Pagination Support

### Security Features
- âœ… Password Hashing (bcrypt)
- âœ… Session Management
- âœ… SQL Injection Protection (Prepared Statements)
- âœ… XSS Protection (Output Sanitization)
- âœ… Role-Based Access Control
- âœ… Permission Checking
- âœ… Audit Trail

### Email Features
- âœ… **Automated Notifications** when assessments are created
- âœ… **Configurable Email List** (comma-separated)
- âœ… **Beautiful HTML Email Templates**
- âœ… **Dual Notifications**: To notification list + assigned assessor
- âœ… **Email Configuration Test** tool
- âœ… **Email Queue System** for reliability
- âœ… Enable/disable notifications per setting

### Offline & PWA Features
- âœ… **Progressive Web App** - Installable on desktop and mobile
- âœ… **Service Worker Caching** - Files cached for offline use
- âœ… **IndexedDB Storage** - Local data persistence
- âœ… **Create Offline** - Create assessments without internet
- âœ… **Auto-Sync** - Automatic sync when connection returns
- âœ… **Manual Sync Button** - Force sync anytime
- âœ… **Sync Status Indicators** - Visual feedback
- âœ… **Pending Sync Badge** - Shows items waiting to sync

### PDF Export Features
- âœ… **One-Click Export** - Generate PDF from any assessment
- âœ… **Professional Layout** - Print-ready formatting
- âœ… **Complete Details** - All assessment data included
- âœ… **Color-Coded Risks** - Visual risk level indicators
- âœ… **Action Items** - Tasks and due dates included
- âœ… **Auto-Generated** - Date and assessment number
- âœ… **Browser Print** - Save or print directly

### Version Control Features
- âœ… **Automatic Versioning** - On status changes and completion
- âœ… **Manual Versions** - Create anytime with summary
- âœ… **Version History** - Complete timeline of changes
- âœ… **Compare Versions** - See what changed between versions
- âœ… **Restore Versions** - Rollback to previous state
- âœ… **Assessment Locking** - Prevent editing when needed
- âœ… **JSON Snapshots** - Full data preservation
- âœ… **Change Attribution** - Track who changed what

### Enhanced Location Features
- âœ… **Latitude/Longitude** - Precise coordinate tracking
- âœ… **what3words** - 3-word location addresses
- âœ… **Google Maps Integration** - Interactive map preview
- âœ… **Map URLs** - One-click navigation
- âœ… **Distance Calculation** - Find nearby locations
- âœ… **Geocoding** - Address to coordinates
- âœ… **Map Embed** - Embedded maps in locations
- âœ… **Coordinate Validation** - Ensure accuracy

### Error Reporting & Monitoring
- âœ… **Automatic Error Logging** - All errors captured
- âœ… **Error Levels** - DEBUG, INFO, WARNING, ERROR, CRITICAL
- âœ… **Context Capture** - User, URL, IP, stack traces
- âœ… **Web Dashboard** - View and manage errors
- âœ… **Search & Filter** - Find specific errors
- âœ… **Email Alerts** - Critical errors notify admin
- âœ… **Export Logs** - CSV and JSON formats
- âœ… **Recurring Detection** - Identify repeated errors
- âœ… **Mark Resolved** - Track resolution status
- âœ… **Automatic Cleanup** - Remove old logs
- âœ… **Top Errors View** - Most common issues
- âœ… **Statistics Dashboard** - Real-time metrics

### Technical Features
- âœ… Pure PHP (No frameworks required)
- âœ… PDO Database Layer
- âœ… MVC-Style Architecture
- âœ… Reusable Components
- âœ… Clean Code Structure
- âœ… Comprehensive Error Handling

## ðŸ“‹ Requirements

- **PHP**: 8.0 or higher
- **MySQL**: 8.0 or higher
- **Web Server**: Apache or Nginx
- **Extensions**: PDO, PDO_MySQL

## ðŸš€ Installation

### Step 1: Extract Files
```bash
# Extract to your web server directory
unzip risk-assessment-system-php.zip
cd risk-assessment-system-php
```

### Step 2: Database Setup
```bash
# Create database and import schema
mysql -u root -p < database/schema.sql

# Or use phpMyAdmin:
# 1. Create database: risk_assessment_db
# 2. Import database/schema.sql
```

### Step 3: Configure Application
```bash
# Edit config/config.php
nano config/config.php

# Update these settings:
define('DB_HOST', 'localhost');
define('DB_NAME', 'risk_assessment_db');
define('DB_USER', 'your_username');
define('DB_PASS', 'your_password');
define('BASE_URL', 'http://your-domain.com');
```

### Step 4: Set Permissions
```bash
# Make uploads and logs directories writable
chmod 755 uploads/
chmod 755 logs/

# For Apache
chown -R www-data:www-data .

# For Nginx
chown -R nginx:nginx .
```

### Step 5: Configure Web Server

#### Apache (.htaccess included)
```bash
# Enable mod_rewrite
sudo a2enmod rewrite
sudo service apache2 restart

# The .htaccess file is already included
```

#### Nginx
Add to your nginx configuration:
```nginx
root /var/www/html/risk-assessment-system-php/public;
index index.php index.html;

location / {
    try_files $uri $uri/ /index.php?$query_string;
}

location ~ \.php$ {
    include fastcgi_params;
    fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
    fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
}
```

### Step 6: Access the System
Open your browser and navigate to:
```
http://your-domain.com
```

**Default Login:**
- Username: `admin`
- Password: `admin123`

âš ï¸ **IMPORTANT: Change the admin password immediately after first login!**

## ðŸ“ Project Structure

```
risk-assessment-system-php/
â”œâ”€â”€ config/
â”‚   â””â”€â”€ config.php              # Configuration settings
â”œâ”€â”€ includes/
â”‚   â”œâ”€â”€ Database.php            # Database connection
â”‚   â”œâ”€â”€ Auth.php                # Authentication
â”‚   â”œâ”€â”€ functions.php           # Helper functions
â”‚   â”œâ”€â”€ header.php              # Page header template
â”‚   â””â”€â”€ footer.php              # Page footer template
â”œâ”€â”€ public/                     # Web root (point your server here)
â”‚   â”œâ”€â”€ index.php              # Entry point
â”‚   â”œâ”€â”€ login.php              # Login page
â”‚   â”œâ”€â”€ dashboard.php          # Dashboard
â”‚   â”œâ”€â”€ assessments.php        # Assessments list
â”‚   â”œâ”€â”€ assessment-create.php  # Create assessment
â”‚   â”œâ”€â”€ assessment-view.php    # View assessment
â”‚   â”œâ”€â”€ assessment-edit.php    # Edit assessment
â”‚   â”œâ”€â”€ locations.php          # Locations management
â”‚   â”œâ”€â”€ users.php              # User management (admin only)
â”‚   â”œâ”€â”€ settings.php           # System settings (admin only)
â”‚   â””â”€â”€ logout.php             # Logout
â”œâ”€â”€ assets/
â”‚   â”œâ”€â”€ css/
â”‚   â”‚   â””â”€â”€ style.css          # Main stylesheet (800+ lines)
â”‚   â”œâ”€â”€ js/                    # JavaScript files
â”‚   â””â”€â”€ images/                # Images
â”œâ”€â”€ uploads/                   # File uploads directory
â”œâ”€â”€ logs/                      # Application logs
â”œâ”€â”€ database/
â”‚   â””â”€â”€ schema.sql             # Database schema (16 tables)
â”œâ”€â”€ .htaccess                  # Apache configuration
â””â”€â”€ README.md                  # This file
```

## ðŸŽ¯ Usage Guide

### Configuring Email Notifications

1. **Go to Settings** (Admin only)
2. **Enable Email Notifications** checkbox
3. **Add Email Addresses** in the notification list field
   - Enter emails separated by commas
   - Example: `manager@company.com, safety@company.com, admin@company.com`
4. **Save Settings**
5. **Test Email Configuration** 
   - Enter your email in the test field
   - Click "Send Test Email"
   - Check your inbox (and spam folder)

**What Happens When Assessment is Created:**
- **Notification List** receives: Assessment details with link to view
- **Assigned Assessor** receives: Assignment notification with next steps
- Both emails include: Assessment number, title, location, date, status, and description

**Email Configuration:**
- Edit `config/config.php` to set:
  ```php
  define('SMTP_FROM', 'noreply@yourdomain.com');
  define('SMTP_FROM_NAME', 'Risk Assessment System');
  ```
- System uses PHP's `mail()` function
- For production, configure your server's mail system or use SMTP service

### Creating an Assessment
1. Navigate to **Assessments** â†’ **+ New Assessment**
2. Select location
3. Enter assessment details
4. Click **Create Assessment**
5. Add risk items on the assessment page
6. Assign likelihood and severity
7. Add control measures

### User Roles

**Admin**
- Full system access
- User management
- Settings configuration
- All CRUD operations

**Assessor**
- Create and edit assessments
- View locations and users
- Manage action items

**Viewer**
- Read-only access
- View assessments and reports
- No editing capabilities

### Managing Locations
1. Go to **Locations**
2. Click **+ New Location**
3. Enter location details
4. Save

### Theme Toggle
- Click the moon/sun icon in the header
- Theme preference is saved in browser
- Automatically applies on next visit

## ðŸ—„ï¸ Database Schema

### Core Tables (16 total)
1. **users** - User accounts and authentication
2. **locations** - Assessment locations
3. **risk_categories** - Risk categorization (10 default)
4. **risk_items** - Risk library (30+ default items)
5. **assessments** - Risk assessments
6. **assessment_risks** - Individual risk records
7. **action_items** - Corrective actions
8. **permissions** - Role-based permissions
9. **custom_fields** - Extensible fields
10. **custom_field_values** - Custom data
11. **system_settings** - Configuration
12. **audit_log** - Activity tracking
13. **email_queue** - Email scheduling
14. **file_versions** - File management
15. **sessions** - Session tracking
16. **sync_queue** - Offline sync support

## ðŸ”’ Security Best Practices

### Implemented Security
- Password hashing with PHP's password_hash()
- Prepared statements for all SQL queries
- Output sanitization (htmlspecialchars)
- Session security (httponly, secure cookies)
- Role-based access control
- Permission checks on every action
- Audit logging

### Production Checklist
- [ ] Change default admin password
- [ ] Set display_errors = 0 in php.ini
- [ ] Configure HTTPS/SSL
- [ ] Update BASE_URL in config.php
- [ ] Set proper file permissions (755 for directories, 644 for files)
- [ ] Configure database backups
- [ ] Review and update SESSION_LIFETIME
- [ ] Secure uploads directory
- [ ] Enable error logging

## ðŸŽ¨ Customization

### Changing Colors
Edit `assets/css/style.css`:
```css
:root {
    --primary: #2563eb;     /* Main color */
    --success: #10b981;     /* Success color */
    --danger: #ef4444;      /* Error color */
    /* ... more variables */
}
```

### Adding Custom Pages
1. Create new PHP file in `public/`
2. Include header.php at the top
3. Add your content
4. Include footer.php at the bottom
5. Add navigation link in header.php

### Modifying Permissions
Update the `permissions` table in MySQL:
```sql
UPDATE permissions 
SET can_create = 1 
WHERE role = 'assessor' AND resource = 'locations';
```

## ðŸ› Troubleshooting

### Database Connection Error
```bash
# Check MySQL is running
sudo service mysql status

# Test connection
mysql -u your_username -p

# Verify credentials in config/config.php
```

### Permission Denied Errors
```bash
# Fix file permissions
sudo chmod -R 755 .
sudo chmod 755 uploads/
sudo chmod 755 logs/

# Fix ownership (Apache)
sudo chown -R www-data:www-data .

# Fix ownership (Nginx)
sudo chown -R nginx:nginx .
```

### Blank Page / White Screen
```bash
# Enable error display temporarily
# Edit config/config.php
error_reporting(E_ALL);
ini_set('display_errors', 1);

# Check PHP error log
tail -f /var/log/php-errors.log

# Check web server error log
tail -f /var/log/apache2/error.log  # Apache
tail -f /var/log/nginx/error.log    # Nginx
```

### Session Issues
```bash
# Check session directory permissions
ls -la /var/lib/php/sessions

# Clear sessions
sudo rm /var/lib/php/sessions/*
```

## ðŸ“Š Performance Tips

1. **Enable OpCache** (add to php.ini):
```ini
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=10000
```

2. **Database Indexes**: Already optimized in schema.sql

3. **Enable Compression** (Apache):
```apache
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/css text/javascript
</IfModule>
```

4. **Browser Caching** (.htaccess):
```apache
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType text/javascript "access plus 1 month"
</IfModule>
```

## ðŸ“ API Integration (Future)

The system is designed to easily add a REST API:

1. Create `api/` directory
2. Add `api/index.php` as router
3. Return JSON responses
4. Use same Auth and Database classes

Example:
```php
<?php
require_once '../includes/Database.php';
require_once '../includes/Auth.php';

header('Content-Type: application/json');

$db = Database::getInstance();
$assessments = $db->query("SELECT * FROM assessments");

echo json_encode(['success' => true, 'data' => $assessments]);
```

## ðŸ”„ Backup & Restore

### Database Backup
```bash
# Backup
mysqldump -u root -p risk_assessment_db > backup_$(date +%Y%m%d).sql

# Restore
mysql -u root -p risk_assessment_db < backup_20260114.sql
```

### File Backup
```bash
# Backup entire system
tar -czf backup_$(date +%Y%m%d).tar.gz /var/www/html/risk-assessment-system-php

# Restore
tar -xzf backup_20260114.tar.gz -C /var/www/html/
```

## ðŸ“ˆ Extending the System

### Adding New Features
1. Create new database tables (if needed)
2. Add new PHP pages in `public/`
3. Update navigation in `includes/header.php`
4. Add permissions in database
5. Test thoroughly

### Adding Reports
1. Create `reports.php` in `public/`
2. Query database for statistics
3. Display in tables or charts
4. Add export functionality (CSV, PDF)

## ðŸ“ž Support

### Checking Logs
```bash
# Application logs
tail -f logs/error.log

# Web server logs
tail -f /var/log/apache2/error.log
tail -f /var/log/nginx/error.log

# PHP logs
tail -f /var/log/php-errors.log
```

### Common Commands
```bash
# Restart Apache
sudo service apache2 restart

# Restart Nginx
sudo service nginx restart

# Restart MySQL
sudo service mysql restart

# Check PHP version
php -v

# Test PHP configuration
php -m  # List modules
php -i  # Full info
```

## ðŸ“„ License

This project is provided as-is for educational and commercial use.

## ðŸ™ Credits

Built with:
- PHP 8+
- MySQL 8+
- Pure CSS (no frameworks)
- Vanilla JavaScript (no jQuery)

## ðŸ“§ Additional Information

- **Version**: 2.0.0
- **Release Date**: January 2026
- **Status**: Production Ready
- **Architecture**: Traditional PHP (No frameworks)
- **Database**: MySQL with PDO

---

**For detailed API documentation, deployment guides, and troubleshooting, see the `/docs` folder.**

**Happy Assessing! ðŸŽ¯**
