# Photo Attachments for Risk Assessments

## ðŸ“¸ Overview

You can now attach photographs to each risk item in your assessments. This allows you to:
- Document hazards visually
- Provide evidence of risks
- Show "before" and "after" photos
- Support your risk ratings with images
- Create comprehensive visual records

---

## âœ¨ Features

### Photo Management
- âœ… Upload multiple photos per risk item
- âœ… Automatic image resizing (max 2000x2000px)
- âœ… Thumbnail generation (300x300px)
- âœ… Supported formats: JPEG, PNG, GIF, WebP
- âœ… Maximum file size: 5MB per photo
- âœ… Delete photos individually
- âœ… View full-size photos in new tab
- âœ… Grid display with thumbnails

### Security
- âœ… Only image files allowed
- âœ… File type validation
- âœ… Size validation
- âœ… Secure storage
- âœ… Permission-based access
- âœ… Audit trail for uploads/deletes

---

## ðŸš€ Quick Start

### 1. Run Database Update

```bash
# Drop and recreate database with photo support
mysql -u root -p -e "DROP DATABASE IF EXISTS risk_assessment_db;"
mysql -u root -p < database/schema.sql
```

### 2. Verify Uploads Directory

```bash
# Check directory exists and is writable
ls -la public/uploads/assessments/
chmod 755 public/uploads/assessments/
```

### 3. Test Photo Upload

1. Go to any assessment
2. Find a risk item
3. Click "ðŸ“· Add Photo"
4. Select an image file
5. Photo uploads and appears in grid
6. Click photo to view full size
7. Click Ã— to delete

---

## ðŸ“‹ How It Works

### Database Structure

**New Column Added:**
```sql
ALTER TABLE assessment_items 
ADD COLUMN photos JSON NULL COMMENT 'Array of photo filenames';
```

**Stores JSON array:**
```json
["assessment_1_item_5_1234567890_abc123.jpg", "assessment_1_item_5_1234567891_def456.png"]
```

### File Storage

**Directory:** `public/uploads/assessments/`

**Filename Format:**
```
assessment_{assessment_id}_item_{item_id}_{timestamp}_{unique_id}.{ext}
```

**Example:**
```
assessment_1_item_5_1737034567_65a1b2c3d4.jpg
thumb_assessment_1_item_5_1737034567_65a1b2c3d4.jpg
```

---

## ðŸ’» Usage in Code

### Including Photo Component

In your assessment form/view page:

```php
<?php
// Load photos for this item
require_once __DIR__ . '/../includes/PhotoUpload.php';
$photoUpload = new PhotoUpload();
$photos = $photoUpload->getPhotos($itemId);

// Set required variables
$assessmentId = 1;
$itemId = 5;

// Include photo component
include __DIR__ . '/../includes/partials/photo-upload.php';
?>
```

### API Endpoints

**Upload Photo:**
```javascript
POST /api/upload-photo.php
FormData:
  - photo: File
  - assessment_id: int
  - item_id: int

Response:
{
  "success": true,
  "photo": {
    "filename": "assessment_1_item_5_...",
    "url": "https://site.com/uploads/assessments/...",
    "thumbnail": "https://site.com/uploads/assessments/thumb_..."
  }
}
```

**Delete Photo:**
```javascript
POST /api/delete-photo.php
FormData:
  - item_id: int
  - filename: string

Response:
{
  "success": true,
  "message": "Photo deleted successfully"
}
```

### PhotoUpload Class Methods

```php
$photoUpload = new PhotoUpload();

// Upload photo
$result = $photoUpload->upload($_FILES['photo'], $assessmentId, $itemId);

// Get photos for item
$photos = $photoUpload->getPhotos($itemId);

// Add photo to item
$result = $photoUpload->addPhotoToItem($itemId, $filename);

// Remove photo from item
$result = $photoUpload->removePhotoFromItem($itemId, $filename);

// Delete photo file
$deleted = $photoUpload->delete($filename);
```

---

## ðŸŽ¨ Customization

### Changing Upload Limits

Edit `includes/PhotoUpload.php`:

```php
class PhotoUpload {
    private $maxFileSize = 10485760; // 10MB
    private $maxWidth = 3000;         // Max width in pixels
    private $maxHeight = 3000;        // Max height in pixels
```

### Adding More File Types

```php
private $allowedTypes = [
    'image/jpeg',
    'image/jpg',
    'image/png',
    'image/gif',
    'image/webp',
    'image/bmp',  // Add this
    'image/tiff'  // Add this
];
```

### Changing Thumbnail Size

```php
private function createThumbnail($filepath, $filename) {
    $thumbWidth = 500;   // Change from 300
    $thumbHeight = 500;  // Change from 300
```

---

## ðŸ”§ Troubleshooting

### Photos Not Uploading

**Check permissions:**
```bash
chmod 755 public/uploads/assessments/
chown www-data:www-data public/uploads/assessments/
```

**Check PHP upload settings:**
```bash
php -i | grep upload_max_filesize
php -i | grep post_max_size
```

**Edit php.ini if needed:**
```ini
upload_max_filesize = 10M
post_max_size = 10M
max_file_uploads = 20
```

### Images Not Displaying

**Check .htaccess:**
```bash
cat public/uploads/assessments/.htaccess
```

Should allow image files:
```apache
<FilesMatch "\.(jpg|jpeg|png|gif|webp)$">
    Require all granted
</FilesMatch>
```

**Check file exists:**
```bash
ls -la public/uploads/assessments/
```

### "Permission Denied" Error

User doesn't have assessment update permission:

```sql
-- Check permissions
SELECT * FROM permissions WHERE role = 'assessor' AND resource = 'assessments';

-- Grant permission
UPDATE permissions 
SET can_update = 1 
WHERE role = 'assessor' AND resource = 'assessments';
```

---

## ðŸ“Š Storage Management

### Check Disk Usage

```bash
# See total size of uploads
du -sh public/uploads/assessments/

# Count photos
ls public/uploads/assessments/ | grep -v thumb_ | wc -l

# Find large files
find public/uploads/assessments/ -type f -size +2M -ls
```

### Cleanup Old Photos

```bash
# Find photos from deleted assessments
# (photos with no matching assessment_items record)

# This would be a custom script - example:
cd public/uploads/assessments/
for file in assessment_*.{jpg,png,gif,webp}; do
    # Check if referenced in database
    # Delete if not found
done
```

### Backup Photos

```bash
# Backup all photos
tar -czf photos_backup_$(date +%Y%m%d).tar.gz public/uploads/assessments/

# Or sync to remote server
rsync -avz public/uploads/assessments/ user@backup:/backups/photos/
```

---

## ðŸŽ¯ Best Practices

### For Users

1. **Take Clear Photos**
   - Good lighting
   - Focused on the hazard
   - Multiple angles if needed

2. **Add Context**
   - Include recognizable features
   - Show scale (person, object)
   - Capture surrounding area

3. **Use Descriptions**
   - Add notes explaining what the photo shows
   - Reference photo in control measures

4. **Organize Photos**
   - Delete blurry/duplicate photos
   - One hazard per photo
   - Before/after photos for controls

### For Administrators

1. **Monitor Storage**
   - Regular disk space checks
   - Archive old assessments
   - Clean up deleted items

2. **Set Policies**
   - How many photos per item?
   - When to use photos vs. not?
   - Photo quality standards

3. **Train Users**
   - What makes a good risk photo
   - How to capture evidence
   - Photo organization

---

## ðŸ” Security Considerations

### File Type Validation

Photos are validated on:
- File extension
- MIME type (using fileinfo)
- Image validity (can be opened as image)

### Storage Security

- Photos stored outside public_html (recommended)
- .htaccess restricts to images only
- No PHP execution in uploads directory
- Unique filenames prevent overwrites

### Access Control

- Only users with assessment update permission can upload
- Only users who can view assessment can view photos
- Audit trail logs all uploads and deletes

---

## ðŸ“± Mobile Usage

Photos work great on mobile:
- Take photo with camera
- Upload immediately
- View thumbnails on small screens
- Tap to view full size

**Recommended:**
- Take photos on-site during assessment
- Upload immediately while connected
- Review photos before finishing assessment

---

## âœ… Feature Checklist

Setup:
- [ ] Database updated with photos column
- [ ] uploads/assessments directory created
- [ ] Directory permissions set (755)
- [ ] .htaccess security in place
- [ ] PHP upload limits configured

Testing:
- [ ] Can upload JPEG photo
- [ ] Can upload PNG photo
- [ ] Thumbnail displays correctly
- [ ] Full photo opens in new tab
- [ ] Can delete photo
- [ ] Multiple photos work
- [ ] File size validation works
- [ ] File type validation works

Integration:
- [ ] Photos show in assessment view
- [ ] Photos included in PDF export (optional)
- [ ] Photos backed up regularly
- [ ] Storage monitored

---

**Status:** âœ… Production Ready  
**Version:** 2.5.0  
**Last Updated:** January 2026
