# Offline Capabilities & PDF Export Guide

## ðŸ“± Progressive Web App (PWA) Features

The Risk Assessment System now includes full offline capabilities, allowing you to work without an internet connection.

---

## ðŸ”Œ Offline Functionality

### What Works Offline?

âœ… **View cached data**:
- Previously loaded assessments
- Locations list
- Risk items library
- User information

âœ… **Create new assessments**:
- Fill in all assessment details
- Add risk items
- Set likelihood and severity
- Add control measures
- Data stored locally in IndexedDB

âœ… **Edit existing assessments**:
- Modify assessment details
- Update risk evaluations
- Add notes and comments
- Changes queued for sync

âœ… **Automatic Synchronization**:
- When connection returns, all offline changes sync automatically
- Visual indicator shows sync status
- Sync badge shows pending items count
- Manual sync button available

---

## ðŸš€ How Offline Mode Works

### 1. Service Worker Caching

The system uses a service worker to cache essential files:
- CSS stylesheets
- JavaScript files  
- PHP pages (when visited)
- Images and assets

**Cached on first visit**, works offline thereafter.

### 2. IndexedDB Storage

Local database stores:
- Pending assessments
- Offline changes
- Sync queue
- Cached locations and risk items

**Capacity**: ~50MB per domain (browser dependent)

### 3. Background Sync

- Automatically detects when connection returns
- Syncs all pending changes in the background
- Retries failed syncs
- Shows sync progress

---

## ðŸ“Š Using Offline Mode

### Creating Assessment Offline

1. **Go offline** (disable WiFi/data)
2. Navigate to **Assessments** â†’ **New Assessment**
3. Fill in all details normally
4. Click **Create Assessment**
5. Assessment saved to IndexedDB
6. See badge showing "1 pending sync"

### When Connection Returns

1. **Automatic sync** starts
2. Sync icon shows progress
3. Success notification appears
4. Pending badge clears
5. Assessment now on server

### Manual Sync

Click the **ðŸ”„ sync button** in header to:
- Force immediate sync
- Check for pending items
- Retry failed syncs

---

## ðŸ“„ PDF Export Feature

### Generating PDF Reports

**From Assessment View Page:**

1. Open any assessment
2. Click **ðŸ“„ Export PDF** button
3. Browser opens PDF in new tab
4. PDF ready to print/save

**What's Included:**

âœ… Cover page with assessment details
âœ… Location and assessor information
âœ… All risk items with scores
âœ… Control measures
âœ… Action items (if any)
âœ… Additional notes
âœ… Professional formatting

### PDF Format

**Features:**
- A4 size, print-ready
- Color-coded risk levels
- Professional header/footer
- Auto-generated date/time
- Assessment number watermark

**Sections:**
1. **Cover Page**: Title, number, location
2. **Assessment Info**: Details and status
3. **Risk Assessment**: All risks in table
4. **Action Items**: Tasks and due dates
5. **Notes**: Additional information

---

## ðŸ”§ Technical Details

### Service Worker

**Location**: `/public/sw.js`

**Capabilities:**
- Caches assets on install
- Serves from cache when offline
- Updates cache on network requests
- Cleans up old cache versions

**Cache Strategy**: Cache-first with network fallback

### IndexedDB Structure

**Stores:**
- `assessments`: Offline assessment data
- `syncQueue`: Pending changes
- `locations`: Cached locations
- `riskItems`: Cached risk library

### Sync Queue

**Fields:**
- `id`: Auto-increment
- `action`: create/update/delete
- `entityType`: assessment/location/risk
- `data`: Serialized entity data
- `timestamp`: When queued
- `status`: pending/complete/failed
- `attempts`: Retry count

---

## ðŸŽ¯ Best Practices

### For Users

**Before Going Offline:**
1. Load all pages you'll need
2. Open assessment list
3. View any assessments you need
4. Let data cache completely

**While Offline:**
1. Create assessments as normal
2. Monitor pending sync badge
3. Don't clear browser data
4. Keep app open if possible

**When Back Online:**
1. Wait for auto-sync to complete
2. Verify synced data on server
3. Check for any failed syncs
4. Manually retry if needed

### For Administrators

**Monitoring:**
- Check sync queue in database
- Monitor failed sync attempts
- Set up alerts for sync failures
- Review audit logs

**Troubleshooting:**
- Clear service worker cache
- Reset IndexedDB if corrupted
- Check browser console for errors
- Verify server is accessible

---

## ðŸ“± Installing as App

### Desktop (Chrome/Edge)

1. Visit the system
2. Look for install icon in address bar
3. Click **Install**
4. App opens in standalone window
5. Add to taskbar/desktop

### Mobile (Android/iOS)

1. Open in browser (Chrome/Safari)
2. Tap menu (â‹® or share icon)
3. Select **Add to Home Screen**
4. Name the app
5. Tap **Add**
6. Icon appears on home screen

### Benefits

âœ… Standalone window (no browser UI)
âœ… Faster loading
âœ… Better offline experience
âœ… Push notifications (ready)
âœ… App-like feel

---

## ðŸ” Monitoring Offline Usage

### Sync Status Indicators

**Online Status Badge:**
- ðŸŸ¢ Green = Connected
- ðŸŸ¡ Yellow = Offline
- Shows in header always

**Sync Button:**
- ðŸ”„ icon in header
- Badge shows pending count
- Click to force sync
- Spins during sync

**Sync Progress:**
- "Syncing..." text appears
- Shows during sync operation
- Disappears when complete

### Database Queries

```sql
-- Check sync queue
SELECT * FROM sync_queue WHERE status = 'pending';

-- Count pending syncs
SELECT COUNT(*) FROM sync_queue WHERE status = 'pending';

-- Failed syncs
SELECT * FROM sync_queue WHERE status = 'failed' AND attempts >= 3;
```

---

## ðŸ› ï¸ Troubleshooting

### Issue: Offline Mode Not Working

**Check:**
1. HTTPS enabled (required for service workers)
2. Browser supports service workers
3. Service worker registered successfully
4. Check browser console for errors

**Solution:**
```javascript
// In browser console
navigator.serviceWorker.getRegistrations().then(registrations => {
    console.log('Registrations:', registrations);
});
```

### Issue: Sync Not Working

**Check:**
1. Internet connection restored
2. Pending items in queue
3. No JavaScript errors
4. Server accessible

**Solution:**
1. Click manual sync button
2. Check browser console
3. Verify server endpoint
4. Review sync queue table

### Issue: PDF Not Generating

**Check:**
1. Assessment exists
2. User has export permission
3. Browser allows pop-ups
4. Print dialog appears

**Solution:**
1. Allow pop-ups for site
2. Check browser print settings
3. Try different browser
4. Check server PHP errors

### Issue: Data Not Syncing

**Check:**
1. Server receiving requests
2. Authentication still valid
3. Permission still granted
4. Network connectivity

**Solution:**
```javascript
// Force sync in console
offlineManager.syncOfflineData();
```

---

## ðŸ“Š Performance Tips

### Optimize Cache

**Keep cache size reasonable:**
- Cache essential files only
- Remove old versions
- Clear unused data

**Service Worker:**
```javascript
// Update cache name when app updates
const CACHE_NAME = 'risk-assessment-v2.0.1';
```

### IndexedDB Maintenance

**Periodic cleanup:**
- Delete synced items
- Remove old cached data
- Compact database

### Network Optimization

- Use compression
- Minimize HTTP requests
- Lazy load images
- Cache API responses

---

## ðŸ” Security Considerations

### Offline Data

**Stored locally:**
- Assessment data
- User information  
- Session tokens

**Protection:**
- Encrypted at rest (browser)
- Not accessible to other sites
- Cleared on logout
- Requires device access

### Sync Security

**Server validation:**
- Authentication required
- Permission checks
- Data validation
- CSRF protection

---

## ðŸ“ˆ Browser Support

### Service Workers

âœ… Chrome 40+
âœ… Firefox 44+
âœ… Safari 11.1+
âœ… Edge 17+

### IndexedDB

âœ… Chrome 24+
âœ… Firefox 16+
âœ… Safari 10+
âœ… Edge 12+

### PDF Export

âœ… All modern browsers
âœ… Print to PDF supported
âœ… Save as PDF available

---

## ðŸŽ“ Advanced Features

### Custom Sync Logic

Edit `assets/js/offline.js`:

```javascript
// Custom sync handler
async syncItem(item) {
    // Your custom logic here
    const response = await fetch('/api/sync', {
        method: 'POST',
        body: JSON.stringify(item)
    });
    return response.json();
}
```

### Cache Management

Edit `public/sw.js`:

```javascript
// Add URLs to precache
const PRECACHE_URLS = [
    '/',
    '/dashboard.php',
    '/your-custom-page.php'
];
```

### Custom PDF Formatting

Edit `public/assessment-export-pdf.php`:

```php
// Customize PDF styles
<style>
    .your-custom-class {
        /* Your styles */
    }
</style>
```

---

## ðŸ“ž Support

### Getting Help

1. Check browser console for errors
2. Review service worker status
3. Inspect IndexedDB contents
4. Check sync queue in database
5. Review server logs

### Useful Commands

```javascript
// Check service worker status
navigator.serviceWorker.ready.then(reg => {
    console.log('Service Worker ready:', reg);
});

// Check IndexedDB
indexedDB.databases().then(dbs => {
    console.log('Databases:', dbs);
});

// Force sync
offlineManager.syncOfflineData();

// Clear offline data
offlineManager.clearOfflineData();
```

---

## âœ… Checklist

**For Going Live:**

- [ ] Service worker registered
- [ ] HTTPS enabled
- [ ] Manifest.json configured
- [ ] Icons created (192x192, 512x512)
- [ ] Offline page created
- [ ] Sync tested offlineâ†’online
- [ ] PDF export tested
- [ ] Browser compatibility checked
- [ ] Mobile devices tested
- [ ] Performance optimized

---

**Feature Status:** âœ… Production Ready  
**Last Updated:** January 2026  
**Version:** 2.0.0
