# Installation Guide - Risk Assessment System v3.0

Complete step-by-step installation guide for the HGV Delivery Risk Assessment System.

## 📋 System Requirements

- **PHP**: 8.0 or higher
- **MySQL**: 8.0 or higher
- **Web Server**: Apache 2.4+ or Nginx 1.18+
- **PHP Extensions**: PDO, PDO_MySQL, JSON, GD (for image processing)
- **Disk Space**: Minimum 500MB free
- **SSL Certificate**: Recommended for production (required for PWA features)

## 🚀 Quick Start Installation

### Step 1: Extract Files

```bash
# Extract the archive
tar -xzf risk-assessment-system-v3.tar.gz

# Move to web server directory
sudo mv risk-assessment-system-v3 /var/www/html/risk-assessment

# Or for cPanel/shared hosting:
# Extract via File Manager or upload via FTP to public_html/
```

### Step 2: Create Database

**Option A: Command Line**
```bash
mysql -u root -p < database/schema.sql
```

**Option B: phpMyAdmin**
1. Open phpMyAdmin
2. Click "Import"
3. Select `database/schema.sql`
4. Click "Go"

**Option C: Manual Creation**
```bash
mysql -u root -p
```
```sql
CREATE DATABASE risk_assessment_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE risk_assessment_db;
SOURCE /path/to/database/schema.sql;
```

### Step 3: Configure Database Connection

Edit `config/config.php`:

```php
// Database Configuration
define('DB_HOST', 'localhost');           // Usually 'localhost'
define('DB_NAME', 'risk_assessment_db');  // Database name
define('DB_USER', 'your_username');       // MySQL username
define('DB_PASS', 'your_password');       // MySQL password
```

**For cPanel/Shared Hosting:**
- DB_HOST: Usually 'localhost'
- DB_NAME: Prefix + database name (e.g., 'username_risk')
- DB_USER: Your cPanel MySQL user
- DB_PASS: Your MySQL password

### Step 4: Set Permissions

**For Linux/Apache:**
```bash
cd /var/www/html/risk-assessment

# Set ownership
sudo chown -R www-data:www-data .

# Set directory permissions
sudo find . -type d -exec chmod 755 {} \;

# Set file permissions
sudo find . -type f -exec chmod 644 {} \;

# Make uploads and logs writable
sudo chmod 755 uploads/assessments logs/
```

**For Nginx:**
```bash
sudo chown -R nginx:nginx .
# (same permission commands as above)
```

**For cPanel/Shared Hosting:**
- Set folders to 755
- Set files to 644
- uploads/assessments: 755
- logs: 755

### Step 5: Configure Web Server

#### Apache (Usually automatic with .htaccess)

The `.htaccess` file is included. Just ensure mod_rewrite is enabled:

```bash
sudo a2enmod rewrite
sudo service apache2 restart
```

#### Nginx Configuration

Add to your nginx configuration:

```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /var/www/html/risk-assessment/public;
    index index.php index.html;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        include fastcgi_params;
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }

    location ~ /\.ht {
        deny all;
    }
}
```

Then restart Nginx:
```bash
sudo nginx -t
sudo systemctl restart nginx
```

### Step 6: Access the System

1. Open your browser
2. Navigate to: `http://your-domain.com/public/`
3. You'll be redirected to the login page

**Default Login:**
- Username: `admin`
- Password: `admin123`

⚠️ **CRITICAL: Change the admin password immediately after first login!**

---

## 🔧 Advanced Configuration

### Email Notifications

Configure email settings in `config/config.php`:

```php
define('SMTP_FROM', 'noreply@yourdomain.com');
define('SMTP_FROM_NAME', 'Risk Assessment System');
define('ENABLE_EMAIL_NOTIFICATIONS', true);
```

For production email, configure your server's mail system or use an SMTP service.

### Google Maps Integration

1. Get a Google Maps API key from: https://console.cloud.google.com/
2. Enable the following APIs:
   - Maps JavaScript API
   - Geocoding API
   - Places API
3. Add the key to database:

```sql
UPDATE system_settings 
SET setting_value = 'YOUR_GOOGLE_MAPS_API_KEY' 
WHERE setting_key = 'google_maps_api_key';
```

### what3words Integration

1. Get a what3words API key from: https://developer.what3words.com/
2. Add the key to database:

```sql
UPDATE system_settings 
SET setting_value = 'YOUR_WHAT3WORDS_API_KEY' 
WHERE setting_key = 'what3words_api_key';
```

### SSL Certificate (Required for PWA)

**Using Let's Encrypt (Free):**
```bash
sudo apt install certbot python3-certbot-apache
sudo certbot --apache -d your-domain.com
```

**For cPanel:**
- Use AutoSSL or Let's Encrypt in cPanel
- Or purchase/install SSL certificate

---

## 🔒 Security Checklist

After installation, complete these security steps:

### 1. Change Default Password
```sql
-- Generate new password hash (use PHP):
-- password_hash('your_new_password', PASSWORD_DEFAULT);

UPDATE users 
SET password_hash = 'NEW_HASH_HERE' 
WHERE username = 'admin';
```

### 2. Update Config for Production

Edit `config/config.php`:
```php
define('ENVIRONMENT', 'production');  // Change from 'development'
```

### 3. Disable Error Display

Ensure in `config/config.php`:
```php
if (ENVIRONMENT === 'production') {
    error_reporting(E_ALL & ~E_DEPRECATED);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
}
```

### 4. Secure File Uploads

Verify `.htaccess` in uploads directory prevents PHP execution.

### 5. Enable HTTPS

Update `config/config.php` to force HTTPS:
```php
ini_set('session.cookie_secure', 1); // Already set to 1
```

Uncomment in `public/.htaccess`:
```apache
# Force HTTPS
RewriteCond %{HTTPS} off
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
```

---

## 🐛 Troubleshooting

### Issue: Blank White Screen

**Check PHP errors:**
```bash
tail -f /var/log/apache2/error.log
# or
tail -f /var/log/nginx/error.log
```

**Enable error display temporarily:**
Edit `config/config.php`:
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

### Issue: Database Connection Error

**Verify credentials:**
```bash
mysql -u YOUR_USERNAME -p
```

**Check if database exists:**
```sql
SHOW DATABASES LIKE 'risk_assessment_db';
```

**Verify user permissions:**
```sql
SHOW GRANTS FOR 'YOUR_USERNAME'@'localhost';
```

### Issue: Permission Denied

```bash
# Fix ownership
sudo chown -R www-data:www-data /var/www/html/risk-assessment

# Fix permissions
sudo chmod -R 755 /var/www/html/risk-assessment
sudo chmod 755 uploads/assessments logs/
```

### Issue: .htaccess Not Working

**Enable mod_rewrite:**
```bash
sudo a2enmod rewrite
sudo service apache2 restart
```

**Check Apache config allows .htaccess:**
Edit `/etc/apache2/sites-available/000-default.conf`:
```apache
<Directory /var/www/html>
    AllowOverride All
</Directory>
```

### Issue: File Uploads Not Working

**Check PHP upload limits:**
```bash
php -i | grep upload_max_filesize
php -i | grep post_max_size
```

**Increase limits in php.ini:**
```ini
upload_max_filesize = 10M
post_max_size = 10M
```

---

## 📱 PWA Installation

For offline functionality:

### 1. Ensure HTTPS is Enabled
PWA requires HTTPS in production.

### 2. Create Service Worker
Place `sw.js` in `public/` directory (to be added in future updates).

### 3. Manifest File
Place `manifest.json` in `public/` directory (to be added in future updates).

---

## 🔄 Updates & Maintenance

### Backup Database
```bash
mysqldump -u root -p risk_assessment_db > backup_$(date +%Y%m%d).sql
```

### Backup Files
```bash
tar -czf backup_files_$(date +%Y%m%d).tar.gz /var/www/html/risk-assessment
```

### Update Database
```bash
mysql -u root -p risk_assessment_db < updates/update_v3.1.sql
```

---

## 📊 Performance Optimization

### Enable OpCache

Add to `php.ini`:
```ini
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=10000
opcache.revalidate_freq=2
```

### Database Optimization

Run periodically:
```sql
OPTIMIZE TABLE assessments, assessment_risks, locations;
```

---

## 🆘 Support & Resources

### Log Files
- Application errors: `logs/php_errors.log`
- Web server errors: `/var/log/apache2/error.log` or `/var/log/nginx/error.log`
- MySQL errors: `/var/log/mysql/error.log`

### Useful Commands

**Check PHP Version:**
```bash
php -v
```

**Check MySQL Version:**
```bash
mysql --version
```

**Test Database Connection:**
```bash
mysql -u YOUR_USERNAME -p -e "SELECT 1"
```

**Check PHP Extensions:**
```bash
php -m | grep -E 'pdo|mysql|json|gd'
```

**Restart Services:**
```bash
sudo systemctl restart apache2  # or nginx
sudo systemctl restart mysql
sudo systemctl restart php8.1-fpm
```

---

## ✅ Post-Installation Checklist

After installation, verify:

- [ ] Can login with admin/admin123
- [ ] Changed default admin password
- [ ] Dashboard loads and shows statistics
- [ ] Can create a location
- [ ] Can create an assessment
- [ ] File permissions are correct (755 for directories, 644 for files)
- [ ] Uploads directory is writable
- [ ] Error logging is working (check logs/ directory)
- [ ] Database connection is stable
- [ ] HTTPS is enabled (for production)
- [ ] Backed up initial database and files

---

## 🎯 Next Steps

1. **Add Users**: Go to Users menu (Admin only)
2. **Add Locations**: Add UK supermarket locations with coordinates
3. **Configure API Keys**: Add Google Maps and what3words keys
4. **Customize Risk Library**: Add/edit risk items for your needs
5. **Set Up Email**: Configure email notifications
6. **Train Users**: Provide training on how to use the system

---

## 📞 Need Help?

- Check log files: `logs/php_errors.log`
- Review database: `mysql -u root -p risk_assessment_db`
- Verify permissions: `ls -la`
- Test PHP: Create `info.php` with `<?php phpinfo();`

---

**Version**: 3.0.0  
**Last Updated**: February 2026  
**Status**: Production Ready

Good luck with your installation! 🚀
