import React, { useState, useEffect, createContext, useContext } from 'react';
import { BrowserRouter, Routes, Route, Navigate, Link, useNavigate } from 'react-router-dom';

// ============================================
// CONTEXT & STATE MANAGEMENT
// ============================================

const AuthContext = createContext(null);
const ThemeContext = createContext(null);

function useAuth() {
  return useContext(AuthContext);
}

function useTheme() {
  return useContext(ThemeContext);
}

// ============================================
// API SERVICE
// ============================================

class API {
  static baseURL = 'http://localhost/api';

  static async request(endpoint, options = {}) {
    const token = localStorage.getItem('token');
    const headers = {
      'Content-Type': 'application/json',
      ...options.headers,
    };

    if (token) {
      headers['Authorization'] = `Bearer ${token}`;
    }

    try {
      const response = await fetch(`${this.baseURL}${endpoint}`, {
        ...options,
        headers,
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.message || 'Request failed');
      }

      return data;
    } catch (error) {
      throw error;
    }
  }

  static async get(endpoint, params = {}) {
    const queryString = new URLSearchParams(params).toString();
    const url = queryString ? `${endpoint}?${queryString}` : endpoint;
    return this.request(url, { method: 'GET' });
  }

  static async post(endpoint, data) {
    return this.request(endpoint, {
      method: 'POST',
      body: JSON.stringify(data),
    });
  }

  static async put(endpoint, data) {
    return this.request(endpoint, {
      method: 'PUT',
      body: JSON.stringify(data),
    });
  }

  static async delete(endpoint) {
    return this.request(endpoint, { method: 'DELETE' });
  }
}

// ============================================
// OFFLINE MANAGER
// ============================================

class OfflineManager {
  static dbName = 'riskAssessmentDB';
  static version = 1;
  static db = null;

  static async init() {
    return new Promise((resolve, reject) => {
      const request = indexedDB.open(this.dbName, this.version);

      request.onerror = () => reject(request.error);
      request.onsuccess = () => {
        this.db = request.result;
        resolve(this.db);
      };

      request.onupgradeneeded = (event) => {
        const db = event.target.result;
        
        if (!db.objectStoreNames.contains('assessments')) {
          db.createObjectStore('assessments', { keyPath: 'id', autoIncrement: true });
        }
        if (!db.objectStoreNames.contains('syncQueue')) {
          db.createObjectStore('syncQueue', { keyPath: 'id', autoIncrement: true });
        }
      };
    });
  }

  static async saveOffline(storeName, data) {
    const db = await this.init();
    return new Promise((resolve, reject) => {
      const transaction = db.transaction([storeName], 'readwrite');
      const store = transaction.objectStore(storeName);
      const request = store.add(data);
      
      request.onsuccess = () => resolve(request.result);
      request.onerror = () => reject(request.error);
    });
  }

  static async sync() {
    if (!navigator.onLine) return;
    
    const db = await this.init();
    const transaction = db.transaction(['syncQueue'], 'readonly');
    const store = transaction.objectStore('syncQueue');
    const request = store.getAll();

    request.onsuccess = async () => {
      const items = request.result;
      if (items.length > 0) {
        try {
          await API.post('/sync', { items });
          // Clear synced items
          const clearTransaction = db.transaction(['syncQueue'], 'readwrite');
          const clearStore = clearTransaction.objectStore('syncQueue');
          clearStore.clear();
        } catch (error) {
          console.error('Sync failed:', error);
        }
      }
    };
  }
}

// ============================================
// AUTHENTICATION PROVIDER
// ============================================

function AuthProvider({ children }) {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const token = localStorage.getItem('token');
    const userData = localStorage.getItem('user');
    if (token && userData) {
      setUser(JSON.parse(userData));
    }
    setLoading(false);
  }, []);

  const login = async (username, password) => {
    const response = await API.post('/auth/login', { username, password });
    localStorage.setItem('token', response.data.token);
    localStorage.setItem('user', JSON.stringify(response.data.user));
    setUser(response.data.user);
    return response.data.user;
  };

  const logout = () => {
    localStorage.removeItem('token');
    localStorage.removeItem('user');
    setUser(null);
  };

  const value = { user, login, logout, loading };

  return <AuthContext.Provider value={value}>{children}</AuthContext.Provider>;
}

// ============================================
// THEME PROVIDER
// ============================================

function ThemeProvider({ children }) {
  const [theme, setTheme] = useState(() => {
    return localStorage.getItem('theme') || 'light';
  });

  useEffect(() => {
    document.documentElement.setAttribute('data-theme', theme);
    localStorage.setItem('theme', theme);
  }, [theme]);

  const toggleTheme = () => {
    setTheme(prev => prev === 'light' ? 'dark' : 'light');
  };

  return (
    <ThemeContext.Provider value={{ theme, toggleTheme }}>
      {children}
    </ThemeContext.Provider>
  );
}

// ============================================
// PROTECTED ROUTE COMPONENT
// ============================================

function ProtectedRoute({ children }) {
  const { user, loading } = useAuth();

  if (loading) {
    return <div className="loading">Loading...</div>;
  }

  if (!user) {
    return <Navigate to="/login" />;
  }

  return children;
}

// ============================================
// LOGIN PAGE
// ============================================

function LoginPage() {
  const [username, setUsername] = useState('');
  const [password, setPassword] = useState('');
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(false);
  const { login } = useAuth();
  const navigate = useNavigate();

  const handleSubmit = async (e) => {
    e.preventDefault();
    setError('');
    setLoading(true);

    try {
      await login(username, password);
      navigate('/dashboard');
    } catch (err) {
      setError(err.message || 'Login failed');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="login-container">
      <div className="login-card">
        <div className="login-header">
          <h1>Risk Assessment System</h1>
          <p>Sign in to your account</p>
        </div>

        <form onSubmit={handleSubmit} className="login-form">
          {error && <div className="alert alert-error">{error}</div>}

          <div className="form-group">
            <label htmlFor="username">Username</label>
            <input
              type="text"
              id="username"
              value={username}
              onChange={(e) => setUsername(e.target.value)}
              required
              autoFocus
            />
          </div>

          <div className="form-group">
            <label htmlFor="password">Password</label>
            <input
              type="password"
              id="password"
              value={password}
              onChange={(e) => setPassword(e.target.value)}
              required
            />
          </div>

          <button type="submit" className="btn btn-primary btn-block" disabled={loading}>
            {loading ? 'Signing in...' : 'Sign In'}
          </button>
        </form>

        <div className="login-footer">
          <p>Default: admin / admin123</p>
        </div>
      </div>
    </div>
  );
}

// ============================================
// DASHBOARD PAGE
// ============================================

function DashboardPage() {
  const [stats, setStats] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadStats();
  }, []);

  const loadStats = async () => {
    try {
      const response = await API.get('/dashboard');
      setStats(response.data);
    } catch (error) {
      console.error('Failed to load stats:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return <div className="loading">Loading dashboard...</div>;
  }

  return (
    <div className="dashboard">
      <h1>Dashboard</h1>

      <div className="stats-grid">
        <div className="stat-card">
          <div className="stat-value">{stats?.stats?.total_assessments || 0}</div>
          <div className="stat-label">Total Assessments</div>
        </div>

        <div className="stat-card">
          <div className="stat-value">{stats?.stats?.completed || 0}</div>
          <div className="stat-label">Completed</div>
        </div>

        <div className="stat-card">
          <div className="stat-value">{stats?.stats?.in_progress || 0}</div>
          <div className="stat-label">In Progress</div>
        </div>

        <div className="stat-card stat-danger">
          <div className="stat-value">{stats?.stats?.high_risk_count || 0}</div>
          <div className="stat-label">High Risk Items</div>
        </div>
      </div>

      <div className="recent-section">
        <h2>Recent Assessments</h2>
        <div className="table-container">
          <table className="data-table">
            <thead>
              <tr>
                <th>Number</th>
                <th>Title</th>
                <th>Location</th>
                <th>Status</th>
                <th>Date</th>
              </tr>
            </thead>
            <tbody>
              {stats?.recent_assessments?.map(assessment => (
                <tr key={assessment.id}>
                  <td>{assessment.assessment_number}</td>
                  <td>{assessment.title}</td>
                  <td>{assessment.location_name}</td>
                  <td>
                    <span className={`badge badge-${assessment.status}`}>
                      {assessment.status}
                    </span>
                  </td>
                  <td>{new Date(assessment.assessment_date).toLocaleDateString()}</td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );
}

// ============================================
// ASSESSMENTS PAGE
// ============================================

function AssessmentsPage() {
  const [assessments, setAssessments] = useState([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [total, setTotal] = useState(0);
  const navigate = useNavigate();

  useEffect(() => {
    loadAssessments();
  }, [page]);

  const loadAssessments = async () => {
    try {
      const response = await API.get('/assessments', { page });
      setAssessments(response.data);
      setTotal(response.pagination.total);
    } catch (error) {
      console.error('Failed to load assessments:', error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="assessments-page">
      <div className="page-header">
        <h1>Risk Assessments</h1>
        <button 
          className="btn btn-primary"
          onClick={() => navigate('/assessments/new')}
        >
          + New Assessment
        </button>
      </div>

      {loading ? (
        <div className="loading">Loading assessments...</div>
      ) : (
        <>
          <div className="table-container">
            <table className="data-table">
              <thead>
                <tr>
                  <th>Number</th>
                  <th>Title</th>
                  <th>Location</th>
                  <th>Assessor</th>
                  <th>Status</th>
                  <th>Risk Level</th>
                  <th>Date</th>
                  <th>Actions</th>
                </tr>
              </thead>
              <tbody>
                {assessments.map(assessment => (
                  <tr key={assessment.id}>
                    <td>{assessment.assessment_number}</td>
                    <td>{assessment.title}</td>
                    <td>{assessment.location_name}</td>
                    <td>{assessment.assessor_name}</td>
                    <td>
                      <span className={`badge badge-${assessment.status}`}>
                        {assessment.status}
                      </span>
                    </td>
                    <td>
                      {assessment.overall_risk_level && (
                        <span className={`badge badge-${assessment.overall_risk_level}`}>
                          {assessment.overall_risk_level}
                        </span>
                      )}
                    </td>
                    <td>{new Date(assessment.assessment_date).toLocaleDateString()}</td>
                    <td>
                      <button 
                        className="btn btn-sm"
                        onClick={() => navigate(`/assessments/${assessment.id}`)}
                      >
                        View
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>

          <div className="pagination">
            <button 
              disabled={page === 1}
              onClick={() => setPage(p => p - 1)}
            >
              Previous
            </button>
            <span>Page {page}</span>
            <button 
              disabled={assessments.length < 20}
              onClick={() => setPage(p => p + 1)}
            >
              Next
            </button>
          </div>
        </>
      )}
    </div>
  );
}

// ============================================
// LAYOUT COMPONENT
// ============================================

function Layout({ children }) {
  const { user, logout } = useAuth();
  const { theme, toggleTheme } = useTheme();
  const [menuOpen, setMenuOpen] = useState(false);

  return (
    <div className="app-layout">
      <header className="app-header">
        <div className="header-left">
          <button className="menu-toggle" onClick={() => setMenuOpen(!menuOpen)}>
            ☰
          </button>
          <Link to="/" className="logo">
            Risk Assessment System
          </Link>
        </div>

        <nav className={`main-nav ${menuOpen ? 'open' : ''}`}>
          <Link to="/dashboard">Dashboard</Link>
          <Link to="/assessments">Assessments</Link>
          <Link to="/locations">Locations</Link>
          <Link to="/reports">Reports</Link>
          {user?.role === 'admin' && (
            <Link to="/settings">Settings</Link>
          )}
        </nav>

        <div className="header-right">
          <button className="theme-toggle" onClick={toggleTheme}>
            {theme === 'light' ? '🌙' : '☀️'}
          </button>
          <div className="user-menu">
            <span className="user-name">{user?.full_name}</span>
            <button onClick={logout} className="btn btn-sm">
              Logout
            </button>
          </div>
        </div>
      </header>

      <main className="app-main">
        {children}
      </main>

      <footer className="app-footer">
        <p>© 2026 Risk Assessment System v2.0.0</p>
      </footer>
    </div>
  );
}

// ============================================
// MAIN APP COMPONENT
// ============================================

function App() {
  useEffect(() => {
    // Initialize offline mode
    OfflineManager.init();

    // Register service worker for PWA
    if ('serviceWorker' in navigator) {
      navigator.serviceWorker.register('/sw.js').catch(err => {
        console.error('Service worker registration failed:', err);
      });
    }

    // Sync when coming back online
    window.addEventListener('online', () => {
      OfflineManager.sync();
    });
  }, []);

  return (
    <BrowserRouter>
      <ThemeProvider>
        <AuthProvider>
          <Routes>
            <Route path="/login" element={<LoginPage />} />
            <Route
              path="/*"
              element={
                <ProtectedRoute>
                  <Layout>
                    <Routes>
                      <Route path="/" element={<Navigate to="/dashboard" />} />
                      <Route path="/dashboard" element={<DashboardPage />} />
                      <Route path="/assessments" element={<AssessmentsPage />} />
                      <Route path="/locations" element={<div>Locations Page</div>} />
                      <Route path="/reports" element={<div>Reports Page</div>} />
                      <Route path="/settings" element={<div>Settings Page</div>} />
                    </Routes>
                  </Layout>
                </ProtectedRoute>
              }
            />
          </Routes>
        </AuthProvider>
      </ThemeProvider>
    </BrowserRouter>
  );
}

export default App;
