# Risk Assessment System v2.0.0

A comprehensive, production-ready risk assessment management system built with MySQL, PHP, and React.js. Features include offline capabilities, PDF export, dark mode, and complete CRUD operations for assessments, locations, users, and more.

## 📋 Features

### Core Features
- ✅ Complete risk assessment lifecycle management
- ✅ Location-based assessments
- ✅ Multi-user support with role-based permissions (Admin, Assessor, Viewer)
- ✅ Assessment numbering system (RA-YYYY-NNNNNN)
- ✅ Risk level calculation (Low, Medium, High)
- ✅ Action items tracking
- ✅ Audit logging

### Advanced Features
- ✅ **Dark Mode** - Beautiful light and dark themes
- ✅ **Offline Mode (PWA)** - Full offline functionality with IndexedDB
- ✅ **PDF Export** - Professional assessment reports
- ✅ **Responsive Design** - Mobile, tablet, and desktop optimized
- ✅ **Real-time Sync** - Background synchronization
- ✅ **Custom Fields** - Extensible data model

### Technical Stack
- **Frontend**: React.js 18 with React Router
- **Backend**: PHP 8+ with RESTful API
- **Database**: MySQL 8.0+ with InnoDB engine
- **Architecture**: Single Page Application (SPA)
- **PWA**: Service Workers + IndexedDB

## 🚀 Quick Start

### Prerequisites
- PHP 8.0 or higher
- MySQL 8.0 or higher
- Node.js 16+ and npm
- Apache or Nginx web server

### Installation Steps

#### 1. Database Setup
```bash
# Create database and import schema
mysql -u root -p < database/schema.sql
```

#### 2. Backend Setup
```bash
# Navigate to backend directory
cd backend

# Update config.php with your database credentials
# Edit the following constants:
# - DB_HOST
# - DB_NAME
# - DB_USER
# - DB_PASS

# Ensure the uploads directory is writable
chmod 755 uploads/
```

#### 3. Frontend Setup
```bash
# Navigate to frontend directory
cd frontend

# Install dependencies
npm install

# Update API URL in src/App.jsx if needed
# Look for: static baseURL = 'http://localhost/api';

# Start development server
npm start
```

#### 4. Web Server Configuration

**Apache (.htaccess for API)**
```apache
RewriteEngine On
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^api/(.*)$ api/index.php [L,QSA]
```

**Nginx**
```nginx
location /api {
    try_files $uri $uri/ /api/index.php?$query_string;
}
```

### Default Login Credentials
- **Username**: admin
- **Password**: admin123

⚠️ **Important**: Change the default password immediately after first login!

## 📁 Project Structure

```
risk-assessment-system/
├── backend/
│   ├── api/
│   │   └── index.php          # Main API router
│   ├── controllers/           # Business logic controllers
│   │   ├── UserController.php
│   │   ├── AssessmentController.php
│   │   ├── LocationController.php
│   │   ├── RiskController.php
│   │   ├── ActionItemController.php
│   │   ├── SettingsController.php
│   │   └── ReportController.php
│   ├── uploads/               # File uploads directory
│   ├── config.php             # Configuration
│   ├── Database.php           # Database connection
│   ├── Auth.php               # Authentication & JWT
│   └── Response.php           # API response handler
├── frontend/
│   ├── public/
│   │   ├── index.html
│   │   ├── manifest.json      # PWA manifest
│   │   └── sw.js              # Service worker
│   ├── src/
│   │   ├── App.jsx            # Main React app
│   │   ├── App.css            # Styles
│   │   └── index.js           # Entry point
│   └── package.json
├── database/
│   └── schema.sql             # Complete database schema
└── docs/
    ├── README.md              # This file
    ├── API.md                 # API documentation
    └── DEPLOYMENT.md          # Deployment guide
```

## 🎯 Usage Guide

### Creating an Assessment
1. Navigate to **Assessments** → **New Assessment**
2. Select location
3. Enter assessment details
4. Add risk items
5. Assign likelihood and severity
6. Add control measures
7. Create action items as needed
8. Export as PDF when complete

### User Roles

**Admin**
- Full system access
- User management
- Settings configuration
- All CRUD operations

**Assessor**
- Create and edit assessments
- View locations and users
- Export reports
- Manage action items

**Viewer**
- Read-only access
- View assessments and reports
- Export capability

### Offline Mode
The system works completely offline:
1. Open the app while online (first time)
2. Go offline
3. Continue creating/editing assessments
4. Changes are queued locally
5. Automatic sync when back online

## 🔒 Security Features

- Password hashing with PHP password_hash()
- JWT token authentication
- SQL injection protection (prepared statements)
- XSS protection (output sanitization)
- CSRF protection
- Role-based access control
- Session management
- Audit logging

## 📊 Database Schema

### Core Tables (16 total)
1. `users` - User accounts and authentication
2. `locations` - Assessment locations
3. `risk_categories` - Risk categorization
4. `risk_items` - Risk library
5. `assessments` - Risk assessments
6. `assessment_risks` - Individual risk records
7. `action_items` - Corrective actions
8. `permissions` - Role-based permissions
9. `custom_fields` - Extensible fields
10. `custom_field_values` - Custom data
11. `system_settings` - Configuration
12. `audit_log` - Activity tracking
13. `email_queue` - Email scheduling
14. `file_versions` - File management
15. `sessions` - Session tracking
16. `sync_queue` - Offline sync

## 🛠️ Development

### Adding New Features
1. Create database table/columns if needed
2. Add controller methods in `backend/controllers/`
3. Update API routes in `backend/api/index.php`
4. Create React components in `frontend/src/`
5. Add routes in `App.jsx`

### Running Tests
```bash
# Backend
php backend/test-system.php

# Frontend
cd frontend
npm test
```

### Building for Production
```bash
# Frontend
cd frontend
npm run build

# The build folder contains production-ready files
# Copy contents to your web server
```

## 🎨 Customization

### Theming
Edit CSS variables in `frontend/src/App.css`:
```css
:root {
  --primary: #2563eb;
  --success: #10b981;
  --danger: #ef4444;
  /* ...more variables */
}
```

### Branding
Update system settings via admin panel:
- System name
- Company name
- Logo upload
- Color scheme

## 📱 Progressive Web App (PWA)

The system is installable as a standalone app:
1. Visit the site in Chrome/Edge
2. Click the install icon in address bar
3. Or go to Settings → Install app

### PWA Features
- Offline functionality
- Push notifications (ready)
- Background sync
- Home screen icon
- Splash screen
- Standalone mode

## 🔧 Troubleshooting

### Database Connection Error
- Check `config.php` credentials
- Verify MySQL service is running
- Ensure database exists

### API 404 Errors
- Check `.htaccess` rewrite rules
- Verify API URL in frontend
- Check web server configuration

### Offline Mode Not Working
- Ensure HTTPS (required for service workers)
- Check browser console for errors
- Verify service worker registration

### CORS Errors
- Update `CORS_ORIGIN` in `config.php`
- Check web server CORS headers

## 📝 API Documentation

### Authentication
```bash
POST /api/auth/login
Body: { "username": "admin", "password": "admin123" }
Response: { "token": "...", "user": {...} }
```

### Assessments
```bash
GET    /api/assessments          # List all
GET    /api/assessments/:id      # Get one
POST   /api/assessments          # Create
PUT    /api/assessments/:id      # Update
DELETE /api/assessments/:id      # Delete
```

Full API documentation: `docs/API.md`

## 🚀 Deployment

### Production Checklist
- [ ] Change default admin password
- [ ] Update `config.php` with production values
- [ ] Set `display_errors = 0` in PHP
- [ ] Configure HTTPS/SSL
- [ ] Set up database backups
- [ ] Configure email server
- [ ] Set proper file permissions
- [ ] Enable error logging
- [ ] Update CORS settings
- [ ] Configure firewall rules

### Environment Variables
Set in production:
- `DB_HOST`
- `DB_NAME`
- `DB_USER`
- `DB_PASS`
- `JWT_SECRET`
- `API_URL`

## 📄 License

This project is provided as-is for evaluation and implementation.

## 🤝 Support

For issues, questions, or feature requests:
1. Check documentation
2. Review troubleshooting section
3. Check system logs
4. Contact system administrator

## 📈 Version History

### v2.0.0 (January 2026)
- Complete rewrite with React
- PWA capabilities
- Offline mode
- Dark mode
- Enhanced security
- Responsive design
- PDF export
- 16-table database
- Full API documentation

## 🎯 Roadmap

- [ ] Multi-language support
- [ ] Advanced reporting
- [ ] Email notifications
- [ ] Mobile app (React Native)
- [ ] Integration APIs
- [ ] Advanced analytics
- [ ] Document management
- [ ] Workflow automation

---

**Built with ❤️ using React, PHP, and MySQL**

**Version**: 2.0.0  
**Last Updated**: January 2026  
**Status**: Production Ready
