# API Documentation - Risk Assessment System v2.0.0

## Base URL
```
http://your-domain.com/api
```

## Authentication

All endpoints except `/auth/login` and `/auth/register` require authentication via JWT token.

### Headers
```
Authorization: Bearer {token}
Content-Type: application/json
```

---

## Authentication Endpoints

### Login
```http
POST /auth/login
```

**Request Body:**
```json
{
  "username": "admin",
  "password": "admin123"
}
```

**Response:**
```json
{
  "success": true,
  "data": {
    "token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
    "user": {
      "id": 1,
      "username": "admin",
      "email": "admin@example.com",
      "full_name": "System Administrator",
      "role": "admin"
    }
  },
  "timestamp": "2026-01-14 10:30:00"
}
```

### Register
```http
POST /auth/register
```

**Request Body:**
```json
{
  "username": "newuser",
  "email": "user@example.com",
  "password": "securepassword",
  "full_name": "John Doe",
  "role": "assessor"
}
```

---

## User Endpoints

### Get All Users
```http
GET /users?page=1&page_size=20&search=john&role=assessor
```

**Query Parameters:**
- `page` (optional): Page number (default: 1)
- `page_size` (optional): Items per page (default: 20, max: 100)
- `search` (optional): Search in username, email, or full name
- `role` (optional): Filter by role (admin, assessor, viewer)
- `is_active` (optional): Filter by active status (0 or 1)

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "username": "admin",
      "email": "admin@example.com",
      "full_name": "System Administrator",
      "role": "admin",
      "is_active": 1,
      "last_login": "2026-01-14 10:30:00",
      "created_at": "2026-01-01 00:00:00"
    }
  ],
  "pagination": {
    "total": 50,
    "page": 1,
    "page_size": 20,
    "total_pages": 3
  }
}
```

### Get User by ID
```http
GET /users/{id}
```

### Create User
```http
POST /users
```

**Request Body:**
```json
{
  "username": "newuser",
  "email": "user@example.com",
  "password": "securepass",
  "full_name": "Jane Smith",
  "role": "assessor"
}
```

### Update User
```http
PUT /users/{id}
```

**Request Body:**
```json
{
  "email": "newemail@example.com",
  "full_name": "Jane Smith Updated",
  "role": "admin",
  "is_active": 1,
  "password": "newpassword"
}
```

### Delete User
```http
DELETE /users/{id}
```

---

## Location Endpoints

### Get All Locations
```http
GET /locations?page=1&search=office
```

### Get Location by ID
```http
GET /locations/{id}
```

### Create Location
```http
POST /locations
```

**Request Body:**
```json
{
  "name": "Main Office",
  "address": "123 Business St",
  "city": "New York",
  "state": "NY",
  "postal_code": "10001",
  "country": "USA",
  "contact_name": "John Manager",
  "contact_email": "manager@company.com",
  "contact_phone": "+1-555-0123"
}
```

### Update Location
```http
PUT /locations/{id}
```

### Delete Location
```http
DELETE /locations/{id}
```

---

## Assessment Endpoints

### Get All Assessments
```http
GET /assessments?page=1&status=completed&location_id=5
```

**Query Parameters:**
- `page`: Page number
- `page_size`: Items per page
- `status`: Filter by status (draft, in_progress, completed, archived)
- `location_id`: Filter by location

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "assessment_number": "RA-2026-000001",
      "title": "Office Safety Assessment",
      "description": "Annual safety review",
      "location_id": 5,
      "location_name": "Main Office",
      "assessor_id": 2,
      "assessor_name": "John Assessor",
      "assessment_date": "2026-01-14",
      "status": "completed",
      "overall_risk_level": "medium",
      "notes": "Some concerns noted",
      "created_at": "2026-01-10 09:00:00",
      "updated_at": "2026-01-14 15:30:00"
    }
  ],
  "pagination": {
    "total": 150,
    "page": 1,
    "page_size": 20,
    "total_pages": 8
  }
}
```

### Get Assessment by ID
```http
GET /assessments/{id}
```

### Create Assessment
```http
POST /assessments
```

**Request Body:**
```json
{
  "location_id": 5,
  "title": "Quarterly Safety Review",
  "description": "Q1 2026 safety assessment",
  "assessment_date": "2026-01-14",
  "status": "draft",
  "assessor_id": 2
}
```

**Response:**
```json
{
  "success": true,
  "data": {
    "id": 25,
    "assessment_number": "RA-2026-000025"
  },
  "message": "Assessment created"
}
```

### Update Assessment
```http
PUT /assessments/{id}
```

**Request Body:**
```json
{
  "title": "Updated Title",
  "status": "in_progress",
  "overall_risk_level": "high",
  "notes": "Updated notes"
}
```

### Delete Assessment
```http
DELETE /assessments/{id}
```

### Get Assessment Risks
```http
GET /assessments/{id}/risks
```

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "assessment_id": 1,
      "risk_item_id": 5,
      "risk_item_name": "Slippery floors",
      "category": "Slip, Trip and Fall",
      "likelihood": 3,
      "severity": 4,
      "risk_score": 12,
      "risk_level": "high",
      "control_measures": "Install anti-slip mats",
      "notes": "Priority issue"
    }
  ]
}
```

### Add Risk to Assessment
```http
POST /assessments/{id}/risks
```

**Request Body:**
```json
{
  "risk_item_id": 5,
  "likelihood": 3,
  "severity": 4,
  "control_measures": "Install anti-slip mats",
  "notes": "Immediate attention required"
}
```

### Export Assessment as PDF
```http
GET /assessments/{id}/export
```

Returns PDF file download.

---

## Risk Item Endpoints

### Get All Risk Items
```http
GET /risks?category_id=1
```

### Get Risk Categories
```http
GET /risks/categories
```

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "name": "Physical Hazards",
      "description": "Risks related to physical environment",
      "display_order": 1
    }
  ]
}
```

---

## Action Item Endpoints

### Get All Action Items
```http
GET /action-items?assessment_id=5&status=pending
```

**Query Parameters:**
- `assessment_id`: Filter by assessment
- `status`: Filter by status (pending, in_progress, completed, cancelled)
- `priority`: Filter by priority (low, medium, high, critical)

### Create Action Item
```http
POST /action-items
```

**Request Body:**
```json
{
  "assessment_id": 5,
  "assessment_risk_id": 10,
  "title": "Install safety signage",
  "description": "Place wet floor signs",
  "responsible_person": "John Maintenance",
  "due_date": "2026-02-01",
  "priority": "high"
}
```

### Update Action Item
```http
PUT /action-items/{id}
```

**Request Body:**
```json
{
  "status": "completed",
  "completed_date": "2026-01-20",
  "notes": "Completed ahead of schedule"
}
```

---

## Settings Endpoints

### Get All Settings
```http
GET /settings
```

**Response:**
```json
{
  "success": true,
  "data": {
    "system_name": "Risk Assessment System",
    "company_name": "ACME Corporation",
    "date_format": "Y-m-d",
    "timezone": "America/New_York",
    "theme": "light"
  }
}
```

### Get Single Setting
```http
GET /settings/{key}
```

### Update Setting
```http
PUT /settings/{key}
```

**Request Body:**
```json
{
  "value": "New Value"
}
```

---

## Dashboard Endpoints

### Get Dashboard Stats
```http
GET /dashboard
```

**Response:**
```json
{
  "success": true,
  "data": {
    "stats": {
      "total_assessments": 150,
      "completed": 95,
      "in_progress": 35,
      "draft": 20,
      "high_risk_count": 12
    },
    "recent_assessments": [...]
  }
}
```

---

## Report Endpoints

### Generate Report
```http
GET /reports/{type}?start_date=2026-01-01&end_date=2026-01-31
```

**Report Types:**
- `assessments`: Assessment summary
- `locations`: Location-based report
- `risks`: Risk analysis
- `actions`: Action items report

**Query Parameters:**
- `start_date`: Filter from date
- `end_date`: Filter to date
- `location_id`: Filter by location
- `status`: Filter by status

---

## Sync Endpoint (Offline Mode)

### Sync Offline Changes
```http
POST /sync
```

**Request Body:**
```json
{
  "items": [
    {
      "action": "create",
      "entity_type": "assessments",
      "entity_id": null,
      "data": {
        "title": "Offline Assessment",
        "location_id": 5
      }
    }
  ]
}
```

---

## Error Responses

### Standard Error Format
```json
{
  "success": false,
  "message": "Error description",
  "errors": {
    "field_name": "Error detail"
  },
  "timestamp": "2026-01-14 10:30:00"
}
```

### HTTP Status Codes
- `200` - Success
- `201` - Created
- `400` - Bad Request
- `401` - Unauthorized
- `403` - Forbidden
- `404` - Not Found
- `500` - Internal Server Error

---

## Rate Limiting

API requests are limited to:
- **Authenticated users**: 1000 requests per hour
- **Public endpoints**: 100 requests per hour

Rate limit headers:
```
X-RateLimit-Limit: 1000
X-RateLimit-Remaining: 995
X-RateLimit-Reset: 1642176000
```

---

## Pagination

All list endpoints support pagination:

**Request:**
```http
GET /assessments?page=2&page_size=50
```

**Response includes:**
```json
{
  "pagination": {
    "total": 150,
    "page": 2,
    "page_size": 50,
    "total_pages": 3
  }
}
```

---

## Filtering & Sorting

### Common Query Parameters
- `search`: Text search
- `sort`: Sort field
- `order`: Sort order (asc, desc)
- `filter[field]`: Filter by field value

**Example:**
```http
GET /assessments?search=office&sort=created_at&order=desc&filter[status]=completed
```

---

## Best Practices

### Request Best Practices
1. Always include `Authorization` header
2. Use appropriate HTTP methods
3. Include `Content-Type: application/json`
4. Handle rate limiting
5. Implement retry logic

### Error Handling
```javascript
try {
  const response = await fetch('/api/assessments');
  if (!response.ok) {
    const error = await response.json();
    console.error('API Error:', error.message);
  }
  const data = await response.json();
} catch (error) {
  console.error('Network Error:', error);
}
```

### Authentication
```javascript
// Store token after login
localStorage.setItem('token', token);

// Include in requests
fetch('/api/assessments', {
  headers: {
    'Authorization': `Bearer ${localStorage.getItem('token')}`,
    'Content-Type': 'application/json'
  }
});
```

---

## Webhooks (Ready for Implementation)

System supports webhooks for real-time notifications:

**Events:**
- `assessment.created`
- `assessment.updated`
- `assessment.completed`
- `action_item.overdue`
- `user.created`

Configure in: Settings → Integrations → Webhooks

---

## API Versioning

Current version: `v1`

Future versions will use URL versioning:
```
/api/v2/assessments
```

---

## Support

For API support:
- Check error messages
- Review API logs
- Verify authentication token
- Check request payload format
- Ensure proper permissions

**API Status**: Production Ready ✅
**Last Updated**: January 2026
**Version**: 2.0.0
