<?php
/**
 * User Controller
 * Handles all user-related operations
 */

class UserController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    /**
     * User login
     */
    public function login($data) {
        Response::validateRequired($data, ['username', 'password']);

        $sql = "SELECT * FROM users WHERE username = ? AND is_active = 1";
        $user = $this->db->queryOne($sql, [$data['username']]);

        if (!$user || !$this->auth->verifyPassword($data['password'], $user['password_hash'])) {
            Response::error('Invalid credentials', 401);
        }

        // Update last login
        $this->db->execute("UPDATE users SET last_login = NOW() WHERE id = ?", [$user['id']]);

        // Generate token
        $token = $this->auth->generateToken($user);

        // Remove password from response
        unset($user['password_hash']);

        Response::success([
            'token' => $token,
            'user' => $user
        ], 'Login successful');
    }

    /**
     * User registration
     */
    public function register($data) {
        Response::validateRequired($data, ['username', 'email', 'password', 'full_name']);

        // Check if username or email already exists
        $sql = "SELECT id FROM users WHERE username = ? OR email = ?";
        $existing = $this->db->queryOne($sql, [$data['username'], $data['email']]);

        if ($existing) {
            Response::error('Username or email already exists', 400);
        }

        // Validate password length
        if (strlen($data['password']) < PASSWORD_MIN_LENGTH) {
            Response::error('Password must be at least ' . PASSWORD_MIN_LENGTH . ' characters', 400);
        }

        // Hash password
        $passwordHash = $this->auth->hashPassword($data['password']);

        // Insert user
        $sql = "INSERT INTO users (username, email, password_hash, full_name, role) 
                VALUES (?, ?, ?, ?, ?)";
        $result = $this->db->execute($sql, [
            $data['username'],
            $data['email'],
            $passwordHash,
            $data['full_name'],
            $data['role'] ?? 'viewer'
        ]);

        if ($result['success']) {
            $this->auth->logAudit('create', 'users', $result['last_insert_id']);
            Response::success(['id' => $result['last_insert_id']], 'User created successfully', 201);
        } else {
            Response::error('Failed to create user', 500);
        }
    }

    /**
     * Get all users
     */
    public function getAll($params) {
        if (!$this->auth->checkPermission('users', 'read')) {
            Response::error('Insufficient permissions', 403);
        }

        $page = $params['page'] ?? 1;
        $pageSize = min($params['page_size'] ?? DEFAULT_PAGE_SIZE, MAX_PAGE_SIZE);
        $offset = ($page - 1) * $pageSize;

        $where = [];
        $queryParams = [];

        // Search filter
        if (!empty($params['search'])) {
            $where[] = "(username LIKE ? OR email LIKE ? OR full_name LIKE ?)";
            $searchTerm = "%{$params['search']}%";
            $queryParams = array_merge($queryParams, [$searchTerm, $searchTerm, $searchTerm]);
        }

        // Role filter
        if (!empty($params['role'])) {
            $where[] = "role = ?";
            $queryParams[] = $params['role'];
        }

        // Active filter
        if (isset($params['is_active'])) {
            $where[] = "is_active = ?";
            $queryParams[] = $params['is_active'];
        }

        $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

        // Get total count
        $countSql = "SELECT COUNT(*) as total FROM users $whereClause";
        $totalResult = $this->db->queryOne($countSql, $queryParams);
        $total = $totalResult['total'];

        // Get users
        $sql = "SELECT id, username, email, full_name, role, is_active, last_login, created_at 
                FROM users $whereClause 
                ORDER BY created_at DESC 
                LIMIT ? OFFSET ?";
        $queryParams[] = $pageSize;
        $queryParams[] = $offset;

        $users = $this->db->query($sql, $queryParams);

        Response::paginated($users, $total, $page, $pageSize);
    }

    /**
     * Get user by ID
     */
    public function getById($id) {
        if (!$this->auth->checkPermission('users', 'read')) {
            Response::error('Insufficient permissions', 403);
        }

        $sql = "SELECT id, username, email, full_name, role, is_active, last_login, created_at 
                FROM users WHERE id = ?";
        $user = $this->db->queryOne($sql, [$id]);

        if (!$user) {
            Response::error('User not found', 404);
        }

        Response::success($user);
    }

    /**
     * Update user
     */
    public function update($id, $data) {
        if (!$this->auth->checkPermission('users', 'update')) {
            Response::error('Insufficient permissions', 403);
        }

        // Get old values for audit
        $oldUser = $this->db->queryOne("SELECT * FROM users WHERE id = ?", [$id]);
        if (!$oldUser) {
            Response::error('User not found', 404);
        }

        $updates = [];
        $params = [];

        if (isset($data['email'])) {
            $updates[] = "email = ?";
            $params[] = $data['email'];
        }
        if (isset($data['full_name'])) {
            $updates[] = "full_name = ?";
            $params[] = $data['full_name'];
        }
        if (isset($data['role'])) {
            $updates[] = "role = ?";
            $params[] = $data['role'];
        }
        if (isset($data['is_active'])) {
            $updates[] = "is_active = ?";
            $params[] = $data['is_active'];
        }
        if (isset($data['password']) && $data['password'] !== '') {
            $updates[] = "password_hash = ?";
            $params[] = $this->auth->hashPassword($data['password']);
        }

        if (empty($updates)) {
            Response::error('No fields to update', 400);
        }

        $params[] = $id;
        $sql = "UPDATE users SET " . implode(", ", $updates) . " WHERE id = ?";
        $result = $this->db->execute($sql, $params);

        if ($result['success']) {
            $this->auth->logAudit('update', 'users', $id, $oldUser, $data);
            Response::success(['id' => $id], 'User updated successfully');
        } else {
            Response::error('Failed to update user', 500);
        }
    }

    /**
     * Delete user
     */
    public function delete($id) {
        if (!$this->auth->checkPermission('users', 'delete')) {
            Response::error('Insufficient permissions', 403);
        }

        $user = $this->db->queryOne("SELECT * FROM users WHERE id = ?", [$id]);
        if (!$user) {
            Response::error('User not found', 404);
        }

        $sql = "DELETE FROM users WHERE id = ?";
        $result = $this->db->execute($sql, [$id]);

        if ($result['success']) {
            $this->auth->logAudit('delete', 'users', $id, $user);
            Response::success(null, 'User deleted successfully');
        } else {
            Response::error('Failed to delete user', 500);
        }
    }

    /**
     * Get current user profile
     */
    public function getProfile() {
        $user = $GLOBALS['current_user'];
        Response::success($user);
    }

    /**
     * Update current user profile
     */
    public function updateProfile($data) {
        $userId = $GLOBALS['current_user']['id'];
        $this->update($userId, $data);
    }
}
