<?php
/**
 * Assessment Controller
 */

class AssessmentController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    public function getAll($params) {
        if (!$this->auth->checkPermission('assessments', 'read')) {
            Response::error('Insufficient permissions', 403);
        }

        $page = $params['page'] ?? 1;
        $pageSize = min($params['page_size'] ?? DEFAULT_PAGE_SIZE, MAX_PAGE_SIZE);
        $offset = ($page - 1) * $pageSize;

        $where = [];
        $queryParams = [];

        if (!empty($params['status'])) {
            $where[] = "a.status = ?";
            $queryParams[] = $params['status'];
        }

        if (!empty($params['location_id'])) {
            $where[] = "a.location_id = ?";
            $queryParams[] = $params['location_id'];
        }

        $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

        $countSql = "SELECT COUNT(*) as total FROM assessments a $whereClause";
        $totalResult = $this->db->queryOne($countSql, $queryParams);
        $total = $totalResult['total'];

        $sql = "SELECT a.*, l.name as location_name, u.full_name as assessor_name 
                FROM assessments a
                LEFT JOIN locations l ON a.location_id = l.id
                LEFT JOIN users u ON a.assessor_id = u.id
                $whereClause
                ORDER BY a.created_at DESC
                LIMIT ? OFFSET ?";
        
        $queryParams[] = $pageSize;
        $queryParams[] = $offset;

        $assessments = $this->db->query($sql, $queryParams);
        Response::paginated($assessments, $total, $page, $pageSize);
    }

    public function getById($id) {
        if (!$this->auth->checkPermission('assessments', 'read')) {
            Response::error('Insufficient permissions', 403);
        }

        $sql = "SELECT a.*, l.name as location_name, u.full_name as assessor_name 
                FROM assessments a
                LEFT JOIN locations l ON a.location_id = l.id
                LEFT JOIN users u ON a.assessor_id = u.id
                WHERE a.id = ?";
        
        $assessment = $this->db->queryOne($sql, [$id]);

        if (!$assessment) {
            Response::error('Assessment not found', 404);
        }

        Response::success($assessment);
    }

    public function create($data) {
        if (!$this->auth->checkPermission('assessments', 'create')) {
            Response::error('Insufficient permissions', 403);
        }

        Response::validateRequired($data, ['location_id', 'title', 'assessment_date']);

        // Generate assessment number
        $year = date('Y');
        $sql = "SELECT COUNT(*) as count FROM assessments WHERE assessment_number LIKE ?";
        $result = $this->db->queryOne($sql, ["RA-$year-%"]);
        $number = str_pad($result['count'] + 1, 6, '0', STR_PAD_LEFT);
        $assessmentNumber = "RA-$year-$number";

        $user = $GLOBALS['current_user'];

        $sql = "INSERT INTO assessments (assessment_number, location_id, title, description, assessor_id, assessment_date, status, created_by) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        
        $result = $this->db->execute($sql, [
            $assessmentNumber,
            $data['location_id'],
            $data['title'],
            $data['description'] ?? null,
            $data['assessor_id'] ?? $user['id'],
            $data['assessment_date'],
            $data['status'] ?? 'draft',
            $user['id']
        ]);

        if ($result['success']) {
            $this->auth->logAudit('create', 'assessments', $result['last_insert_id']);
            Response::success(['id' => $result['last_insert_id'], 'assessment_number' => $assessmentNumber], 'Assessment created', 201);
        }
    }

    public function update($id, $data) {
        if (!$this->auth->checkPermission('assessments', 'update')) {
            Response::error('Insufficient permissions', 403);
        }

        $oldData = $this->db->queryOne("SELECT * FROM assessments WHERE id = ?", [$id]);
        if (!$oldData) {
            Response::error('Assessment not found', 404);
        }

        $updates = [];
        $params = [];

        $fields = ['title', 'description', 'assessment_date', 'status', 'overall_risk_level', 'notes'];
        foreach ($fields as $field) {
            if (isset($data[$field])) {
                $updates[] = "$field = ?";
                $params[] = $data[$field];
            }
        }

        if (empty($updates)) {
            Response::error('No fields to update', 400);
        }

        $params[] = $id;
        $sql = "UPDATE assessments SET " . implode(", ", $updates) . " WHERE id = ?";
        $result = $this->db->execute($sql, $params);

        if ($result['success']) {
            $this->auth->logAudit('update', 'assessments', $id, $oldData, $data);
            Response::success(['id' => $id], 'Assessment updated');
        }
    }

    public function delete($id) {
        if (!$this->auth->checkPermission('assessments', 'delete')) {
            Response::error('Insufficient permissions', 403);
        }

        $assessment = $this->db->queryOne("SELECT * FROM assessments WHERE id = ?", [$id]);
        if (!$assessment) {
            Response::error('Assessment not found', 404);
        }

        $sql = "DELETE FROM assessments WHERE id = ?";
        $result = $this->db->execute($sql, [$id]);

        if ($result['success']) {
            $this->auth->logAudit('delete', 'assessments', $id, $assessment);
            Response::success(null, 'Assessment deleted');
        }
    }

    public function getDashboardStats() {
        $sql = "SELECT 
                    COUNT(*) as total_assessments,
                    SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                    SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress,
                    SUM(CASE WHEN status = 'draft' THEN 1 ELSE 0 END) as draft
                FROM assessments";
        
        $stats = $this->db->queryOne($sql);

        $sql = "SELECT COUNT(*) as high_risk_count 
                FROM assessment_risks WHERE risk_level = 'high'";
        $riskStats = $this->db->queryOne($sql);

        $sql = "SELECT a.*, l.name as location_name 
                FROM assessments a
                LEFT JOIN locations l ON a.location_id = l.id
                ORDER BY a.created_at DESC LIMIT 5";
        $recentAssessments = $this->db->query($sql);

        Response::success([
            'stats' => array_merge($stats, $riskStats),
            'recent_assessments' => $recentAssessments
        ]);
    }
}
