<?php
/**
 * API Router - Main entry point for all API requests
 * Routes requests to appropriate controllers
 */

require_once 'config.php';
require_once 'Database.php';
require_once 'Auth.php';
require_once 'Response.php';

// Controllers
require_once 'controllers/UserController.php';
require_once 'controllers/LocationController.php';
require_once 'controllers/AssessmentController.php';
require_once 'controllers/RiskController.php';
require_once 'controllers/ActionItemController.php';
require_once 'controllers/SettingsController.php';
require_once 'controllers/ReportController.php';

// Set CORS headers
header('Access-Control-Allow-Origin: ' . CORS_ORIGIN);
header('Access-Control-Allow-Methods: ' . CORS_METHODS);
header('Access-Control-Allow-Headers: ' . CORS_HEADERS);
header('Content-Type: application/json');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Get request method and path
$method = $_SERVER['REQUEST_METHOD'];
$path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$path = str_replace('/api', '', $path);
$pathParts = array_filter(explode('/', $path));
$pathParts = array_values($pathParts);

// Get request body for POST/PUT
$requestBody = file_get_contents('php://input');
$data = json_decode($requestBody, true) ?? [];

// Routing
try {
    // Public routes (no authentication required)
    if ($pathParts[0] === 'auth') {
        if ($pathParts[1] === 'login' && $method === 'POST') {
            $controller = new UserController();
            $controller->login($data);
            exit();
        }
        if ($pathParts[1] === 'register' && $method === 'POST') {
            $controller = new UserController();
            $controller->register($data);
            exit();
        }
    }

    // Protected routes (require authentication)
    $auth = new Auth();
    $user = $auth->authenticate();

    // Route to appropriate controller
    switch ($pathParts[0]) {
        case 'users':
            $controller = new UserController();
            handleCrudRoutes($controller, $pathParts, $method, $data);
            break;

        case 'locations':
            $controller = new LocationController();
            handleCrudRoutes($controller, $pathParts, $method, $data);
            break;

        case 'assessments':
            $controller = new AssessmentController();
            handleAssessmentRoutes($controller, $pathParts, $method, $data);
            break;

        case 'risks':
            $controller = new RiskController();
            handleCrudRoutes($controller, $pathParts, $method, $data);
            break;

        case 'action-items':
            $controller = new ActionItemController();
            handleCrudRoutes($controller, $pathParts, $method, $data);
            break;

        case 'settings':
            $controller = new SettingsController();
            handleSettingsRoutes($controller, $pathParts, $method, $data);
            break;

        case 'reports':
            $controller = new ReportController();
            handleReportRoutes($controller, $pathParts, $method, $data);
            break;

        case 'dashboard':
            $controller = new AssessmentController();
            if ($method === 'GET') {
                $controller->getDashboardStats();
            }
            break;

        case 'sync':
            // Handle offline sync
            if ($method === 'POST') {
                handleSync($data, $user);
            }
            break;

        default:
            Response::error('Endpoint not found', 404);
    }

} catch (Exception $e) {
    Response::error($e->getMessage(), 500);
}

/**
 * Handle standard CRUD routes
 */
function handleCrudRoutes($controller, $pathParts, $method, $data) {
    $id = $pathParts[1] ?? null;

    switch ($method) {
        case 'GET':
            if ($id) {
                $controller->getById($id);
            } else {
                $controller->getAll($_GET);
            }
            break;
        case 'POST':
            $controller->create($data);
            break;
        case 'PUT':
            if ($id) {
                $controller->update($id, $data);
            }
            break;
        case 'DELETE':
            if ($id) {
                $controller->delete($id);
            }
            break;
        default:
            Response::error('Method not allowed', 405);
    }
}

/**
 * Handle assessment-specific routes
 */
function handleAssessmentRoutes($controller, $pathParts, $method, $data) {
    $id = $pathParts[1] ?? null;
    $action = $pathParts[2] ?? null;

    if ($id && $action === 'risks') {
        // Handle assessment risks
        if ($method === 'GET') {
            $controller->getAssessmentRisks($id);
        } elseif ($method === 'POST') {
            $controller->addRisk($id, $data);
        }
    } elseif ($id && $action === 'export') {
        // Export assessment as PDF
        if ($method === 'GET') {
            $controller->exportPdf($id);
        }
    } else {
        handleCrudRoutes($controller, $pathParts, $method, $data);
    }
}

/**
 * Handle settings routes
 */
function handleSettingsRoutes($controller, $pathParts, $method, $data) {
    $key = $pathParts[1] ?? null;

    if ($method === 'GET') {
        if ($key) {
            $controller->getSetting($key);
        } else {
            $controller->getAllSettings();
        }
    } elseif ($method === 'PUT' && $key) {
        $controller->updateSetting($key, $data);
    }
}

/**
 * Handle report routes
 */
function handleReportRoutes($controller, $pathParts, $method, $data) {
    $reportType = $pathParts[1] ?? null;

    if ($method === 'GET' && $reportType) {
        $controller->generateReport($reportType, $_GET);
    }
}

/**
 * Handle offline sync
 */
function handleSync($data, $user) {
    $db = Database::getInstance();
    
    try {
        $db->beginTransaction();
        
        foreach ($data['items'] ?? [] as $item) {
            // Process each sync item
            $sql = "INSERT INTO sync_queue (user_id, action, entity_type, entity_id, data, status) 
                    VALUES (?, ?, ?, ?, ?, 'pending')";
            $db->execute($sql, [
                $user['id'],
                $item['action'],
                $item['entity_type'],
                $item['entity_id'] ?? null,
                json_encode($item['data'])
            ]);
        }
        
        $db->commit();
        Response::success(['message' => 'Sync queued successfully']);
    } catch (Exception $e) {
        $db->rollback();
        throw $e;
    }
}
