<?php
/**
 * Authentication Handler
 * Manages JWT tokens and user authentication
 */

class Auth {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Authenticate user from request headers
     */
    public function authenticate() {
        $headers = getallheaders();
        $token = null;

        // Get token from Authorization header
        if (isset($headers['Authorization'])) {
            $matches = [];
            if (preg_match('/Bearer\s+(.*)$/i', $headers['Authorization'], $matches)) {
                $token = $matches[1];
            }
        }

        if (!$token) {
            Response::error('No token provided', 401);
            exit();
        }

        // Verify token
        $user = $this->verifyToken($token);
        if (!$user) {
            Response::error('Invalid or expired token', 401);
            exit();
        }

        // Store user in global context
        $GLOBALS['current_user'] = $user;
        return $user;
    }

    /**
     * Generate JWT token
     */
    public function generateToken($user) {
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        $payload = json_encode([
            'user_id' => $user['id'],
            'username' => $user['username'],
            'email' => $user['email'],
            'role' => $user['role'],
            'exp' => time() + SESSION_LIFETIME
        ]);

        $base64UrlHeader = $this->base64UrlEncode($header);
        $base64UrlPayload = $this->base64UrlEncode($payload);
        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, JWT_SECRET, true);
        $base64UrlSignature = $this->base64UrlEncode($signature);

        return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
    }

    /**
     * Verify JWT token
     */
    public function verifyToken($token) {
        $tokenParts = explode('.', $token);
        if (count($tokenParts) !== 3) {
            return false;
        }

        $header = base64_decode($tokenParts[0]);
        $payload = base64_decode($tokenParts[1]);
        $signatureProvided = $tokenParts[2];

        // Verify signature
        $base64UrlHeader = $this->base64UrlEncode($header);
        $base64UrlPayload = $this->base64UrlEncode($payload);
        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, JWT_SECRET, true);
        $base64UrlSignature = $this->base64UrlEncode($signature);

        if ($base64UrlSignature !== $signatureProvided) {
            return false;
        }

        // Decode payload
        $payload = json_decode(base64_decode($tokenParts[1]), true);

        // Check expiration
        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return false;
        }

        // Get user from database
        $sql = "SELECT id, username, email, full_name, role, is_active 
                FROM users WHERE id = ? AND is_active = 1";
        $user = $this->db->queryOne($sql, [$payload['user_id']]);

        return $user ?: false;
    }

    /**
     * Check if user has permission
     */
    public function checkPermission($resource, $action) {
        $user = $GLOBALS['current_user'] ?? null;
        if (!$user) {
            return false;
        }

        // Admin has all permissions
        if ($user['role'] === 'admin') {
            return true;
        }

        // Check specific permission
        $sql = "SELECT can_$action FROM permissions WHERE role = ? AND resource = ?";
        $permission = $this->db->queryOne($sql, [$user['role'], $resource]);

        return $permission && $permission["can_$action"] == 1;
    }

    /**
     * Verify password
     */
    public function verifyPassword($password, $hash) {
        return password_verify($password, $hash);
    }

    /**
     * Hash password
     */
    public function hashPassword($password) {
        return password_hash($password, PASSWORD_DEFAULT);
    }

    /**
     * Base64 URL encode
     */
    private function base64UrlEncode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    /**
     * Log audit action
     */
    public function logAudit($action, $entityType, $entityId, $oldValues = null, $newValues = null) {
        $user = $GLOBALS['current_user'] ?? null;
        $userId = $user['id'] ?? null;

        $sql = "INSERT INTO audit_log (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        
        $this->db->execute($sql, [
            $userId,
            $action,
            $entityType,
            $entityId,
            $oldValues ? json_encode($oldValues) : null,
            $newValues ? json_encode($newValues) : null,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    }
}
