# Installation Checklist - Risk Assessment System v2.0.0

## ✅ Pre-Installation

- [ ] PHP 8.0+ installed
- [ ] MySQL 8.0+ installed
- [ ] Node.js 16+ installed
- [ ] Apache/Nginx web server running
- [ ] mod_rewrite enabled (Apache)
- [ ] Composer installed (optional)

## ✅ Step 1: Extract Files (1 min)

- [ ] Extract zip file to web server directory
- [ ] Verify all files are present
- [ ] Check file permissions

```bash
cd /var/www/html
unzip risk-assessment-system.zip
cd risk-assessment-system
ls -la  # Verify files
```

## ✅ Step 2: Database Setup (2 min)

- [ ] Create MySQL database
- [ ] Import schema.sql
- [ ] Verify tables created (16 tables)
- [ ] Check default data loaded

```bash
mysql -u root -p
CREATE DATABASE risk_assessment_db;
exit

mysql -u root -p risk_assessment_db < database/schema.sql
```

**Verify:**
```sql
USE risk_assessment_db;
SHOW TABLES;  # Should show 16 tables
SELECT * FROM users;  # Should show admin user
```

## ✅ Step 3: Backend Configuration (2 min)

- [ ] Edit backend/config.php
- [ ] Set database credentials
- [ ] Update JWT secret
- [ ] Set proper URLs
- [ ] Create uploads directory

```bash
nano backend/config.php
```

**Update these values:**
```php
define('DB_HOST', 'localhost');
define('DB_NAME', 'risk_assessment_db');
define('DB_USER', 'your_username');
define('DB_PASS', 'your_password');
define('JWT_SECRET', 'your-secret-key-here');  # IMPORTANT: Change this!
```

**Create directories:**
```bash
mkdir -p backend/uploads
chmod 755 backend/uploads
```

## ✅ Step 4: Frontend Setup (5 min)

- [ ] Install Node.js dependencies
- [ ] Update API URL
- [ ] Build for production
- [ ] Copy build files

```bash
cd frontend

# Install dependencies
npm install

# Edit API URL in src/App.jsx (line ~40)
nano src/App.jsx
# Change: static baseURL = 'http://your-domain.com/api';

# Build production
npm run build

# Copy to web root
cp -r build/* ../public/
```

## ✅ Step 5: Web Server Configuration (2 min)

### Apache
- [ ] Copy .htaccess to web root
- [ ] Enable mod_rewrite
- [ ] Restart Apache

```bash
cp .htaccess /var/www/html/
sudo a2enmod rewrite
sudo service apache2 restart
```

### Nginx
- [ ] Add configuration to nginx.conf
- [ ] Test configuration
- [ ] Reload Nginx

```bash
sudo nano /etc/nginx/sites-available/default
# Add location blocks (see QUICK_START.md)
sudo nginx -t
sudo service nginx reload
```

## ✅ Step 6: File Permissions (1 min)

- [ ] Set proper ownership
- [ ] Set directory permissions
- [ ] Verify uploads directory writable

```bash
# For Apache
sudo chown -R www-data:www-data .
sudo chmod -R 755 .

# For Nginx
sudo chown -R nginx:nginx .
sudo chmod -R 755 .

# Uploads directory
chmod 755 backend/uploads
```

## ✅ Step 7: Testing (3 min)

- [ ] Access application URL
- [ ] Login with default credentials
- [ ] Check dashboard loads
- [ ] Test API endpoints
- [ ] Verify database connection

```bash
# Test database connection
php backend/test-connection.php

# Test API
curl http://your-domain.com/api/settings
```

**Browser tests:**
1. Open: http://your-domain.com
2. Login: admin / admin123
3. View Dashboard
4. Create test assessment
5. Check all pages work

## ✅ Step 8: Security (5 min)

- [ ] Change admin password
- [ ] Update JWT_SECRET
- [ ] Set SECRET_KEY
- [ ] Disable PHP errors in production
- [ ] Configure HTTPS/SSL
- [ ] Set CORS properly

```bash
# Edit config.php
nano backend/config.php

# Update:
define('JWT_SECRET', 'your-unique-secret-here');
define('SECRET_KEY', 'another-unique-secret-here');

# Edit php.ini for production
display_errors = Off
log_errors = On
```

## ✅ Step 9: SSL/HTTPS Setup (5 min)

- [ ] Install SSL certificate
- [ ] Configure HTTPS redirect
- [ ] Update URLs in config
- [ ] Test HTTPS access

```bash
# Using Certbot (Let's Encrypt)
sudo apt install certbot python3-certbot-apache
sudo certbot --apache -d your-domain.com
```

## ✅ Step 10: Final Checks (5 min)

- [ ] All pages accessible
- [ ] Authentication works
- [ ] CRUD operations work
- [ ] File uploads work
- [ ] PDF export works (if enabled)
- [ ] Offline mode works
- [ ] Mobile responsive
- [ ] Dark mode works

## ✅ Post-Installation

- [ ] Create additional users
- [ ] Add locations
- [ ] Configure system settings
- [ ] Set up email (if needed)
- [ ] Configure backups
- [ ] Set up monitoring
- [ ] Train users

## 📝 Configuration Checklist

### Database (config.php)
```php
✓ DB_HOST
✓ DB_NAME
✓ DB_USER
✓ DB_PASS
```

### Security (config.php)
```php
✓ SECRET_KEY (changed from default)
✓ JWT_SECRET (changed from default)
✓ PASSWORD_MIN_LENGTH
```

### Application (config.php)
```php
✓ APP_URL
✓ API_URL
✓ CORS_ORIGIN (production domain)
```

### Frontend (App.jsx)
```javascript
✓ API baseURL updated
✓ Production build created
✓ Files copied to public/
```

## 🔒 Security Checklist

- [ ] Changed default admin password
- [ ] Updated all secret keys
- [ ] Disabled directory browsing
- [ ] Protected sensitive files
- [ ] Configured HTTPS
- [ ] Set proper CORS
- [ ] Enabled security headers
- [ ] Set file upload limits

## 🎯 Default Credentials

**IMPORTANT:** Change these immediately!

```
Username: admin
Password: admin123
Email: admin@riskassessment.local
```

## 📊 Verification Commands

```bash
# Check PHP version
php -v  # Should be 8.0+

# Check MySQL version
mysql --version  # Should be 8.0+

# Check Node version
node -v  # Should be 16+

# Check Apache modules
apache2ctl -M | grep rewrite  # Should show rewrite_module

# Check file permissions
ls -la backend/uploads  # Should be writable

# Check database
mysql -u root -p -e "USE risk_assessment_db; SHOW TABLES;"

# Test API
curl http://your-domain.com/api/settings
```

## 🚨 Troubleshooting

If something doesn't work:

1. **Check error logs**
   ```bash
   tail -f /var/log/apache2/error.log
   tail -f backend/logs/error.log
   ```

2. **Verify file permissions**
   ```bash
   ls -la backend/uploads
   ```

3. **Test database connection**
   ```bash
   php backend/test-connection.php
   ```

4. **Check web server config**
   ```bash
   apache2ctl -t  # Apache
   nginx -t       # Nginx
   ```

5. **Review documentation**
   - docs/README.md
   - docs/QUICK_START.md
   - docs/API.md

## ✅ Success Indicators

You'll know everything works when:

✓ Login page loads
✓ Can log in with admin credentials
✓ Dashboard shows stats
✓ Can create new assessment
✓ All pages accessible
✓ No console errors
✓ API calls succeed
✓ Mobile view works

## 📞 Need Help?

1. Check documentation in `docs/` folder
2. Review error logs
3. Verify all steps completed
4. Check prerequisites met
5. Ensure proper permissions

## 🎉 Installation Complete!

Once all items are checked:
- [ ] All checklist items completed
- [ ] System tested and working
- [ ] Security configured
- [ ] Users can log in
- [ ] Ready for production use

**Congratulations! Your Risk Assessment System is ready to use!** 🚀

---

**Installation Time**: ~30 minutes for full setup
**Difficulty**: Intermediate
**Status**: Production Ready

**Next Steps:**
1. Change admin password
2. Create your first location
3. Create your first assessment
4. Invite team members
5. Start assessing risks!
