<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$error = '';
$success = '';
$active_tab = $_GET['tab'] ?? 'vehicles';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $table = $_POST['table']; // 'vehicle_types' or 'delivery_volumes'
        $name_field = $table === 'vehicle_types' ? 'type_name' : 'volume_name';
        
        switch ($_POST['action']) {
            case 'add':
                $stmt = $pdo->prepare("
                    INSERT INTO $table ($name_field, description, display_order, is_active)
                    VALUES (?, ?, ?, 1)
                ");
                $stmt->execute([
                    $_POST['name'],
                    $_POST['description'] ?? '',
                    $_POST['display_order'] ?? 0
                ]);
                $success = ucfirst(str_replace('_', ' ', rtrim($table, 's'))) . ' added successfully!';
                break;
            
            case 'update':
                $stmt = $pdo->prepare("
                    UPDATE $table 
                    SET $name_field = ?, description = ?, display_order = ?, is_active = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $_POST['name'],
                    $_POST['description'] ?? '',
                    $_POST['display_order'] ?? 0,
                    isset($_POST['is_active']) ? 1 : 0,
                    $_POST['item_id']
                ]);
                $success = 'Updated successfully!';
                break;
            
            case 'delete':
                // Check if in use
                $field = $table === 'vehicle_types' ? 'vehicle_type' : 'delivery_volume';
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM assessments WHERE $field = (SELECT $name_field FROM $table WHERE id = ?)");
                $stmt->execute([$_POST['item_id']]);
                $count = $stmt->fetchColumn();
                
                if ($count > 0) {
                    $error = "Cannot delete: This item is used in $count assessment(s). Deactivate it instead.";
                } else {
                    $stmt = $pdo->prepare("DELETE FROM $table WHERE id = ?");
                    $stmt->execute([$_POST['item_id']]);
                    $success = 'Deleted successfully!';
                }
                break;
        }
    } catch (PDOException $e) {
        $error = 'Database error: ' . $e->getMessage();
    }
}

// Fetch vehicle types
$stmt = $pdo->query("SELECT * FROM vehicle_types ORDER BY display_order, type_name");
$vehicle_types = $stmt->fetchAll();

// Fetch delivery volumes
$stmt = $pdo->query("SELECT * FROM delivery_volumes ORDER BY display_order, volume_name");
$delivery_volumes = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vehicle Types & Delivery Volumes - Risk System</title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .tabs {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 1.5rem;
            border-bottom: 2px solid var(--border-color);
        }
        .tab {
            padding: 0.75rem 1.5rem;
            background: none;
            border: none;
            cursor: pointer;
            font-size: 1rem;
            color: var(--text-muted);
            border-bottom: 3px solid transparent;
            transition: all 0.2s;
        }
        .tab:hover {
            color: var(--text-primary);
        }
        .tab.active {
            color: var(--primary);
            border-bottom-color: var(--primary);
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }
        .usage-badge {
            background: var(--bg-tertiary);
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-size: 0.85rem;
            color: var(--text-muted);
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">🚚 Vehicle Types & Delivery Volumes</h1>
                <p class="page-subtitle">Manage dropdown options for assessments</p>
            </div>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <span>✅</span>
                <span><?php echo sanitizeOutput($success); ?></span>
            </div>
        <?php endif; ?>

        <!-- Tabs -->
        <div class="tabs">
            <button class="tab <?php echo $active_tab === 'vehicles' ? 'active' : ''; ?>" 
                    onclick="switchTab('vehicles')">
                🚚 Vehicle Types
            </button>
            <button class="tab <?php echo $active_tab === 'volumes' ? 'active' : ''; ?>" 
                    onclick="switchTab('volumes')">
                📦 Delivery Volumes
            </button>
        </div>

        <!-- Vehicle Types Tab -->
        <div id="vehicles-tab" class="tab-content <?php echo $active_tab === 'vehicles' ? 'active' : ''; ?>">
            <div class="card">
                <div class="card-header" style="display: flex; justify-content: space-between; align-items: center;">
                    <h2 class="card-title">Vehicle Types (<?php echo count($vehicle_types); ?>)</h2>
                    <button onclick="openModal('vehicle')" class="btn btn-primary">➕ Add Vehicle Type</button>
                </div>

                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Vehicle Type</th>
                                <th>Description</th>
                                <th>Order</th>
                                <th>Usage</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($vehicle_types as $type): ?>
                                <?php
                                // Count usage
                                $stmt = $pdo->prepare("SELECT COUNT(*) FROM assessments WHERE vehicle_type = ?");
                                $stmt->execute([$type['type_name']]);
                                $usage = $stmt->fetchColumn();
                                ?>
                                <tr>
                                    <td><strong><?php echo sanitizeOutput($type['type_name']); ?></strong></td>
                                    <td><?php echo sanitizeOutput($type['description']); ?></td>
                                    <td><?php echo $type['display_order']; ?></td>
                                    <td>
                                        <span class="usage-badge">
                                            <?php echo $usage; ?> assessment<?php echo $usage != 1 ? 's' : ''; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($type['is_active']): ?>
                                            <span class="badge badge-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge" style="background: rgba(148, 163, 184, 0.2);">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="btn-group" style="gap: 0.5rem;">
                                            <button onclick='editItem("vehicle", <?php echo htmlspecialchars(json_encode($type)); ?>)' 
                                                    class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                                ✏️ Edit
                                            </button>
                                            <?php if ($usage == 0): ?>
                                                <form method="POST" style="display: inline;" 
                                                      onsubmit="return confirm('Delete this vehicle type?');">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="table" value="vehicle_types">
                                                    <input type="hidden" name="item_id" value="<?php echo $type['id']; ?>">
                                                    <button type="submit" class="btn btn-danger" style="padding: 0.5rem 1rem;">
                                                        🗑️ Delete
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Delivery Volumes Tab -->
        <div id="volumes-tab" class="tab-content <?php echo $active_tab === 'volumes' ? 'active' : ''; ?>">
            <div class="card">
                <div class="card-header" style="display: flex; justify-content: space-between; align-items: center;">
                    <h2 class="card-title">Delivery Volumes (<?php echo count($delivery_volumes); ?>)</h2>
                    <button onclick="openModal('volume')" class="btn btn-primary">➕ Add Delivery Volume</button>
                </div>

                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Delivery Volume</th>
                                <th>Description</th>
                                <th>Order</th>
                                <th>Usage</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($delivery_volumes as $volume): ?>
                                <?php
                                $stmt = $pdo->prepare("SELECT COUNT(*) FROM assessments WHERE delivery_volume = ?");
                                $stmt->execute([$volume['volume_name']]);
                                $usage = $stmt->fetchColumn();
                                ?>
                                <tr>
                                    <td><strong><?php echo sanitizeOutput($volume['volume_name']); ?></strong></td>
                                    <td><?php echo sanitizeOutput($volume['description']); ?></td>
                                    <td><?php echo $volume['display_order']; ?></td>
                                    <td>
                                        <span class="usage-badge">
                                            <?php echo $usage; ?> assessment<?php echo $usage != 1 ? 's' : ''; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($volume['is_active']): ?>
                                            <span class="badge badge-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge" style="background: rgba(148, 163, 184, 0.2);">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="btn-group" style="gap: 0.5rem;">
                                            <button onclick='editItem("volume", <?php echo htmlspecialchars(json_encode($volume)); ?>)' 
                                                    class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                                ✏️ Edit
                                            </button>
                                            <?php if ($usage == 0): ?>
                                                <form method="POST" style="display: inline;" 
                                                      onsubmit="return confirm('Delete this delivery volume?');">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="table" value="delivery_volumes">
                                                    <input type="hidden" name="item_id" value="<?php echo $volume['id']; ?>">
                                                    <button type="submit" class="btn btn-danger" style="padding: 0.5rem 1rem;">
                                                        🗑️ Delete
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Modal -->
    <div id="edit-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 class="modal-title" id="modal-title">Add Item</h2>
                <button type="button" class="modal-close" onclick="window.app.closeModal('edit-modal')">&times;</button>
            </div>

            <form method="POST" id="item-form">
                <input type="hidden" name="action" id="form-action" value="add">
                <input type="hidden" name="table" id="form-table">
                <input type="hidden" name="item_id" id="item-id">

                <div class="form-group">
                    <label class="form-label required" for="name" id="name-label">Name</label>
                    <input type="text" id="name" name="name" class="form-control" required>
                </div>

                <div class="form-group">
                    <label class="form-label" for="description">Description</label>
                    <textarea id="description" name="description" class="form-textarea" rows="3"
                              placeholder="Optional description or notes..."></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label" for="display_order">Display Order</label>
                    <input type="number" id="display_order" name="display_order" class="form-control" 
                           value="0" min="0" max="100">
                    <small style="color: var(--text-muted); display: block; margin-top: 0.25rem;">
                        Lower numbers appear first in dropdowns
                    </small>
                </div>

                <div class="form-group" id="active-group" style="display: none;">
                    <div style="display: flex; align-items: center; gap: 0.5rem;">
                        <input type="checkbox" id="is_active" name="is_active" checked style="width: auto;">
                        <label for="is_active" style="margin: 0; font-weight: 600;">Active</label>
                    </div>
                    <small style="color: var(--text-muted); display: block; margin-top: 0.25rem;">
                        Inactive items won't appear in dropdowns but keep historical data
                    </small>
                </div>

                <div class="btn-group" style="margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">💾 Save</button>
                    <button type="button" class="btn btn-secondary" onclick="window.app.closeModal('edit-modal')">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <script src="js/app.js"></script>
    <script>
        function switchTab(tab) {
            // Update tab buttons
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            event.target.classList.add('active');
            
            // Update tab content
            document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
            document.getElementById(tab + '-tab').classList.add('active');
            
            // Update URL
            history.pushState(null, '', '?tab=' + tab);
        }

        function openModal(type) {
            const isVehicle = type === 'vehicle';
            document.getElementById('modal-title').textContent = 'Add ' + (isVehicle ? 'Vehicle Type' : 'Delivery Volume');
            document.getElementById('form-action').value = 'add';
            document.getElementById('form-table').value = isVehicle ? 'vehicle_types' : 'delivery_volumes';
            document.getElementById('name-label').textContent = isVehicle ? 'Vehicle Type Name' : 'Volume Name';
            document.getElementById('item-form').reset();
            document.getElementById('item-id').value = '';
            document.getElementById('active-group').style.display = 'none';
            window.app.openModal('edit-modal');
        }

        function editItem(type, item) {
            const isVehicle = type === 'vehicle';
            const nameField = isVehicle ? 'type_name' : 'volume_name';
            
            document.getElementById('modal-title').textContent = 'Edit ' + (isVehicle ? 'Vehicle Type' : 'Delivery Volume');
            document.getElementById('form-action').value = 'update';
            document.getElementById('form-table').value = isVehicle ? 'vehicle_types' : 'delivery_volumes';
            document.getElementById('name-label').textContent = isVehicle ? 'Vehicle Type Name' : 'Volume Name';
            document.getElementById('item-id').value = item.id;
            document.getElementById('name').value = item[nameField];
            document.getElementById('description').value = item.description || '';
            document.getElementById('display_order').value = item.display_order;
            document.getElementById('is_active').checked = item.is_active == 1;
            document.getElementById('active-group').style.display = 'block';
            
            window.app.openModal('edit-modal');
        }
    </script>
</body>
</html>
