<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$error = '';
$success = '';
$category_id = $_GET['category'] ?? null;

if (!$category_id) {
    header('Location: risk-categories.php');
    exit;
}

// Get category details
$stmt = $pdo->prepare("SELECT * FROM risk_categories WHERE id = ?");
$stmt->execute([$category_id]);
$category = $stmt->fetch();

if (!$category) {
    header('Location: risk-categories.php');
    exit;
}

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        switch ($_POST['action']) {
            case 'add':
                $stmt = $pdo->prepare("
                    INSERT INTO risk_items (category_id, item_name, description, display_order, is_active)
                    VALUES (?, ?, ?, ?, 1)
                ");
                $stmt->execute([
                    $category_id,
                    $_POST['item_name'],
                    $_POST['description'] ?? '',
                    $_POST['display_order'] ?? 0
                ]);
                $success = 'Risk item added successfully!';
                break;
            
            case 'update':
                $stmt = $pdo->prepare("
                    UPDATE risk_items 
                    SET item_name = ?, description = ?, display_order = ?, is_active = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $_POST['item_name'],
                    $_POST['description'] ?? '',
                    $_POST['display_order'] ?? 0,
                    isset($_POST['is_active']) ? 1 : 0,
                    $_POST['item_id']
                ]);
                $success = 'Risk item updated successfully!';
                break;
            
            case 'delete':
                // Check if item is used in assessments
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM assessment_details WHERE risk_item_id = ?");
                $stmt->execute([$_POST['item_id']]);
                $count = $stmt->fetchColumn();
                
                if ($count > 0) {
                    $error = "Cannot delete: This item is used in $count assessment(s). Deactivate it instead.";
                } else {
                    $stmt = $pdo->prepare("DELETE FROM risk_items WHERE id = ?");
                    $stmt->execute([$_POST['item_id']]);
                    $success = 'Risk item deleted successfully!';
                }
                break;
        }
    } catch (PDOException $e) {
        $error = 'Database error: ' . $e->getMessage();
    }
}

// Fetch risk items for this category
$stmt = $pdo->prepare("
    SELECT ri.*,
           (SELECT COUNT(*) FROM assessment_details WHERE risk_item_id = ri.id) as usage_count
    FROM risk_items ri
    WHERE ri.category_id = ?
    ORDER BY ri.display_order, ri.item_name
");
$stmt->execute([$category_id]);
$risk_items = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Risk Items - <?php echo sanitizeOutput($category['category_name']); ?> - Risk System</title>
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">📋 <?php echo sanitizeOutput($category['category_name']); ?></h1>
                <p class="page-subtitle">Manage risk items in this category</p>
            </div>
            <div class="btn-group">
                <button onclick="openAddItem()" class="btn btn-primary">➕ Add Risk Item</button>
                <a href="risk-categories.php" class="btn btn-secondary">← Back to Categories</a>
            </div>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <span>✅</span>
                <span><?php echo sanitizeOutput($success); ?></span>
            </div>
        <?php endif; ?>

        <!-- Category Info -->
        <div class="card mb-3">
            <div style="padding: 1.5rem; background: var(--bg-tertiary);">
                <h3 style="color: var(--text-primary); margin-bottom: 0.5rem;">
                    Category: <?php echo sanitizeOutput($category['category_name']); ?>
                </h3>
                <?php if ($category['description']): ?>
                    <p style="color: var(--text-secondary); margin: 0;">
                        <?php echo sanitizeOutput($category['description']); ?>
                    </p>
                <?php endif; ?>
            </div>
        </div>

        <!-- Risk Items Table -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Risk Items (<?php echo count($risk_items); ?>)</h2>
            </div>

            <?php if (empty($risk_items)): ?>
                <div style="text-align: center; padding: 4rem 2rem; color: var(--text-muted);">
                    <div style="font-size: 4rem; margin-bottom: 1rem;">📋</div>
                    <h3 style="margin-bottom: 1rem;">No Risk Items Yet</h3>
                    <p style="margin-bottom: 2rem;">
                        Add risk items to this category to use them in assessments.
                    </p>
                    <button onclick="openAddItem()" class="btn btn-primary">Add First Risk Item</button>
                </div>
            <?php else: ?>
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Risk Item</th>
                                <th>Description</th>
                                <th>Display Order</th>
                                <th>Usage</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($risk_items as $item): ?>
                                <tr>
                                    <td><strong><?php echo sanitizeOutput($item['item_name']); ?></strong></td>
                                    <td><?php echo sanitizeOutput($item['description']); ?></td>
                                    <td><?php echo $item['display_order']; ?></td>
                                    <td>
                                        <span style="background: var(--bg-tertiary); padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.85rem;">
                                            <?php echo $item['usage_count']; ?> assessment<?php echo $item['usage_count'] != 1 ? 's' : ''; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($item['is_active']): ?>
                                            <span class="badge badge-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge" style="background: rgba(148, 163, 184, 0.2); color: var(--text-muted);">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="btn-group" style="gap: 0.5rem;">
                                            <button onclick='editItem(<?php echo htmlspecialchars(json_encode($item)); ?>)' 
                                                    class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                                ✏️ Edit
                                            </button>
                                            <?php if ($item['usage_count'] == 0): ?>
                                                <form method="POST" style="display: inline;" 
                                                      onsubmit="return confirm('Delete this risk item?');">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="item_id" value="<?php echo $item['id']; ?>">
                                                    <button type="submit" class="btn btn-danger" style="padding: 0.5rem 1rem;">
                                                        🗑️ Delete
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>

        <!-- Help Section -->
        <div style="margin-top: 2rem; padding: 1.5rem; background: var(--bg-tertiary); border-radius: 8px;">
            <h3 style="color: var(--text-primary); margin-bottom: 1rem;">💡 Tips for Risk Items</h3>
            <ul style="color: var(--text-secondary); line-height: 1.8; margin-left: 1.5rem;">
                <li><strong>Item Name:</strong> Clear, specific description of the risk (e.g., "Uneven loading surface")</li>
                <li><strong>Description:</strong> Additional context or what assessors should look for</li>
                <li><strong>Display Order:</strong> Controls the order items appear in assessments (0 = first)</li>
                <li><strong>Deactivate vs Delete:</strong> Deactivate items that are no longer relevant but keep historical data</li>
                <li><strong>Usage Count:</strong> Shows how many assessments have evaluated this risk item</li>
            </ul>
        </div>
    </div>

    <!-- Add/Edit Modal -->
    <div id="item-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 class="modal-title" id="modal-title">Add Risk Item</h2>
                <button type="button" class="modal-close" onclick="window.app.closeModal('item-modal')">&times;</button>
            </div>

            <form method="POST" id="item-form">
                <input type="hidden" name="action" id="form-action" value="add">
                <input type="hidden" name="item_id" id="item-id">

                <div class="form-group">
                    <label class="form-label required" for="item_name">Risk Item Name</label>
                    <input type="text" id="item_name" name="item_name" class="form-control" required
                           placeholder="e.g., Uneven loading surface">
                    <small style="color: var(--text-muted); display: block; margin-top: 0.5rem;">
                        Clear, specific description of the risk to be assessed
                    </small>
                </div>

                <div class="form-group">
                    <label class="form-label" for="description">Description</label>
                    <textarea id="description" name="description" class="form-textarea" rows="3"
                              placeholder="Additional details about what to look for or consider..."></textarea>
                    <small style="color: var(--text-muted); display: block; margin-top: 0.5rem;">
                        Optional guidance for assessors
                    </small>
                </div>

                <div class="form-group">
                    <label class="form-label" for="display_order">Display Order</label>
                    <input type="number" id="display_order" name="display_order" class="form-control" 
                           value="0" min="0" max="100">
                    <small style="color: var(--text-muted); display: block; margin-top: 0.5rem;">
                        Order in which this item appears in assessments (0 = first, higher numbers = later)
                    </small>
                </div>

                <div class="form-group" id="active-group" style="display: none;">
                    <div style="display: flex; align-items: center; gap: 0.5rem;">
                        <input type="checkbox" id="is_active" name="is_active" checked style="width: auto;">
                        <label for="is_active" style="margin: 0; font-weight: 600;">Active</label>
                    </div>
                    <small style="color: var(--text-muted); display: block; margin-top: 0.5rem;">
                        Inactive items won't appear in new assessments but keep historical data
                    </small>
                </div>

                <div class="btn-group" style="margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">💾 Save Risk Item</button>
                    <button type="button" class="btn btn-secondary" onclick="window.app.closeModal('item-modal')">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <script src="js/app.js"></script>
    <script>
        function openAddItem() {
            document.getElementById('modal-title').textContent = 'Add Risk Item';
            document.getElementById('form-action').value = 'add';
            document.getElementById('item-form').reset();
            document.getElementById('item-id').value = '';
            document.getElementById('active-group').style.display = 'none';
            window.app.openModal('item-modal');
        }

        function editItem(item) {
            document.getElementById('modal-title').textContent = 'Edit Risk Item';
            document.getElementById('form-action').value = 'update';
            document.getElementById('item-id').value = item.id;
            document.getElementById('item_name').value = item.item_name;
            document.getElementById('description').value = item.description || '';
            document.getElementById('display_order').value = item.display_order;
            document.getElementById('is_active').checked = item.is_active == 1;
            document.getElementById('active-group').style.display = 'block';
            window.app.openModal('item-modal');
        }
    </script>
</body>
</html>
