/**
 * Vue.js Components Library
 * Reusable components for Risk Assessment System
 * Include this file after Vue.js
 */

// Global Component Registration
const registerGlobalComponents = (app) => {
    app.component('stat-card', StatCard);
    app.component('assessment-card', AssessmentCard);
    app.component('location-card', LocationCard);
    app.component('risk-badge', RiskBadge);
    app.component('status-badge', StatusBadge);
    app.component('loading-spinner', LoadingSpinner);
    app.component('empty-state', EmptyState);
    app.component('modal-dialog', ModalDialog);
    app.component('confirm-dialog', ConfirmDialog);
    app.component('search-filter', SearchFilter);
};

// ============================================================================
// STAT CARD COMPONENT
// ============================================================================
const StatCard = {
    props: {
        label: { type: String, required: true },
        value: { type: [String, Number], required: true },
        icon: { type: String, default: '📊' },
        color: { type: String, default: '#3B82F6' },
        link: { type: String, default: null }
    },
    template: `
        <div 
            class="stat-card" 
            :style="{ cursor: link ? 'pointer' : 'default' }"
            @click="navigate"
        >
            <div class="stat-icon" :style="{ color: color }">
                {{ icon }}
            </div>
            <div>
                <div class="stat-value">{{ value }}</div>
                <div class="stat-label">{{ label }}</div>
            </div>
        </div>
    `,
    methods: {
        navigate() {
            if (this.link) {
                window.location.href = this.link;
            }
        }
    }
};

// ============================================================================
// ASSESSMENT CARD COMPONENT
// ============================================================================
const AssessmentCard = {
    props: {
        assessment: { type: Object, required: true }
    },
    template: `
        <div class="card assessment-card">
            <div class="card-body">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1rem;">
                    <div>
                        <h3 style="margin: 0; color: var(--text-primary);">
                            {{ assessment.assessment_number }}
                        </h3>
                        <p style="margin: 0.25rem 0; color: var(--text-muted); font-size: 0.9rem;">
                            {{ assessment.location_name }}
                        </p>
                    </div>
                    <risk-badge :level="assessment.overall_risk_level" />
                </div>
                
                <div style="display: flex; gap: 1rem; margin-bottom: 1rem; font-size: 0.85rem; color: var(--text-muted);">
                    <span>📅 {{ formatDate(assessment.assessment_date) }}</span>
                    <span>👤 {{ assessment.assessor_name }}</span>
                </div>
                
                <div style="display: flex; gap: 0.5rem;">
                    <a :href="'view-assessment.php?id=' + assessment.id" class="btn btn-primary" style="flex: 1;">
                        View Details
                    </a>
                    <button class="btn btn-secondary">
                        📄 PDF
                    </button>
                </div>
            </div>
        </div>
    `,
    methods: {
        formatDate(dateString) {
            return new Date(dateString).toLocaleDateString('en-GB', {
                day: '2-digit',
                month: 'short',
                year: 'numeric'
            });
        }
    }
};

// ============================================================================
// LOCATION CARD COMPONENT
// ============================================================================
const LocationCard = {
    props: {
        location: { type: Object, required: true }
    },
    template: `
        <div class="card location-card">
            <div class="card-body">
                <h3 style="margin: 0 0 0.5rem 0; color: var(--text-primary);">
                    📍 {{ location.location_name }}
                </h3>
                <p style="margin: 0; color: var(--text-muted); font-size: 0.9rem;">
                    {{ location.address_line1 }}<br>
                    {{ location.city }}, {{ location.postcode }}
                </p>
                <div style="margin-top: 1rem; display: flex; gap: 0.5rem;">
                    <a :href="'view-location.php?id=' + location.id" class="btn btn-secondary">
                        View
                    </a>
                    <a :href="'edit-location.php?id=' + location.id" class="btn btn-secondary">
                        Edit
                    </a>
                </div>
            </div>
        </div>
    `
};

// ============================================================================
// RISK BADGE COMPONENT
// ============================================================================
const RiskBadge = {
    props: {
        level: { type: String, required: true }
    },
    computed: {
        badgeClass() {
            const classes = {
                'low': 'badge-success',
                'medium': 'badge-warning',
                'high': 'badge-danger',
                'critical': 'badge-danger'
            };
            return `badge ${classes[this.level] || 'badge-info'}`;
        }
    },
    template: `
        <span :class="badgeClass">
            {{ level }}
        </span>
    `
};

// ============================================================================
// STATUS BADGE COMPONENT
// ============================================================================
const StatusBadge = {
    props: {
        status: { type: String, required: true }
    },
    computed: {
        badgeClass() {
            const classes = {
                'draft': 'badge-warning',
                'completed': 'badge-success',
                'archived': 'badge-info'
            };
            return `badge ${classes[this.status] || 'badge-info'}`;
        }
    },
    template: `
        <span :class="badgeClass">
            {{ status }}
        </span>
    `
};

// ============================================================================
// LOADING SPINNER COMPONENT
// ============================================================================
const LoadingSpinner = {
    props: {
        message: { type: String, default: 'Loading...' }
    },
    template: `
        <div style="text-align: center; padding: 3rem;">
            <div class="spinner" style="margin: 0 auto;"></div>
            <p style="margin-top: 1rem; color: var(--text-muted);">{{ message }}</p>
        </div>
    `
};

// ============================================================================
// EMPTY STATE COMPONENT
// ============================================================================
const EmptyState = {
    props: {
        icon: { type: String, default: '📭' },
        title: { type: String, default: 'No items found' },
        message: { type: String, default: 'There are no items to display.' },
        actionText: { type: String, default: null },
        actionLink: { type: String, default: null }
    },
    template: `
        <div style="text-align: center; padding: 3rem;">
            <div style="font-size: 4rem; margin-bottom: 1rem;">{{ icon }}</div>
            <h3 style="color: var(--text-primary); margin-bottom: 0.5rem;">{{ title }}</h3>
            <p style="color: var(--text-muted); margin-bottom: 1.5rem;">{{ message }}</p>
            <a v-if="actionText && actionLink" :href="actionLink" class="btn btn-primary">
                {{ actionText }}
            </a>
        </div>
    `
};

// ============================================================================
// MODAL DIALOG COMPONENT
// ============================================================================
const ModalDialog = {
    props: {
        show: { type: Boolean, default: false },
        title: { type: String, required: true },
        size: { type: String, default: 'medium' } // small, medium, large
    },
    emits: ['close'],
    template: `
        <div v-if="show" class="modal" @click.self="$emit('close')">
            <div class="modal-content" :style="{ maxWidth: modalWidth }">
                <div class="modal-header">
                    <h2 class="modal-title">{{ title }}</h2>
                    <button class="modal-close" @click="$emit('close')">&times;</button>
                </div>
                <div style="padding: 1.5rem;">
                    <slot></slot>
                </div>
            </div>
        </div>
    `,
    computed: {
        modalWidth() {
            const sizes = {
                small: '400px',
                medium: '600px',
                large: '800px'
            };
            return sizes[this.size] || sizes.medium;
        }
    }
};

// ============================================================================
// CONFIRM DIALOG COMPONENT
// ============================================================================
const ConfirmDialog = {
    props: {
        show: { type: Boolean, default: false },
        title: { type: String, default: 'Confirm Action' },
        message: { type: String, required: true },
        confirmText: { type: String, default: 'Confirm' },
        cancelText: { type: String, default: 'Cancel' },
        danger: { type: Boolean, default: false }
    },
    emits: ['confirm', 'cancel'],
    template: `
        <modal-dialog :show="show" :title="title" size="small" @close="$emit('cancel')">
            <p style="margin-bottom: 1.5rem; color: var(--text-secondary);">{{ message }}</p>
            <div style="display: flex; gap: 0.5rem; justify-content: flex-end;">
                <button class="btn btn-secondary" @click="$emit('cancel')">
                    {{ cancelText }}
                </button>
                <button 
                    :class="'btn ' + (danger ? 'btn-danger' : 'btn-primary')" 
                    @click="$emit('confirm')"
                >
                    {{ confirmText }}
                </button>
            </div>
        </modal-dialog>
    `
};

// ============================================================================
// SEARCH FILTER COMPONENT
// ============================================================================
const SearchFilter = {
    props: {
        modelValue: { type: String, default: '' },
        placeholder: { type: String, default: 'Search...' }
    },
    emits: ['update:modelValue'],
    template: `
        <div style="position: relative;">
            <input 
                type="text" 
                :value="modelValue"
                @input="$emit('update:modelValue', $event.target.value)"
                :placeholder="placeholder"
                class="form-control"
                style="padding-left: 2.5rem;"
            >
            <span style="position: absolute; left: 0.75rem; top: 50%; transform: translateY(-50%); font-size: 1.2rem;">
                🔍
            </span>
        </div>
    `
};

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================
const utils = {
    // Format date
    formatDate(dateString, format = 'short') {
        const date = new Date(dateString);
        if (format === 'short') {
            return date.toLocaleDateString('en-GB', {
                day: '2-digit',
                month: 'short',
                year: 'numeric'
            });
        } else if (format === 'long') {
            return date.toLocaleDateString('en-GB', {
                weekday: 'long',
                day: '2-digit',
                month: 'long',
                year: 'numeric'
            });
        }
        return date.toLocaleDateString();
    },

    // Format time
    formatTime(timeString) {
        return timeString.substring(0, 5); // HH:MM
    },

    // Debounce function
    debounce(func, wait = 300) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },

    // API call wrapper
    async api(url, options = {}) {
        try {
            const response = await axios({
                url,
                method: options.method || 'GET',
                data: options.data || null,
                headers: options.headers || {},
                ...options
            });
            return { success: true, data: response.data };
        } catch (error) {
            console.error('API Error:', error);
            return { 
                success: false, 
                error: error.response?.data?.message || 'An error occurred' 
            };
        }
    }
};

// Export for global use
window.VueComponents = {
    registerGlobalComponents,
    StatCard,
    AssessmentCard,
    LocationCard,
    RiskBadge,
    StatusBadge,
    LoadingSpinner,
    EmptyState,
    ModalDialog,
    ConfirmDialog,
    SearchFilter,
    utils
};
