// Service Worker for Risk Assessment App
const CACHE_NAME = 'risk-assessment-v1';
const urlsToCache = [
    '/',
    '/css/style.css',
    '/js/app.js',
    '/login.php',
    '/dashboard.php',
    '/offline.html'
];

// Install event - cache resources
self.addEventListener('install', (event) => {
    event.waitUntil(
        caches.open(CACHE_NAME)
            .then((cache) => {
                console.log('Opened cache');
                return cache.addAll(urlsToCache);
            })
            .catch((error) => {
                console.error('Cache install failed:', error);
            })
    );
});

// Activate event - clean up old caches
self.addEventListener('activate', (event) => {
    event.waitUntil(
        caches.keys().then((cacheNames) => {
            return Promise.all(
                cacheNames.map((cacheName) => {
                    if (cacheName !== CACHE_NAME) {
                        console.log('Deleting old cache:', cacheName);
                        return caches.delete(cacheName);
                    }
                })
            );
        })
    );
});

// Fetch event - serve from cache when offline
self.addEventListener('fetch', (event) => {
    event.respondWith(
        caches.match(event.request)
            .then((response) => {
                // Cache hit - return response
                if (response) {
                    return response;
                }

                // Clone the request
                const fetchRequest = event.request.clone();

                return fetch(fetchRequest).then((response) => {
                    // Check if valid response
                    if (!response || response.status !== 200 || response.type !== 'basic') {
                        return response;
                    }

                    // Clone the response
                    const responseToCache = response.clone();

                    caches.open(CACHE_NAME)
                        .then((cache) => {
                            cache.put(event.request, responseToCache);
                        });

                    return response;
                }).catch(() => {
                    // Network request failed, try to return offline page
                    if (event.request.destination === 'document') {
                        return caches.match('/offline.html');
                    }
                });
            })
    );
});

// Background sync
self.addEventListener('sync', (event) => {
    if (event.tag === 'sync-assessments') {
        event.waitUntil(syncAssessments());
    }
});

async function syncAssessments() {
    // This would sync pending assessments from IndexedDB to server
    console.log('Syncing assessments...');
}

// Push notifications (optional future feature)
self.addEventListener('push', (event) => {
    const options = {
        body: event.data.text(),
        icon: '/images/icon-192.png',
        badge: '/images/badge.png'
    };

    event.waitUntil(
        self.registration.showNotification('Risk Assessment Update', options)
    );
});
